/*****************************************************************************
 * Copyright (c) 1997-1998 The Java Apache Project. All rights reserved.     *
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions are    *
 * met:                                                                      *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Every modification must be notified to the "Java Apache Project"       *
 *    and redistribution of the modified code without prior notification is  *
 *    NOT permitted in any form.                                             *
 *                                                                           *
 * 4. All advertising materials mentioning features or use of this software  *
 *    must display the following acknowledgment:                             *
 *    "This product includes software developed by the Java Apache Project   *
 *     <http://java.apache.org>."                                            *
 *                                                                           *
 * 5. The names "JServ", "JServ Servlet Engine" and "Java Apache Project"    *
 *    must not be used to endorse or promote products derived from this      *
 *    software without prior written permission.                             *
 *                                                                           *
 * 6. Redistributions of any form whatsoever must retain the following       *
 *    acknowledgment:                                                        *
 *    "This product includes software developed by the Java Apache Project   *
 *     <http://java.apache.org>."                                            *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY      *
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE       *
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR        *
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE JAVA APACHE PROJECT OR ITS  *
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,     *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
 *                                                                           *
 * This software consists of voluntary contributions made by many            *
 * individuals on behalf of the Java Apache Project and was originally based *
 * on public domain software written by by Alexei Kosut <akosut@apache.org>. *
 * For more information on the Java Apache Project and the JServ Servlet     *
 * Engine project, please see <http://java.apache.org/>.                     *
 *****************************************************************************/

/*****************************************************************************
 * Description: protocol balancer, used to call local or remote jserv hosts    *
 * Author:      Bernard Bernstein <bernard@corp.talkcity.com>                *
 * Version:     1.0 Beta                                                     *
 *****************************************************************************/
#include "jserv.h"


#ifdef LOAD_BALANCE

/*****************************************************************************
 * Code for protocol balancer                                                  *
 *****************************************************************************/


/* *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** */
/* The SESSION_ID is also defined in the Java. If one changes              */
/* then so must the other. This must stay in sync with the session cookie  */
/* or parameter set by the java code                                       */

#define SESSION_IDENTIFIER "JServSessionId"
#define ROUTING_IDENTIFIER "JSERV_ROUTE"

/* ========================================================================= */
/* Retrieve the parameter with the given name                                */
static char*
get_param(char *name, request_rec *r)
{
  char *pname = ap_pstrdup(r->pool, name);
  char *value = NULL;
  char *varg = NULL;
  int len = 0;

  pname = ap_pstrcat(r->pool, pname, "=", NULL);

  if (!r->args) {
    return NULL;
  }

  value = strstr(r->args, pname);
  if (value) {
    value += strlen(pname);
    varg = value;
    while (*varg && *varg != '&') {
      varg++;
      len++;
    }
  }

  if (len == 0) {
    return NULL;
  }

  return ap_pstrndup(r->pool, value, len);
}

/* ========================================================================= */
/* Retrieve the cookie with the given name                                */
static char*
get_cookie(char *name, request_rec *r)
{
  const char *cookie;
  char *value;
  char *cname = ap_pstrdup(r->pool, name);
  char *varg;

  cname = ap_pstrcat(r->pool, cname, "=", NULL);
  if ((cookie = ap_table_get(r->headers_in, "Cookie")) != NULL)
    if ((value = strstr(cookie, cname)) != NULL) {
      char *cookiebuf, *cookieend;
      value += strlen(cname);
      cookiebuf = ap_pstrdup(r->pool, value);
      cookieend = strchr(cookiebuf, ';');
      if (cookieend)
        *cookieend = '\0';      /* Ignore anything after a ; */

      return cookiebuf;    /* Theres already a cookie, no new one */
    }

  return NULL;
}


/* ========================================================================= */
/* Retrieve session id from the cookie or the parameter                      */
/* (parameter first)                                                         */
static char *
get_jserv_sessionid(request_rec *r)
{
  char *val;
  val = get_param(SESSION_IDENTIFIER, r);
  if (val == NULL)
    val = get_cookie(SESSION_IDENTIFIER, r);
  return val;
}

static int
get_jserv_session_balance(const char **hostid, request_rec *r)
{
  char *sessionid = get_jserv_sessionid(r);
  char *ch;
  char *addr_port;
  int segment = 0;    /* segments of the id. they are: */
                      /* 0 - random */
                      /* 1 - session_count */
                      /* 2 - time */
                      /* 3 - hostid */

  if (sessionid == NULL)
    return 0;
  
  ch = sessionid;
  while (*ch != '\0') {
    if (*ch == '.') {
      segment++;
    }
    else if (segment == 3 ) {
      *hostid = ap_pstrdup(r->pool, ch);
      return 1;
    }
    ch++;
  }
  return 0;
}

/* ========================================================================= */
/* Our request handler */
static int balance_handler(jserv_config *cfg, jserv_request *req, 
                          request_rec *r) {
    int ret, sock;
    BUFF *buffsocket;
    const char *header;
    jserv_host *first;
    jserv_host *cur;
    int result;
    const char *hostid;

    /* debug message */
    jserv_error(JSERV_LOG_INFO,cfg,"balance: %s",
                    "got another balance request");
    

    /* Check for correct config member */
    if (cfg==NULL) {
        jserv_error(JSERV_LOG_EMERG,cfg,"balance: %s",
                    "unknown configuration member for request");
        return SERVER_ERROR;
    }

    /* Check for correct jserv request member */
    if (req==NULL) {
        jserv_error(JSERV_LOG_EMERG,cfg,"balance: %s",
                    "null request not handled");
        return SERVER_ERROR;
    }
    if (req->mount==NULL) {
        jserv_error(JSERV_LOG_EMERG,cfg,"balance: %s",
                    "unknown mount for request");
        return SERVER_ERROR;
    }

    /* Check parameters and cookies for a current session */
    if (get_jserv_session_balance(&hostid, r) && hostid != NULL) {
        cur = cfg->hosturls;
        
        /* debug message */
        jserv_error(JSERV_LOG_INFO,cfg,
                   "balance: continuing session: %s",
                   hostid);
        
        /* found a session id, now get host with the given id */
        while (cur != NULL) {
          if (!strcmp(cur->id, hostid)) {
            /* found a matching host for the given id */
            req->mount->hostaddr = cur->hostaddr;
            req->mount->port = cur->port;
            req->mount->host = cur->host;
            req->mount->secretfile = cur->secretfile;
            req->mount->secret = cur->secret;
            req->mount->secretsize = cur->secretsize;
            
            if (r->subprocess_env) {
              ap_table_set(r->subprocess_env, ROUTING_IDENTIFIER, cur->id);
            }
            
            result = jserv_protocol_handler(cur->protocol, cfg, req, r);
            
            /* debug message */
            jserv_error(JSERV_LOG_INFO,cfg,
                       "balance: continuing to %s:%d",
                       cur->host, cur->port);

            if (result != SERVER_ERROR && result != HTTP_INTERNAL_SERVER_ERROR) {
              return result;
            }
            
            /* if that server failed, break out and use regular load-balancing */
            break;
          }
          cur = cur->next;
        }
    }

    /* Find a real server from the virtual list */
    first = cur = req->mount->hosturls;
    if (cur==NULL) {
        jserv_error(JSERV_LOG_EMERG,cfg,"balance: %s",
                    "virtual host not specified");
        return SERVER_ERROR;
    }
 
    /* find the next server that succeeds for fault tolerance. */
    
    /* *** BB: We probably want some mechanism to take non-working servers */
    /* *** out of the loop temporarily and then retry in some interval */
    /* *** rather than potentially check failing servers for every */
    /* *** request. */
    /* *** But, even if we do that, it wouldn't work across all of the */
    /* *** httpd processes. I don't know the best solution for this and */
    /* *** I yield to others to solve that one. */
    
    do {
      /* use the next host in the ring */
      cur = cur->next;
      req->mount->hosturls = cur;
  
      /* debug message */
      jserv_error(JSERV_LOG_INFO,cfg,
                 "balance: attempting to connect to server %s: %s://%s(%lx):%d",
                 cur->name, cur->protocol?cur->protocol->name:"DEFAULT", 
                 cur->host?cur->host:"DEFAULT", cur->hostaddr, cur->port);
      
      /* we don't insert the protocol or else the protocol would */
      /* not get switched back to "balance" automatically */
      req->mount->hostaddr = cur->hostaddr;
      req->mount->port = cur->port;
      req->mount->host = cur->host;
      req->mount->secretfile = cur->secretfile;
      req->mount->secret = cur->secret;
      req->mount->secretsize = cur->secretsize;
  
      if (r->subprocess_env) {
        ap_table_set(r->subprocess_env, ROUTING_IDENTIFIER, cur->id);
      }

      /* try this handler. If it fails, it will keep looking for a */
      /* server that works. If it succeeds, we return here */
      result = jserv_protocol_handler(cur->protocol, cfg, req, r);
      if (result != SERVER_ERROR && result != HTTP_INTERNAL_SERVER_ERROR) {
      
        /* debug message */
        jserv_error(JSERV_LOG_INFO,cfg,
             "balance: successfully made request to server %s: %s://%s(%lx):%d",
             cur->name, cur->protocol?cur->protocol->name:"DEFAULT", 
             cur->host?cur->host:"DEFAULT", cur->hostaddr, cur->port);

        return result;
      }

    } while (cur != first);
    

    jserv_error(JSERV_LOG_EMERG,cfg,"balance: %s",
                "virtual host not found");
    return SERVER_ERROR;
}


/*****************************************************************************
 * AJPv11 Protocol Structure definition                                      *
 *****************************************************************************/
jserv_protocol jserv_balancep = {
    "balance",                  /* Name for this protocol */
    8007,                       /* Default port for this protocol */
    NULL,                       /* init() */
    NULL,                       /* cleanup() */
    NULL,                       /* child_init() */
    NULL,                       /* child_cleanup() */
    balance_handler,            /* handler() */
    NULL,                       /* function() */
    NULL,                       /* parameter() */
};

#endif
