/*
 * Copyright (c) 1997-1998 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */

/*****************************************************************************
 * Author:      Pierpaolo Fumagalli <ianosh@iname.com>                       *
 * Based on:    mod_jserv.c by Alexei Kosut <akosut@apache.org>              *
 *              mod_example.c by Apache Group <apache@apache.org>            *
 * Modified by: Pierpaolo Fumagalli <ianosh@iname.com> June, 12 1998         *
 * Version:     1.0                                                          *
 *****************************************************************************/
#include "jserv.h"

/*****************************************************************************
 * Local procedures for jserv_status protocol                                *
 *****************************************************************************/

/* ========================================================================= */
/* Remote configured hosts (JVMs) */
typedef struct jserv_status_host jserv_status_host;

struct jserv_status_host {
    char *url;
    jserv_mount *mount;
    jserv_status_host *next;
};

/* ========================================================================= */
/* Remote configured hosts (JVMs) */
jserv_status_host *jserv_status_hosts=NULL;
pool *jserv_status_pool=NULL;

/* ========================================================================= */
/* Local output functions */
static int jserv_status_out_menu(jserv_config *cfg, jserv_request *req,
                                  request_rec *r);
static int jserv_status_out_image(jserv_config *cfg, jserv_request *req,
                                   request_rec *r);
static int jserv_status_out_server(jserv_config *cfg, jserv_request *req,
                                   request_rec *r);
static int jserv_status_out_jserv(jserv_config *cfg, jserv_request *req,
                                   request_rec *r, int mount_number);

/*****************************************************************************
 * Code for jserv_status protocol                                            *
 *****************************************************************************/

/* ========================================================================= */
/* Our request handler */
static int jserv_status_handler(jserv_config *cfg,jserv_request *req,
                                request_rec *r) {
    int ret;
    char *tmp;

    ap_hard_timeout("status-handler", r);

    /* If we had a directory "engine" then we want the JSERV STATUS */
    if ((tmp=strstr(r->uri,"/engine/"))!=NULL) {
         int mount_number=0;
         /* Check if we had module=(null) or less... no server*/
         if (*tmp == '\0') { /* invalid URL -- must have an integer there */
             ret = FORBIDDEN;
         } else {
             mount_number=atoi(tmp+8); /* well, it might not be an int, but */
             ret=jserv_status_out_jserv(cfg,req,r,mount_number); /* if not */
         }                             /* mount_number will be 0, which is. */
    /* If we had no query then we want the MENU */
    } else if (r->args==NULL) {
        ret=jserv_status_out_menu(cfg,req,r);
    /* If we had query="menu" then we want the MENU */
    } else if (strcasecmp(r->args,"menu")==0) {
        ret=jserv_status_out_menu(cfg,req,r);
    /* If we had query="image" then we want the IMAGE */
    } else if (strcasecmp(r->args,"image")==0) {
        ret=jserv_status_out_image(cfg,req,r);
    /* If we had query="module" then we want the SERVER STATUS */
    } else if (strcasecmp(r->args,"module")>0) {
        /* Check if we had module=(null) or less... no server*/
        if (strlen(r->args)<=7) ret=FORBIDDEN;
        else {
        /* If we have module=server_name */
            char *buf=&r->args[7];
            jserv_config *cur=jserv_servers;

            /* Set default return 500 (internal server error) */
            ret=SERVER_ERROR;
            /* Iterate thru all configurations to match server name */
            while (cur!=NULL) {
                if (cur->server!=NULL) {
                    if (strcmp(cur->server->server_hostname,buf)==0) {
                        /* Return what server returned */
                        ret=jserv_status_out_server(cur,req,r);
                        break;
                    }
                }
                cur=cur->next;
            }
        }
    /* Default we return FORBIDDEN */
    } else ret=FORBIDDEN;
    
    ap_kill_timeout(r);
    return ret;
}

/* ========================================================================= */
/* Check and add if not found (except zone) a mountpoint to hosts */
static void jserv_status_add_host(pool *p, jserv_mount *mount) {
    jserv_status_host *cur=jserv_status_hosts;
    jserv_status_host *new=NULL;

    /* Iterate thru all mount points */
    while (cur!=NULL) {
        /* Check if we found the same mount point */
        if (cur->mount==mount) return;

        /* Check protocol,host and port */
        if ((cur->mount->protocol==mount->protocol) |
            (cur->mount->hostaddr==mount->hostaddr) |
            (cur->mount->port==mount->port)) return;

        cur=cur->next;
    }

    /* Add to the list */
    new=(jserv_status_host *)ap_pcalloc(p,sizeof(jserv_status_host));
    new->url=ap_psprintf(p, "%s://%s:%d", mount->protocol->name, mount->host,
                         mount->port);
    new->mount=mount;
    new->next=jserv_status_hosts;
    jserv_status_hosts=new;
    return;
}

/* ========================================================================= */
/* Our protocol initializer */
static int jserv_status_init(jserv_config *cfg) {
    jserv_config *cur=jserv_servers;
    jserv_status_pool=ap_make_sub_pool(jserv_pool);
    jserv_status_hosts = NULL;

    while (cur!=NULL) {
        jserv_mount *rcur=cur->mount;

        while (rcur!=NULL) {
            jserv_status_add_host(jserv_status_pool, rcur);
            rcur=rcur->next;
        }        
        cur=cur->next;
    }
    return 0;
}

/*****************************************************************************
 * Local output functions                                                    *
 *****************************************************************************/

/* ========================================================================= */
/* Output a table (<td>...</td>) */
static void jserv_status_out_table(request_rec *r, const char *bkg,
                                   const char *param, const char *fmt, ...) {
    va_list ap;
    char *buf;

    va_start(ap,fmt);
    buf=ap_pvsprintf(r->pool,fmt,ap);
    va_end(ap);

    ap_rputs("    <td", r);
    if (bkg!=NULL) ap_rprintf(r," bgcolor=\"%s\"",bkg);
    if (param!=NULL) ap_rprintf(r," %s",param);
    ap_rputs(">\n",r);
    ap_rprintf(r,"      %s\n",buf);
    ap_rputs("    </td>\n",r);
}

/* ========================================================================= */
/* Output the HEAD (head, title, banner ...) */
static void jserv_status_out_head(request_rec *r) {
    ap_rputs("<html>\n",r);
    ap_rputs("<head>\n",r);
    ap_rputs("<meta name=\"GENERATOR\" CONTENT=\"Apache JServ " JSERV_VERSION "\">\n",r);
    ap_rputs("<title>Apache JServ " JSERV_VERSION " Status</title>\n",r);
    ap_rputs("</head>\n",r);
    ap_rputs("<body bgcolor=\"#ffffff\" text=\"#000000\">\n",r);
    ap_rputs("<p align=center><img src=\"./status?image\"></p>\n",r);
}

/* ========================================================================= */
/* Output the TAIL (copyright, link...) */
static void jserv_status_out_tail(request_rec *r) {
    ap_reset_timeout(r);
    ap_rputs("<p align=\"center\"><font size=-1>\n",r);
    ap_rputs("Copyright (c) 1997-98 <a href=\"http://java.apache.org\">",r);
    ap_rputs("The Java Apache Project</a>.<br>\n",r);
    ap_rputs("All rights reserved.\n",r);
    ap_rputs("</font></p>\n",r);
    ap_rputs("</body>\n",r);
    ap_rputs("</html>\n",r);    
}

/* ========================================================================= */
/* Output the MENU */
static int jserv_status_out_menu(jserv_config *cfg, jserv_request *req,
                                  request_rec *r) {
    jserv_config *cur=jserv_servers;
    jserv_status_host *rcur=jserv_status_hosts;

    /* Set type and evaluate if this is a handler-only request */
    r->content_type = "text/html";
    ap_send_http_header(r);
    if (r->header_only) return OK;

    /* Real output */
    jserv_status_out_head(r);
    
    ap_rputs("<center><table border=\"0\" width=\"60%\" bgcolor=\"#000000\" cellspacing=\"0\" cellpadding=\"0\">\n",r);
    ap_rputs(" <tr>\n",r);
    ap_rputs("  <td width=\"100%\">\n",r);
    ap_rputs("   <table border=\"0\" width=\"100%\" cellpadding=\"4\">\n",r);
    ap_rputs("    <tr>\n",r);
    ap_rputs("     <td width=\"100%\" bgcolor=\"#c0c0c0\"><p align=\"right\"><big><big>Apache JServ " JSERV_VERSION " Status</big></big></td>\n",r);
    ap_rputs("    </tr><tr>\n",r);
    ap_rputs("     <td width=\"100%\" bgcolor=\"#e0e0e0\">Welcome to the dynamic status page of the Apache JServ\n",r);
    ap_rputs("     servlet engine. All these pages are dynamically created to show you the status of your\n",r);
    ap_rputs("     servlet execution environment, both on the web server side (generated by mod_jserv) and on\n",r);
    ap_rputs("     the servlet engine side (yes, Apache JServ is actually a servlet and executes itself!)</td>\n",r);
    ap_rputs("    </tr>\n",r);
    ap_rputs("    <tr>\n",r);
    ap_rputs("     <td width=\"100%\" bgcolor=\"#f0f0f0\"><br><br>\n",r);
    ap_rputs("      <center>\n",r);
    ap_rputs("       <table border=\"0\" bgcolor=\"#000000\" cellspacing=\"0\" cellpadding=\"0\">\n",r);
    ap_rputs("        <tr>\n",r);
    ap_rputs("         <td width=\"100%\">\n",r);
    ap_rputs("          <table border=\"0\" width=\"100%\" cellpadding=\"4\">\n",r);
    ap_rputs("           <tr>\n",r);
    ap_rputs("           <td bgcolor=\"#c0c0c0\" width=\"50%\" valign=\"middle\" align=\"center\" nowrap><b>Configured hosts</b></td>",r);
    ap_rputs("           <td bgcolor=\"#c0c0c0\" width=\"50%\" valign=\"middle\" align=\"center\" nowrap><b>Mapped servlet engines</b></td>",r);
    ap_rputs("           </tr>\n",r);
    ap_rputs("           <tr>\n",r);
    ap_rputs("            <td bgcolor=\"#ffffff\" valign=\"middle\" align=\"center\" nowrap>",r);
                  
    /* Examine list of Apache virtualhosts currently configured for Apache JServ */
    ap_reset_timeout(r);
    while (cur!=NULL) {
    char *name=cur->server->server_hostname;
        ap_rprintf(r,"<a href=\"./status?module=%s\">%s</a>",name,name);
        if (cur==cfg) ap_rputs(" <font size=-1><i>(current)</i></font>",r);
        ap_rputs("<br>\n",r);
        cur=cur->next;
    }
    
    ap_rputs("            </td>\n",r);
    ap_rputs("            <td bgcolor=\"#ffffff\" valign=\"middle\" align=\"center\" nowrap>",r);
    
    /* List of JServ hosts */
    while (rcur!=NULL) {
        int count = 0;
        ap_rprintf(r,"<a href=\"./engine/%d/\">%s</a>", count++, rcur->url);
        ap_rputs("<br>\n",r);
        rcur=rcur->next;
    }

    ap_rputs("            </td>\n",r);
    ap_rputs("           </tr>\n",r);
    ap_rputs("          </table>\n",r);
    ap_rputs("         </td>\n",r);
    ap_rputs("        </tr>\n",r);
    ap_rputs("       </table>\n",r);
    ap_rputs("      </center><br>\n",r);
    ap_rputs("     </td>\n",r);
    ap_rputs("    </tr>\n",r);
    ap_rputs("    <tr>\n",r);
    ap_rputs("     <td width=\"100%\" bgcolor=\"#FFFFFF\"><strong>Warning</strong>: you should restrict access to\n",r);
    ap_rputs("     this page on a production environment since it may give untrusted users access to\n",r);
    ap_rputs("     information you may want to remain secret.</td>\n",r);
    ap_rputs("    </tr>\n",r);
    ap_rputs("   </table>\n",r);
    ap_rputs("  </td>\n",r);
    ap_rputs(" </tr>\n",r);
    ap_rputs("</table><center>\n",r);
    jserv_status_out_tail(r);
    
    return OK;
}

/* ========================================================================= */
/* Output the Server Configuration */
static int jserv_status_out_server(jserv_config *cfg, jserv_request *req,
                                   request_rec *r) {
    jserv_mount *cur=NULL;

    /* This is so we can use inet_ntoa, which is used for reporting
     * a network address in human-readable format. */
    struct in_addr network_address;
    network_address.s_addr = cfg->hostaddr;

    /* Set type and evaluate if this is a handler-only request */
    r->content_type = "text/html";
    ap_send_http_header(r);
    if (r->header_only) return OK;

    /* Real output */
    jserv_status_out_head(r);

    /* Output table headers */
    ap_rputs("<center>\n",r);
    ap_rputs("<table border=0>\n",r);
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#c0c0c0","width=50% valign=top align=center",
                           "<b>Parameter</b>");
    jserv_status_out_table(r,"#c0c0c0","width=50% valign=top align=center",
                           "<b>Value</b>");
    ap_rputs("  </tr>\n",r);

    /* Output Server Name */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "<b>Server Name</b>");
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "%s",cfg->server->server_hostname);
    ap_rputs("  </tr>\n",r);


    /* Output Manual Mode */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "<b>ApJServManual</b>");
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                    "%s",(cfg->manual==JSERV_TRUE)?
                    "TRUE <font size=-2>(STANDALONE OPERATION</font>)":
                    "FALSE <font size=-2>(AUTOMATIC OPERATION</font>)");

    /* Output Properties File */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "<b>ApJServProperties</b>");

    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "%s <font size=-2>%s</font>",cfg->properties,
                           (cfg->manual==JSERV_TRUE)?
                           "(IGNORED)":
                           "");

    /* Output Protocol */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "<b>ApJServDefaultProtocol</b>");
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "%s <font size=-2>(PORT %d)</font>",
                           cfg->protocol->name,cfg->protocol->port);

    /* Output Host */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "<b>ApJServDefaultHost</b>");
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "%s <font size=-2>(ADDR %s)</font>",
                           cfg->host,
                           inet_ntoa(network_address));

    /* Output Port */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "<b>ApJServDefaultPort</b>");
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "%d",cfg->port);

    /* Output Logfile */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "<b>ApJServLogFile</b>");
    jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "%s <font size=-2>(DESCRIPTOR %d)</font>",
                           cfg->logfile, cfg->logfilefd);

    /* Output Mountcopy */
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "<b>ApJServMountCopy</b>");
    jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                           "%s",(cfg->mountcopy==JSERV_TRUE)?
                                       "TRUE":"FALSE");

    /* Properties done */
    ap_rputs("  </tr>\n",r);
    ap_rputs("</table>\n",r);

    /* Output table headers for mounts */
    ap_rputs("<br>\n",r);
    ap_rputs("<table border=0>\n",r);
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#b0b0b0","valign=top align=center",
                           "<b>MountPoint</b>");
    jserv_status_out_table(r,"#b0b0b0","valign=top align=center",
                           "<b>Server</b>");
    jserv_status_out_table(r,"#b0b0b0","valign=top align=center",
                           "<b>Protocol</b>");
    jserv_status_out_table(r,"#b0b0b0","valign=top align=center",
                           "<b>Host</b>");
    jserv_status_out_table(r,"#b0b0b0","valign=top align=center",
                           "<b>Port</b>");
    jserv_status_out_table(r,"#b0b0b0","valign=top align=center",
                           "<b>Zone</b>");
    ap_rputs("  </tr>\n",r);

    /* Check for valid mountpoints */
    cur=cfg->mount;
    ap_reset_timeout(r);
    while (cur!=NULL) {
        ap_rputs("  <tr>\n",r);
        jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                               "%s",cur->mountpoint);

        jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                               "%s",
                               cur->config->server->server_hostname);

        jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                               "%s",cur->protocol?
                                           cur->protocol->name:
                                           "DEFAULT");

        jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                           "%s <font size=-2>(ADDR %s)</font>",
                           cur->host?cur->host:"DEFAULT",
                           inet_ntoa(network_address));

        jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                               "%d",cur->port);

        jserv_status_out_table(r,"#f0f0f0","valign=top align=left",
                               "%s",cur->zone?cur->zone:"URI MAPPED");
                               
        ap_rputs("  </tr>\n",r);
        ap_reset_timeout(r);
        cur=cur->next;
    }

    /* Mounts done */
    ap_rputs("  </tr>\n",r);
    ap_rputs("</table>\n",r);
    
    /* See ACTIONS commands */
    ap_reset_timeout(r);
    ap_rputs("<br>\n",r);
    ap_rputs("<table border=0>\n",r);
    ap_rputs("  <tr>\n",r);
    jserv_status_out_table(r,"#c0c0c0","valign=top align=center",
                           "<b>Extension</b>");
    jserv_status_out_table(r,"#c0c0c0","valign=top align=center",
                           "<b>Servlet</b>");

    if (!ap_is_empty_table(cfg->actions)) {
        array_header *actions=ap_table_elts(cfg->actions);
        table_entry *elts=(table_entry *) actions->elts;
        int i;

        for (i = 0; i < actions->nelts; ++i) {
            ap_rputs("  <tr>\n",r);
            jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                                   "%s",elts[i].key);
            jserv_status_out_table(r,"#e0e0e0","valign=top align=left",
                                   "%s",elts[i].val);
            ap_rputs("  </tr>\n",r);
            ap_reset_timeout(r);
        }
    }
    ap_rputs("</table>\n",r);

    /* Output Menu, Base server and next server links */
    ap_reset_timeout(r);
    ap_rputs("<br>\n",r);
    ap_rputs("<table border=0>\n",r);
    ap_rputs("  <tr>\n",r);
 
    jserv_status_out_table(r,"#ffffff","width=33% valign=top align=center",
    "<a href=\"./status?menu\">Back to menu</a>");

    if (jserv_servers!=NULL)
    jserv_status_out_table(r,"#ffffff","width=33% valign=top align=center",
    "<a href=\"./status?module=%s\">Base server <i>(%s)</i></a>",
                jserv_servers->server->server_hostname, 
                jserv_servers->server->server_hostname);

    if (cfg->next!=NULL)
    jserv_status_out_table(r,"#ffffff","width=33% valign=top align=center",
    "<a href=\"./status?module=%s\">Next server <i>(%s)</i></a>",
                cfg->next->server->server_hostname, 
                cfg->next->server->server_hostname);

    ap_rputs("  </tr>\n",r);
    ap_rputs("</table>\n",r);

    /* Done */
    ap_rputs("</center>\n",r);
    jserv_status_out_tail(r);
    return OK;
}

/* ========================================================================= */
/* Output the IMAGE */
static int jserv_status_out_image(jserv_config *cfg, jserv_request *req,
                                   request_rec *r) {
    r->content_type = "image/gif";
    ap_set_content_length(r,jserv_image_size);
    ap_send_http_header(r);
    if (r->header_only) return OK;

    /* Image output */
    ap_kill_timeout(r);
    ap_rwrite(jserv_image, jserv_image_size, r);
    return OK;
}

/* ========================================================================= */
/* Output the JServ status configuration calling JServ as a sevlet */
static int jserv_status_out_jserv(jserv_config *cfg, jserv_request *req,
                                   request_rec *r, int mount_number) {
     jserv_status_host *rcur=jserv_status_hosts;
     int tmp_int = 0;
 
     req->isdir=JSERV_FALSE;
     req->servlet=JSERV_SERVLET;
 
     while (tmp_int++ < mount_number && rcur != NULL) {
         rcur = rcur->next;
     }
 
     if (rcur == NULL) { /* weird -- we walked to the end.  oh well -- */
         return FORBIDDEN;     /* it's not currently a valid 'engine'. */
     } else {
         req->mount=rcur->mount;
         req->zone=rcur->mount->zone;
         return jserv_protocol_handler(rcur->mount->protocol, cfg, req, r);
     }
}

/*****************************************************************************
 * Dummy Protocol Structure definition                                       *
 *****************************************************************************/
jserv_protocol jserv_status = {
    "status",                           /* Name for this protocol */
    0,                                  /* Default port for this protocol */
    jserv_status_init,                  /* init() */
    NULL,                               /* cleanup() */
    NULL,                               /* child_init() */
    NULL,                               /* child_cleanup() */
    jserv_status_handler,               /* handler() */
    NULL,                               /* function() */
    NULL,                               /* parameter() */
};
