/*
 * Copyright (c) 1997-1999 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    <http://java.apache.org/>."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    <http://java.apache.org/>."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */
 
/*****************************************************************************
 * Description: Apache JServ 1.0 apache backward compatibility functions     *
 * Author:      Pierpaolo Fumagalli <ianosh@iname.com>                       *
 * Version:     $Revision: 1.6 $                                            *
 *****************************************************************************/
#if MODULE_MAGIC_NUMBER < 19980527
#include "jserv.h"

/*****************************************************************************
 * Functions ported from Apache 1.3 to Apache 1.2 for backward compatibility *
 *****************************************************************************/

/* ========================================================================= */
/* Prints VA_ARG to a pool string */
char *compat_pvsprintf(pool *p, const char *fmt, va_list ap) {
    char *ret;
    char buf[MAX_STRING_LEN];
    int len;

    ap_vsnprintf(buf,MAX_STRING_LEN,fmt,ap);
    len=strlen(buf);
    ret=pstrdup(p,buf);
    return ret;
}

/* ========================================================================= */
/* Print string using pool */
char *compat_psprintf(pool *p, const char *fmt, ...) {
    va_list ap;
    char *ret;

    va_start(ap,fmt);
    ret=compat_pvsprintf(p,fmt,ap);
    va_end(ap);
    return ret;
}

/* ========================================================================= */
/* Null function used for cleanups and add_component */
void compat_null(void *data) {
    return;
}

/* ========================================================================= */
/* Error handling in an Apache 1.3 way */
void compat_log_error(const char *file, int line, int level,
                      const server_rec *s, const char *fmt, ...) {
    va_list ap;
    char *buf;
    char *tmp;
    pool *p=make_sub_pool(NULL);

    open_logs((server_rec *)s,p);
    va_start(ap,fmt);
    tmp=compat_psprintf(p,"(LEVEL=%d) %s",level,fmt);
    buf=compat_pvsprintf(p,tmp,ap);
    va_end(ap);

    log_error(buf,(server_rec *)s);
    
    destroy_pool(p);
    return;
}

/* ========================================================================= */
/* Socket cleanup function */
static void socket_cleanup(void *fdv) {
    close((int) (long) fdv);
}

/* ========================================================================= */
/* Opens a socket and put a cleanup function for closing in pool */
int compat_psocket(pool *p, int domain, int type, int protocol) {
    int fd;

    block_alarms();
    fd = socket(domain, type, protocol);
    if (fd == -1) {
    	int save_errno = errno;
    	unblock_alarms();
    	errno = save_errno;
    	return -1;
    }
    register_cleanup(p,(void *) (long) fd, socket_cleanup, socket_cleanup);
    unblock_alarms();
    return fd;
}

/* ========================================================================= */
/* Close socket and removes function cleanup from pool */
int compat_pclosesocket(pool *a, int sock) {
    int res;
    int save_errno;

    block_alarms();
    res = close(sock);
#ifdef WIN32
    errno = WSAGetLastError();
#endif /* WIN32 */
    save_errno = errno;
    kill_cleanup(a, (void *) (long) sock, socket_cleanup);
    unblock_alarms();
    errno = save_errno;
    return res;
}

/* ========================================================================= */
/* Scan script header using BUFF instead of FILE */
int compat_scan_script_header_err_buff(request_rec *r, BUFF *f, char *buffer) {
    const char *MALFORMED_MESSAGE="malformed header from script. Bad header=";
    int MALFORMED_HEADER_LENGTH_TO_SHOW=30;
    char x[MAX_STRING_LEN];
    char *w, *l;
    int p;

    if (buffer) *buffer = '\0';
    w = buffer ? buffer : x;

    hard_timeout ("read script header", r);
    
    while(1) {
        if (bgets(w, MAX_STRING_LEN-1, f) <= 0) {
            kill_timeout (r);
            log_reason ("Premature end of script headers", r->filename, r);
            return SERVER_ERROR;
        }

        /* Delete terminal (CR?)LF */
        
        p = strlen(w);
        if (p > 0 && w[p-1] == '\n')
        {
            if (p > 1 && w[p-2] == '\015') w[p-2] = '\0';
            else w[p-1] = '\0';
        }

        if(w[0] == '\0') {
            kill_timeout (r);
            return OK;
        }

        /* if we see a bogus header don't ignore it. Shout and scream */
        
        if(!(l = strchr(w,':'))) {
            char malformed[(sizeof MALFORMED_MESSAGE)+1+MALFORMED_HEADER_LENGTH_TO_SHOW];
            strcpy(malformed, MALFORMED_MESSAGE);
            strncat(malformed, w, MALFORMED_HEADER_LENGTH_TO_SHOW);

            if (!buffer)
              /* Soak up all the script output --- may save an outright kill */
              while (bgets(w, MAX_STRING_LEN-1, f) > 0)
                continue;
            
            kill_timeout (r);
            log_reason (malformed, r->filename, r);
            return SERVER_ERROR;
        }

        *l++ = '\0';
        while (*l && isspace (*l)) ++l;
        
        if(!strcasecmp(w,"Content-type")) {

            /* Nuke trailing whitespace */
            
            char *endp = l + strlen(l) - 1;
            while (endp > l && isspace(*endp)) *endp-- = '\0';
            
            r->content_type = pstrdup (r->pool, l);
        }
        else if(!strcasecmp(w,"Status")) {
            sscanf(l, "%d", &r->status);
            r->status_line = pstrdup(r->pool, l);
        }
        else if(!strcasecmp(w,"Location")) {
            table_set (r->headers_out, w, l);
        }   
        else if(!strcasecmp(w,"Content-Length")) {
            table_set (r->headers_out, w, l);
        }   
        else if(!strcasecmp(w,"Transfer-Encoding")) {
            table_set (r->headers_out, w, l);
        }   

        else if(!strcasecmp(w, "Set-Cookie")) {
            table_add(r->err_headers_out, w, l);
        }
        else {
            table_merge (r->err_headers_out, w, l);
        }
    }
}

/* ========================================================================= */
/* Sends BUFF contents to request_rec */
long compat_send_fb(BUFF *fb, request_rec *r) {
    char buf[MAX_STRING_LEN];
    while (bgets(buf, MAX_STRING_LEN-1, fb) > 0) rputs(buf,r);
}

#endif /* if MODULE_MAGIC_NUMBER < 19980527 */