/*
 * Copyright (c) 1997-1998 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */

package org.apache.jservssi;

import java.io.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;
import org.apache.java.util.*;

/**
 * <CODE>JServSSI</CODE> provides support for including dynamic servlet
 * output from within HTML documents via the <CODE>&lt;SERVLET&gt;</CODE> tag.
 *
 * <h3>Configuring the Apache Server</h3>
 * <blockquote>
 *
 * <p>In order to take advantage of the capabilities provided by this
 * class, you must first configure your Apache web server so that it
 * knows how to pass documents which will contain
 * <CODE>&lt;SERVLET&gt;</CODE> tags to this class.
 *
 * <p>Typically, this is done by naming files with <CODE>SERVLET</CODE>
 * tags with a ".jhtml" extension, and adding the following lines to
 * your Apache server configuration file:
 *
 *   <code><pre>
 *   # enable &lt;servlet&gt; tag in .jhtml files
 *   ApJServAction .jhtml /<i>servletZone</i>/org.apache.jserv.servlets.JServSSI
 *   </pre></code>
 *
 * where <i>servletZone</i> is the URI from which your servlets
 * are accessed. (This is often "<code>/servlets</code>".)
 *
 * </blockquote>
 * <h3>Including Servlets in HTML Pages</h3>
 * <blockquote>
 *
 * <h4>The <CODE>&lt;SERVLET&gt;</CODE>
 *  and <CODE>&lt;/SERVLET&gt;</CODE> tags</h4>
 *
 * <p>Servlet output may be included in an HTML document by use of
 * the <CODE>&lt;SERVLET&gt;</CODE> tag.  For example, to embed the
 * output of the demo "HelloWorldServlet" servlet in an HTML page, you
 * might write the following:
 *
 *   <code><pre>
 *   <i>... (some HTML code) ...</i>
 *   &lt;SERVLET CODE="HelloWorldServlet.class"&gt;
 *   Your web server has not been configured to support servlet tags.
 *   &lt;/SERVLET&gt;
 *   <i>... (more HTML code) ...</i>
 *   </pre></code>
 *
 * <p>When this page is served from the web server, the code above
 * will be replaced with the output of HelloWorldServlet.
 * If you see the message between the tags instead, there is a problem
 * with your server configuration.  If this happens, check to make sure
 * your file has a ".jhtml" extension and that the Apache server is
 * configured as described above.
 *
 * <p>Two attributes are used by the <CODE>SERVLET</CODE> tag,
 * and they are roughly equivalent:
 * <ol>
 * <p><li>The <CODE>CODE</CODE> attribute may be set to the name of a class file
 *     (the ".class" extension is optional) containing the servlet
 *     to be run.  Currently, this servlet must be installed in the
 *     same directory as the other servlets, not in the directory
 *     where your HTML resides.
 * <p><li>The <CODE>NAME</CODE> attribute may also be set to the name of the
 *     servlet to be run (with no class extension).
 * </ol>
 * <p>In some implementations of <CODE>SERVLET</CODE> tags, if
 * both <CODE>NAME</CODE> and <CODE>CODE</CODE> attributes are set, the
 * servlet designated by <CODE>CODE</CODE> will then become available for
 * future use under the symbolic named designated by the <CODE>NAME</CODE>
 * attribute.  This is not currently supported.
 *
 * <p>Note that both the <CODE>&lt;SERVLET&gt;</CODE> and
 * <CODE>&lt;/SERVLET&gt;</CODE> tags must be present.
 *
 * <h4>Initialization parameters</h4>
 *
 * <p>With this newer version of JServSSI, it is now possible to
 * to specify initialization parameters correctly -- in the
 * <CODE>&lt;SERVLET&gt;</CODE> tag, that is.
 *
 * <p>If no extra parameters beyond <CODE>NAME</CODE> and
 * <CODE>CODE</CODE> are supplied, initialization does not happen
 * when the servlet is invoked.  If a set of parameters is supplied to the
 * servlet the <CODE>SERVLET</CODE> tag, like so:
 *
 * <PRE>
 * &lt;SERVLET code="SnoopSerlvet" abc="One Two Three" Xyz=456&gt;
 * </PRE>
 *
 * <p>Then SnoopServlet will be initialized with these parameters
 * when this servlet is invoked this way.  If two <CODE>SERVLET</CODE>
 * tags supply two different set of parameters, then each one is
 * supplied before the <code>service</code> method is invoked.
 * This is hardly optimal, but is dictated by the architecture.
 *
 * <h4>The <CODE>&lt;PARAM&gt;</CODE> tag</h4>
 *
 * <p>You may send parameters to a servlet via the <CODE>PARAM</CODE> tag, which
 * should be placed between the <CODE>&lt;SERVLET ... &gt;</CODE> and
 * <CODE>&lt;/SERVLET&gt;</CODE> tags, like so:
 *
 *   <code><pre>
 *   &lt;SERVLET CODE="MyServlet.class"&gt;
 *   &lt;PARAM NAME="param1" VALUE="valueOfParam1"&gt;
 *   &lt;PARAM NAME="anotherParam" VALUE="valueOfAnotherParam"&gt;
 *   &lt;/SERVLET&gt;
 *   </pre></code>
 *
 * <p>You could then access these parameters from your
 * servlet as follows:
 *
 *   <code><pre>
 *   public void doGet(HttpServletRequest req, HttpServletResponse res)
 *       throws ServletException, IOException
 *   {
 *       String param1 = req.getParameter("param1");
 *       String anotherParam = req.getParameter("anotherParam");
 *   }
 *   </pre></code>
 *
 * </blockquote>
 * <h3>Notes</h3>
 * <blockquote>
 *
 * <dl>
 * <p><dt><b>Attribute values</b>
 * <dd>Attribute values, <CODE>NAME</CODE> and <CODE>VALUE</CODE>
 * in the <CODE>PARAM</CODE> tag may be a single word
 * (<CODE>NAME=xyz value=1</CODE>) or must be enclosed in quotes
 * if they contain whitespace
 * (<CODE>NAME=xyz VALUE="This is a test"</CODE>).
 *
 * <p><dt><b>Case sensitivity and SGML</b>
 * <dd>This class does not care about case when reading
 * SGML tags or their attributes, so uppercase, or lowercase, or any
 * combination thereof may be used.  The text in attribute values
 * is not translated to upper- or lowercase but is passed on intact.
 *
 * <p><dt><b>Error handling</b>
 * <dd>To simplify error detection and correction,
 * exceptions thrown by JServSSI or called servlets are printed,
 * enclosed in comments ("&lt;!-- ... --&gt;"), in the HTML output.
 *
 * <p><dt><b>Known bugs</b>
 * <dd>Currently, the parameters set for the first
 * servlet on a page will still be set in the next servlet.
 * </dl>
 * </blockquote>
 *
 * @author Roger Zeng
 * @author Tim Williams
 * @version $Revision: 1.1.1.1 $ $Date: 1998/11/26 23:23:04 $
 * @see javax.servlet.http.HttpServlet
 */

/* TODO:
 * (from Tim Williams, williams@ugsolutions.com:)
 * - Support "CODEBASE" servlet tag, allowing remote loading of servlets
 * - Make "NAME" parameter work for symbolic names, not just "CODE"
 * - Integrate functionality with James' modifications (reddirt@texas.edu)
 */
 
public class JServSSI extends HttpServlet {
    
    // cache parsed pages and last modification time for such
    private Hashtable pages = null;
    private Hashtable times = null;

    /**
     * Initialize JServSSI.
     *
     * @param config servlet configuration, stored by superclass
     * @exception ServletException passed on from superclass
     */
    public void init(ServletConfig config) throws ServletException {
        super.init(config);
        if (pages == null) {
            pages = new Hashtable();
            times = new Hashtable();
        }
    }

    /**
     * Handle POST the same as GET.
     * This method is simply a call to doGet().
     *
     * @param req encapsulates the request to the servlet
     * @param resp encapsulates the response from the servlet
     * @see javax.servlet.http.HttpServletRequest#getPathTranslated
     * @exception ServletException will be passed on from included servlets
     * @see #doGet
     */
    public void doPost(HttpServletRequest req, HttpServletResponse res)
    throws ServletException, IOException {
        doGet(req, res);
    }

    /**
     * Process named page HTML passed found via getPathTranslated().
     *
     * @param req encapsulates the request to the servlet
     * @param resp encapsulates the response from the servlet
     * @see javax.servlet.http.HttpServletRequest#getPathTranslated
     * @exception ServletException will be passed on from included servlets
     */
    public void doGet(HttpServletRequest req, HttpServletResponse res)
    throws ServletException, IOException {
        res.setContentType("text/html");

        // parse file contents
        String path = req.getPathTranslated();

        if (path == null) {
            error("Could not resolve the given URI <B>" + req.getRequestURI()
                + "<P><B>NOTE</B>: JServSSI cannot work if the requested "
                + "file is not accessible by the servlet engine. "
                + "Please, make sure the requested file is readable by "
                + "this servlet and located on the same machine.", 
                res.getOutputStream());
        } else {
            File file = new File(path);
            if (!file.isFile()) {
                error("Requested URI <B>" + req.getRequestURI()
                    + "</B> is not a file. "
                    + "Please, make sure you request a "
                    + "java server side include file.", res.getOutputStream());
            } else if (!file.exists()) {
                error("Requested URI <B>" + req.getRequestURI()
                    + "</B> was not found. "
                    + "Please, make sure this file exists."
                    + "<P><B>NOTE</B>: JServSSI cannot work if the requested "
                    + "file is not accessible by the servlet engine. "
                    + "Please, make sure the requested file is located on the"
                    + "same machine.", res.getOutputStream());
            } else if (!file.canRead()) {
                error("Requested URI <B>" + req.getRequestURI()
                    + "</B> could not be accessed. "
                    + "Please, make sure JServSSI has rights "
                    + "to read the requested file.", res.getOutputStream());
            } else {
                // retrieve and interpret page
                interpretPage(getPageParts(file), req, res);
            }
        }

        res.getOutputStream().close();
    }

    /**
     * Sends error as servlet output
     */
    private void error(String message, ServletOutputStream output)
    throws IOException {
        output.println("<HTML><BODY>"
            + "<H2 ALIGN=center>Java Server Side Include Error</H2><p>"
            + message
            + "<hr><center>Automatically generated by:"
            + "<br><b>Apache JServSSI Servlet</b></center>"
            + "</BODY></HTML>");
    }

    /**
     * Clean up servlet when finished.
     */
    public void destroy() {
        super.destroy();
        pages = times = null;
    }

    /**
     * If HTML file has changed or has not yet been loaded, load
     * file and chop into sections, storing result for future use.
     * Otherwise, return stored preprocessed page.
     *
     * @param file file for which we want page section list
     * @return list of page sections, as described in parsePage().
     * @see #parsePage
     */
    private Vector getPageParts(File file) throws IOException {
        // first, check to see if we have cached version
        Vector parts = (Vector) pages.get(file);
        if (parts == null || ((Long)times.get(file)).longValue() < file.lastModified()) {
            // if no cached version, or modified, load
            times.put(file, new Long(file.lastModified()));
            parts = parsePage(file);
            pages.put(file, parts);
        }
        return parts;
    }

    /**
     * Scan through vector of page sections, either printing them
     * or invoking the servlets to which they refer.
     *
     * @param parts page sections, as provide by parsePage()
     * @see #parsePage
     */
    private void interpretPage(Vector parts, HttpServletRequest req, HttpServletResponse res)
    throws ServletException, IOException {
        ServletOutputStream out = res.getOutputStream();
        for (int i = 0; i < parts.size(); i++) {
            Object part = parts.elementAt(i);
            if (part instanceof ServletInfo) {
                ServletInfo info = (ServletInfo) part;
                ServletInfoRequest ireq = info.createRequest(req);
                ireq.runServlet(info, getServletContext(), res);
            } else {
                out.print(part.toString());
            }
        }
    }

    /**
     * Open and read file, returning list of contents.
     * The returned vector will contain several types of object:
     * <ul>
     * <li>Objects of class java.lang.String which are parts of
     *     the page which are simply to be echoed verbatim
     * <li>Objects of class SGMLTag which represent any &lt;servlet&gt;
     *     tags encountered and their attributes
     * <li>After each SGMLTag is a java.util.Hashtable, containing
     *     PARAM values set for the preceeding SERVLET tag.
     * </ul>
     * @param file file to open and process
     * @return Vector containing page elements.
     */
    private Vector parsePage(File file) throws IOException {
        // read file contents
        FileReader fin = new FileReader(file);
        BufferedReader br = new BufferedReader(fin);
        String line, text = "";
        while ((line = br.readLine()) != null) {
            text += line + "\n";
        }

        // scan through page parsing servlet statements
        Vector sections = new Vector();
        int start = 0;
        for (SGMLTag tag = new SGMLTag(text, 0); !tag.finished();
            tag = new SGMLTag(text, tag.end))
        {
            if (tag.isNamed("SERVLET") && tag.isWellFormed()) {
                SGMLTag servlet = tag;
                Hashtable params = new Hashtable();
                
                while (!tag.finished()) {
                    tag = new SGMLTag(text, tag.end);
                    if (tag.isNamed("PARAM") && tag.isWellFormed()) {
                        // set parameter
                        String name = tag.value("NAME", null);
                        String value = tag.value("VALUE", null);
                        if (name != null && value != null) {
                            params.put(name, value);
                        }
                    } else if (tag.isNamed("/SERVLET") && tag.isWellFormed()) {
                        // handle end servlet tag
                        sections.addElement(text.substring(start, servlet.start));
                        ServletInfo info = new ServletInfo(getServletConfig(), servlet, params);
                        sections.addElement(info);
                        start = tag.end;
                        break;
                    }
                }
            }
        }
        
        sections.addElement(text.substring(start, text.length()));
        return sections;
    }

    /**
     * Return information about JServSSI to caller.
     *
     * @return string explaining this what this servlet does
     * and referring the user to the javadocs.
     */
    public String getServletInfo() {
        return "Apache JServSSI is a servelet which processes <SERVLET> tags "
            + "in .jhtml files and replaces such tags with the "
            + "output of the servlets to which they refer.";
    }
}

/**
 * Class to assist in setting attributes and parameters for
 * sub-servlets.  One must call setRequest before using this class as
 * a ServletRequest object for a sub-servlet.
 *
 * @author Tim Williams <williams@ugsolutions.com>
 */
class ServletInfo implements ServletConfig {
    
    private SGMLTag tag;
    private ServletConfig config;
    private Hashtable params;
    private boolean requiresInit = false;

    /*---- Class instantiation and maintenance ----*/

    ServletInfo(ServletConfig config, SGMLTag tag, Hashtable params) {
        // store passed information
        this.tag = tag;
        this.config = config;
        this.params = params;
    
        // determine if this has any initialization parameters of its own
        Enumeration names = getInitParameterNames();
        while (names.hasMoreElements() && !requiresInit) {
            String name = ((String) names.nextElement()).toUpperCase();
            if (!name.equals("NAME") && !name.equals("CODE"))
            requiresInit = true;
        }
    }

    SGMLTag getSGMLTag() {
        return tag;
    }

    ServletInfoRequest createRequest(HttpServletRequest req) {
        return new ServletInfoRequest(params, req);
    }

    boolean requiresInit() {
        return requiresInit;
    }

    /*---- ServletConfig methods ----*/

    /**
     * Returns the context for the servlet.
     */
    public ServletContext getServletContext() {
        return config.getServletContext();
    }

    /**
     * Returns a string containing the value of the named
     * initialization parameter of the servlet, or null if the
     * parameter does not exist.  Init parameters have a single string
     * value; it is the responsibility of the servlet writer to
     * interpret the string.
     *
     * @param name the name of the parameter whose value is requested
     * @return value of name, or null if unset
     */
    public String getInitParameter(String name) {
        return tag.value(name, config.getInitParameter(name));
    }

    /**
     * Returns the names of the servlet's initialization parameters
     * as an enumeration of strings, or an empty enumeration if there
     * are no initialization parameters.
     *
     * @return enumeration of key values as strings
     */
    public Enumeration getInitParameterNames() {
        return union(tag.attributes(false), config.getInitParameterNames());
    }

    /*---- Internal utilities ----*/

    /**
     * Given two enumerations, create an enumeration containing
     * elements in both of them, with redundant elements only occuring
     * once.  Elements are returned in no particular order.
     *
     * @param a first enumeration
     * @param b second enumeration
     * @return union of elements found in sets
     */
    static Enumeration union(Enumeration a, Enumeration b) {
        Hashtable seen = new Hashtable();
        while (a.hasMoreElements())
            seen.put(a.nextElement(), new Boolean(true));
        while (b.hasMoreElements())
            seen.put(b.nextElement(), new Boolean(true));
        return seen.keys();
    }
}

class ServletInfoRequest implements HttpServletRequest {
    private Hashtable params;
    private HttpServletRequest req;
    private static Hashtable lastInfo = new Hashtable();

    ServletInfoRequest(Hashtable params, HttpServletRequest req) {
        this.params = params;
        this.req = req;
    }

    /*---- ServletRequest methods ----*/

    /**
     * Returns a buffered reader for reading text in the request body.
     * This translates character set encodings as appropriate.
     * @see getInputStream
     */
    public BufferedReader getReader() throws IOException {
        return req.getReader();
    }

    /**
     * Returns the size of the request entity data, or -1 if not known.
     * Same as the CGI variable CONTENT_LENGTH.
     */
    public int getContentLength() {
        return req.getContentLength();
    }

    /**
     * Returns the Internet Media Type of the request entity data, or
     * null if not known. Same as the CGI variable CONTENT_TYPE.
     */
    public String getContentType() {
        return req.getContentType();
    }

    /**
     * Returns the protocol and version of the request as a string of
     * the form <code>&lt;protocol&gt;/&lt;major version&gt;.&lt;minor
     * version&gt</code>.  Same as the CGI variable SERVER_PROTOCOL.
     */
    public String getProtocol() {
        return req.getProtocol();
    }

    /**
     * Returns the scheme of the URL used in this request, for example
     * "http", "https", or "ftp".  Different schemes have different
     * rules for constructing URLs, as noted in RFC 1738.  The URL used
     * to create a request may be reconstructed using this scheme, the
     * server name and port, and additional information such as URIs.
     */
    public String getScheme() {
        return req.getScheme();
    }

    /**
     * Returns the host name of the server that received the request.
     * Same as the CGI variable SERVER_NAME.
     */
    public String getServerName() {
        return req.getServerName();
    }

    /**
     * Returns the port number on which this request was received.
     * Same as the CGI variable SERVER_PORT.
     */
    public int getServerPort() {
        return req.getServerPort();
    }

    /**
     * Returns the IP address of the agent that sent the request.
     * Same as the CGI variable REMOTE_ADDR.
     */
    public String getRemoteAddr() {
        return req.getRemoteAddr();
    }

    /**
     * Returns the fully qualified host name of the agent that sent the
     * request. Same as the CGI variable REMOTE_HOST.
     */
    public String getRemoteHost() {
        return req.getRemoteHost();
    }

    /**
     * Applies alias rules to the specified virtual path and returns
     * the corresponding real path, or null if the translation can not
     * be performed for any reason.  For example, an HTTP servlet would
     * resolve the path using the virtual docroot, if virtual hosting
     * is enabled, and with the default docroot otherwise.  Calling
     * this method with the string "/" as an argument returns the
     * document root.
     *
     * @param path the virtual path to be translated to a real path
     */
    public String getRealPath(String path) {
        return req.getRealPath(path);
    }

    /**
     * Returns an input stream for reading the request body.
     */
    public ServletInputStream getInputStream() throws IOException {
        return req.getInputStream();
    }

    /**
     * Returns a string containing the lone value of the specified
     * parameter, or null if the parameter does not exist. For example,
     * in an HTTP servlet this method would return the value of the
     * specified query string parameter. Servlet writers should use
     * this method only when they are sure that there is only one value
     * for the parameter.  If the parameter has (or could have)
     * multiple values, servlet writers should use
     * getParameterValues. If a multiple valued parameter name is
     * passed as an argument, the return value is implementation
     * dependent: for this class, the parameter set with a
     * <tt>&lt;PARAM&gt;</tt> tag will be preferred to those posted by
     * the user.
     *
     * @param name the name of the parameter whose value is required.
     * @deprecated Please use getParameterValues
     * @see javax.servlet.ServletRequest#getParameterValues
     */
    public String getParameter(String name) {
        String value = (String) params.get(name);
        String rvalues[] = req.getParameterValues(name);
        return (value != null) ? value : (rvalues.length > 0 ? rvalues[0] : null);
    }

    /**
     * Returns the values of the specified parameter for the request as
     * an array of strings, or null if the named parameter does not
     * exist.  In an HTTP servlet included via the
     * <tt>&lt;SERVLET&gt;</tt> tag, these values would also include
     * any values set via a <tt>&lt;PARAM&gt;</tt> tag.
     *
     * @param name the name of the parameter whose value is required
     * @return list of parameter values
     * @see javax.servlet.ServletRequest#getParameter
     */
    public String[] getParameterValues(String name) {
        String[] values = req.getParameterValues(name);
        if (params.containsKey(name)) {
            String[] these;
            if (values != null) {
                these = new String[values.length+1];
                System.arraycopy(values, 0, these, 1, values.length);
            } else {
                these = new String[1];
            }
            these[0] = (String) params.get(name);
            values = these;
        }
        return values;
    }

    /**
     * Returns the parameter names for this request as an enumeration
     * of strings, or an empty enumeration if there are no parameters
     * or the input stream is empty.  The input stream would be empty
     * if all the data had been read from the stream returned by the
     * method getInputStream.
     */
    public Enumeration getParameterNames() {
        return ServletInfo.union(params.keys(), req.getParameterNames());
    }

    /**
     * Returns the value of the named attribute of the request, or
     * null if the attribute does not exist.  This method allows
     * access to request information not already provided by the other
     * methods in this interface.  Attribute names should follow the
     * same convention as package names.  The package names java.*,
     * and javax.* are reserved for use by Javasoft, and com.sun.* is
     * reserved for use by Sun Microsystems.
     *
     * @param name the name of the attribute whose value is required
     */
    public Object getAttribute(String name) {
        return req.getAttribute(name);
    }

    /*---- HttpServletRequest ----*/

    /**
     * Returns the method with which the request was made. The returned
     * value can be "GET", "HEAD", "POST", or an extension method. Same
     * as the CGI variable REQUEST_METHOD.
     */
    public String getMethod() {
        return req.getMethod();
    }

    /**
     * Returns the request URI as a URL object.
     */
    public String getRequestURI() {
        return req.getRequestURI();
    }

    /**
     * Returns the part of the request URI that refers to the servlet
     * being invoked. Analogous to the CGI variable SCRIPT_NAME.
     */
    public String getServletPath() {
        // FIXME: giving JServSSI instead of sub-servlet!
        return req.getServletPath();
    }

    /**
     * Returns optional extra path information following the servlet
     * path, but immediately preceding the query string. Returns null if
     * not specified. Same as the CGI variable PATH_INFO.
     */
    public String getPathInfo() {
        return req.getPathInfo();
    }

    /**
     * Returns extra path information translated to a real path. Returns
     * null if no extra path information specified. Same as the CGI variable
     * PATH_TRANSLATED.
     */
    public String getPathTranslated() {
        return req.getPathTranslated();
    }

    /**
     * Returns the query string part of the servlet URI, or null if none.
     * Same as the CGI variable QUERY_STRING.
     */
    public String getQueryString() {
        return req.getQueryString();
    }

    /**
     * Returns the name of the user making this request, or null if not
     * known.  The user name is set with HTTP authentication.  Whether
     * the user name will continue to be sent with each subsequent
     * communication is browser-dependent.  Same as the CGI variable
     * REMOTE_USER.
     */
    public String getRemoteUser() {
        return req.getRemoteUser();
    }

    /**
     * Returns the authentication scheme of the request, or null if none.
     * Same as the CGI variable AUTH_TYPE.
     */
    public String getAuthType() {
        return req.getAuthType();
    }

    /**
     * Returns the value of a header field, or null if not known.
     * The case of the header field name is ignored.
     *
     * @param name the case-insensitive header field name
     */
    public String getHeader(String name) {
        return req.getHeader(name);
    }

    /**
     * Returns the value of an integer header field, or -1 if not found.
     * The case of the header field name is ignored.
     *
     * @param name the case-insensitive header field name
     */
    public int getIntHeader(String name) {
        return req.getIntHeader(name);
    }

    /**
     * Returns the value of a date header field, or -1 if not found.
     * The case of the header field name is ignored.
     *
     * @param name the case-insensitive header field name
     */
    public long getDateHeader(String name) {
        return req.getDateHeader(name);
    }

    /**
     * Returns an enumeration of strings representing the header names
     * for this request. Some server implementations do not allow headers
     * to be accessed in this way, in which case this method will return null.
     */
    public Enumeration getHeaderNames() {
        return req.getHeaderNames();
    }

    /**
     * Checks whether this request is associated with a session that is
     * valid in the current session context. If it is not valid, the
     * requested session will never be returned from the getSession
     * method.
     */
    public boolean isRequestedSessionIdValid() {
        return req.isRequestedSessionIdValid();
    }

    /**
     * Checks whether the session id specified by this request came
     * in as part of the URL. (The requested session may not be the
     * one returned by the getSession method.)
     */
    public boolean isRequestedSessionIdFromCookie() {
        return req.isRequestedSessionIdFromCookie();
    }

    public HttpSession getSession(boolean value) {
        return req.getSession(value);
    }

    public String getCharacterEncoding() {
        return req.getCharacterEncoding();
    }

    public String getRequestedSessionId() {
        return req.getRequestedSessionId();
    }

    public Cookie[] getCookies() {
        return req.getCookies();
    }

    public boolean isRequestedSessionIdFromUrl() {
        return req.isRequestedSessionIdFromUrl();
    }

    /*---- Local Stuff ----*/

    /**
     * Invoke servlet for this page section.
     */
    void runServlet(ServletInfo info, ServletContext context, HttpServletResponse res)
    throws ServletException, IOException {
        try {
            // figure which servlet we want
            SGMLTag tag = info.getSGMLTag();
            String className = tag.value("CODE", null);
            if (className != null) {
                if (className.endsWith(".class")) {
                    className = className.substring(0,
                        className.length() - new String(".class").length());
                }
            } else {
                className = tag.value("NAME", null);
                if (className == null) {
                    throw new ServletException("no class named in servlet tag");
                }
            }
    
            // load servlet
            HttpServlet servlet = (HttpServlet) context.getServlet(className);
            if (servlet == null) {
                throw new ClassNotFoundException("can't find servlet named \"" 
                    + className + "\"");
            }
    
            // determine if initialization is required
            if (info.requiresInit() && lastInfo.get(servlet) != info) {
                servlet.init(info);
                lastInfo.put(servlet, info);
            }
    
            // execute servlet
            servlet.service((ServletRequest) this, (ServletResponse) res);
        }  catch (Exception e) {
            // display error messages
            PrintWriter out = new PrintWriter(res.getOutputStream());
            out.print("<!-- ");
            e.printStackTrace(out);
            out.print("-->");
            out.flush();
    
            // pass servlet/IO problems along
            if (e instanceof ServletException) {
                throw (ServletException) e;
            } else if (e instanceof IOException) {
                throw (IOException) e;
            }
        }
    }
}

