// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

// JServConnection.java:
// - org.apache.jserv.JServConnection
// - org.apache.jserv.JServConnection.JServInputStream
// - org.apache.jserv.JServConnection.JServOutputStrem

package org.apache.jserv;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.DataInputStream;
import java.io.OutputStream;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;

import java.net.Socket;
import java.net.InetAddress;

import java.util.Date;
import java.util.Vector;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;
import java.util.TimeZone;

import java.text.SimpleDateFormat;
import java.text.ParseException;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.ServletInputStream;
import javax.servlet.ServletOutputStream;

import javax.servlet.http.HttpUtils;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * This class is the thread that handles all communications between
 * the Java VM and the web server.
 *
 * @author Alexei Kosut
 **/
public class JServConnection extends Thread
implements HttpServletRequest, HttpServletResponse, JServSendError,
    JServDebug.DebugConstants
{
    private Socket client;
    private DataInputStream in;
    private OutputStream out;

    private Hashtable servletMgrTable;

    // Servlet setup stuff
    private JServServletManager mgr;
    private JServContext context;
    private Servlet servlet;
    private String servletname;
    private String hostname;

    private String auth;
    private String signal;

    private JServInputStream servlet_in;
    private JServOutputStream servlet_out;

    // HTTP Stuff
    private Hashtable headers_in;
    private Hashtable headers_out;
    private Hashtable env_vars;

    private int status;
    private String status_string;

    private Hashtable params;
    private boolean got_input;

    private boolean sent_header;


    /**
     * Initalize the streams and starts the thread.
     */
    public JServConnection(Socket client_socket, Hashtable mgrTable) {
	this.client = client_socket;
	this.servletMgrTable = mgrTable;

	// Check that the request comes from localhost.  We check
	// 127.0.0.1 explicitly, even though theoretically the first
	// check should find it, because apparently the Linux JDK is
	// buggy here.

	// mbp: I think the reason is that you have to check for both
	// the host's internet address (eg 147.132.24.207) and the
	// loopback address (127.0.0.1): the client may appear to have
	// connected from either, depending on the stack.
        if (JServHandler.localhostcheck) {
	    if ( !client.getInetAddress().equals(client.getLocalAddress()) &&
		 !client.getInetAddress().getHostAddress().equals("127.0.0.1"))
		{
		    try {
			client.close();
		    } catch  (IOException e) {
		    }
		    return;
		}
        }

	try {
	    this.in = new DataInputStream ( new BufferedInputStream( client.getInputStream() ) );
	    this.out = new BufferedOutputStream( client.getOutputStream() );
	} catch(IOException e) {
	    try {
		client.close();
	    } catch(IOException e2) {
	    }
	    JServDebug.trace( "Exception while getting socket streams: ",
			      SERVICE_REQUEST );
	    JServDebug.trace( e );
	    return;
	}
	this.start();
    }

    // Provide the service.
    public void run() {
	JServDebug.trace( "Initializing servlet request", SERVICE_REQUEST );

	// Set up hash tables
	headers_in = new Hashtable();
	headers_out = new Hashtable();
	env_vars = new Hashtable();

	// Set up HTTP defaults
	status = SC_OK;
	status_string = null;

	sent_header = false;
	got_input = false;

	auth = null;
	signal = null;

	// Read in the data
	try {
	    JServDebug.trace( "Reading request data", SERVICE_REQUEST );
	    // Read data, and check to make sure it read ok
	    boolean readok = readData();

	    if (!readok) {
		sendError(SC_INTERNAL_SERVER_ERROR,
			  "Malformed data send to JServ");
		return;
	    }

	    if (!JServHandler.checkAuthString(auth)) {
		sendError(SC_FORBIDDEN,
			  "Illegal access to JServ: incorrect auth string");
		return;
	    }
	}
	catch(IOException e) {
	    try {
		client.close();
	    }
	    catch(IOException e2) {
	    }
	    return;
	}

        // Look for a signal 
        if (signal != null) {
 	    JServDebug.trace("received signal "+signal, SIGNAL);
            if (signal.equals("01"))  // SIGHUP
                JServHandler.signal_handler.restartServlets();
            else if (signal.equals("15")) // SIGTERM
                JServHandler.signal_handler.destroyServlets(); 
            return;
        }

	// Look for the servlet

	if (servletname == null) {
	    sendError(SC_NOT_FOUND, "Received empty servlet name");
	    return;
	}

	//Find the servlet manager that handles this host
	JServServletManager mgr =
	    (JServServletManager) servletMgrTable.get(hostname);

	if (mgr == null) {
	    mgr = (JServServletManager) servletMgrTable.get("_default_");
	}

	if (mgr == null) {
	    sendError(SC_INTERNAL_SERVER_ERROR,
		      "No servlet manager setup for host \"" +
		      hostname + "\".");
	    return;
	}
	mgr.checkReload( this );
	
	try {
	  context = mgr.loadServlet(servletname, this);
	} catch ( ServletException initError ) {
	  sendError( initError );
	  return;
	}
	if (context == null  ||  context.servlet == null) {
	    sendError(SC_INTERNAL_SERVER_ERROR,
		      "An unknown error occured loading the servlet.");
	    return;
	}
	servlet = context.servlet;

	// Set up a read lock on the servlet. Note that anytime
	// we return, we need to make sure to unlock this. Otherwise,
	// we'll end up holding onto the lock forever. Oops.
	// This is done in the finally clause.
	try {
	    context.lock.readLock();

	    // Set up the servlet's I/O streams
	    servlet_in = new JServInputStream();
	    servlet_out = new JServOutputStream();

	    // Start up the servlet
	    try {
		JServDebug.trace( "Calling service()", SERVICE_REQUEST );
		servlet.service(this, this);
	    } catch(Exception e) {
		sendError(e);
		return;
	    } catch(Error e) {
		sendError(e);
		throw (Error) e.fillInStackTrace();
	    }

	    // Make sure we've send the HTTP header, even if no 
	    // entity data has been
	    sendHttpHeaders();

	    // All done; close the connection
	    try {
		out.flush();
		client.close();
	    }
	    catch(IOException e) {
	    }
	    return;
	}
	finally {
	    //Clean up 
	    context.lock.readUnlock();
	}
    }

    // Read in a line from the input

    private String readHexLine() throws IOException {
	int len=0;
	byte hex[] = new byte[4];
	byte str[];

	try {
	    // Read four bytes from the input stream
	    in.readFully(hex);

	    // Convert them from hex to decimal
	    len = Character.digit((char)hex[0], 16);
	    len = len << 4;
	    len += Character.digit((char)hex[1], 16);
	    len = len << 4;
	    len += Character.digit((char)hex[2], 16);
	    len = len << 4;
	    len += Character.digit((char)hex[3], 16);
	    
 	    JServDebug.trace( "will read "+len+" bytes for this line", 
 			      REQUEST_DATA );
	    // Read len bytes from the input stream
	    str = new byte[len];
	    in.readFully(str);
	}
	catch  (NumberFormatException e) {
	    return null;
	}

	// Return str as an String
	return new String(str);
    }

    // Read in all the data
    private boolean readData() throws IOException {
	for (;;) {
	    String line = readHexLine();
	    JServDebug.trace ("read " + line, REQUEST_DATA );
	    char id;

	    // Oops..
	    if   (line == null) {
		JServDebug.trace("Unexpected end of data.", REQUEST_DATA );
		return false;
	    }			
	    // Alldone
	    if    (line.length() == 0) {
		JServDebug.trace("All data read.", REQUEST_DATA);
		return true;
	    }

	    // Get the identifier from the first character
	    try {
		id = line.charAt(0);
		line = line.substring(1);
	    }
	    catch(StringIndexOutOfBoundsException e) {
 		JServDebug.trace( "No data to command: " + line.charAt(0),
 				  REQUEST_DATA );
		return false;
	    }

	    // All ids' take one or two pieces of data separated by a tab.
	    StringTokenizer tokens = new StringTokenizer(line, "\t");
	    String token1, token2;

	    try {
		token1 = tokens.nextToken();
	    }
	    catch(NoSuchElementException e) {
 		JServDebug.trace("Missing first piece of data for " + id,
 				 REQUEST_DATA );
		return false;
	    }

	    try {
		token2 = tokens.nextToken();
	    }
	    catch(NoSuchElementException e) {
		token2 = null;
	    }

	    // Switch, depending on what the id is
	    switch (id) {
	    case 'A':		
		// Auth data
 		JServDebug.trace("Authorization: " + token1, REQUEST_DATA );
		auth = token1;
		//Token2 is ignored
		break;
	    case 'C':
//		String dirname = token1;	//At present we don't need this.
		servletname = token2;
 		JServDebug.trace("Servlet: " + token2, REQUEST_DATA );
		break;
	    case 'S':
		hostname = token1;
		//Token2 is ingored
 		JServDebug.trace("Hostname: " + token1, REQUEST_DATA );
		break;
	    case 'E':		// Env variable
		env_vars.put(token1, (token2 != null) ? token2 : "");
 		JServDebug.trace("Env: " + token1 + "=" + token2, 
 				 REQUEST_DATA);
		break;
	    case 'H':		// Header
		headers_in.put(token1.toLowerCase(),
			       (token2 != null) ? token2 : "");
 		JServDebug.trace("Header: " + token1 + "=" + token2,
 				 REQUEST_DATA );
		break;
           case 's':            // Signal
                signal = token1;
 		JServDebug.trace("Signal: " + token1, REQUEST_DATA );
                break;
	    default:		// What the heck is this?
 		JServDebug.trace("Unknown: " + token1 + "=" + token2,
 				 REQUEST_DATA);
		return false;
	    }
	}
    }

    /**
     * Finds a status string from one of the standard
     * status code.
     * @param sc The status code to find a descriptive string.
     * @return A string describing this status code.
     */
    public static final String findStatusString(int sc) {
	switch (sc) {
	case SC_ACCEPTED:
	    return "Accepted";
	case SC_BAD_GATEWAY:
	    return "Bad Gateway";
	case SC_BAD_REQUEST:
	    return "Bad Request";
	case SC_CREATED:
	    return "Created";
	case SC_FORBIDDEN:
	    return "Forbidden";
	case SC_INTERNAL_SERVER_ERROR:
	    return "Internal Server Error";
	case SC_MOVED_PERMANENTLY:
	    return "Moved Permanently";
	case SC_MOVED_TEMPORARILY:
	    return "Moved Temporarily";
	case SC_NO_CONTENT:
	    return "No Content";
	case SC_NOT_FOUND:
	    return "Not Found";
	case SC_NOT_IMPLEMENTED:
	    return "Method Not Implemented";
	case SC_NOT_MODIFIED:
	    return "Not Modified";
	case SC_OK:
	    return "OK";
	case SC_SERVICE_UNAVAILABLE:
	    return "Service Temporarily Unavailable";
	case SC_UNAUTHORIZED:
	    return "Authorization Required";
	default:
	    return "Response";
	}
    }				
    
    // Send the HTTP headers, prepare to send response
    protected void sendHttpHeaders() {
	if (sent_header)
	    return;

	sent_header = true;

	//Use a PrintWriter around the socket out.
	PrintWriter printOut = new PrintWriter( this.out );

 	JServDebug.trace( "Sending response headers.", SERVICE_REQUEST );
	// Send logging data
	if (context != null) {
	    synchronized(context.log_messages) {
		while (!context.log_messages.empty()) {
		    String logMsg = "Servlet-Log: " + 
			context.log_messages.pop();
		    JServDebug.trace( logMsg, RESPONSE_HEADERS );
		    printOut.print( logMsg + "\r\n");
		}
	    }
	}			
	// Send the status info

	if (status_string == null) {
	    status_string = findStatusString(status);
	}

 	String statusLine = "Status: " + status + " " + status_string;
 	printOut.print( statusLine + "\r\n" );
 	JServDebug.trace( statusLine, RESPONSE_HEADERS );

	if (headers_out != null) {
	    // Send the headers
	    for (Enumeration e = headers_out.keys(); e.hasMoreElements();) {
		String key = (String) e.nextElement();
 		String hdr = key + ": " + headers_out.get(key);
 		printOut.print( hdr + "\r\n");
 		JServDebug.trace( hdr, RESPONSE_HEADERS );
	    }
	}

	// Send a terminating blank line
	printOut.print("\r\n");
	//Flush the PrintWriter
	printOut.flush();
    }


    //---------------------------------------- Implementation of ServletRequest
    /**
     * Returns the size of the request entity data, or -1 if not
     * known. Same as the CGI variable CONTENT_LENGTH.  
     */
    public int getContentLength() {
	String lenstr = (String) env_vars.get("CONTENT_LENGTH");
	if (lenstr == null)
	  return -1;

	try {
	    return Integer.parseInt( lenstr );
	} catch(NumberFormatException e) {
	    return -1;
	}
    }

    /**
     * Returns the Internet Media Type of the request entity data, 
     * or null if not known. Same as the CGI variable
     * CONTENT_TYPE. 
     */
    public String getContentType() {
	return (String) env_vars.get("CONTENT_TYPE");
    } 

    /**
     * Returns the protocol and version of the request as a string of
     * the form <code>&lt;protocol&gt;/&lt;major version&gt;.&lt;minor
     * version&gt</code>.  Same as the CGI variable SERVER_PROTOCOL.
     */
    public String getProtocol() {
	return (String) env_vars.get("SERVER_PROTOCOL");
    } 

    /**
     * Returns the scheme of the URL used in this request, for example
     * "http", "https", or "ftp".  Different schemes have different
     * rules for constructing URLs, as noted in RFC 1738.  The URL used
     * to create a request may be reconstructed using this scheme, the
     * server name and port, and additional information such as URIs.
     */
    public String getScheme() {
	// Stronghold and Ben Laurie's Apache-SSL set the "HTTPS" environment
 	// variable in secure mode.
	String val = (String) env_vars.get("HTTPS");
	if (val != null) {
	    if (val.equalsIgnoreCase("on"))
		return "https";
	    else if (val.equalsIgnoreCase("off"))
		return "http";
	}			
	// FIXME: We just don't have this information available. We'll
	// look at the port, and return https if it's 443, but that's
	// not a real solution.
	if (getServerPort() == 443)
	    return "https";

	return "http";
    }

    /**
     * Returns the host name of the server that received the request.
     * Same as the CGI variable SERVER_NAME.
     */
    public String getServerName() {
	return (String) env_vars.get("SERVER_NAME");
    } 

    /**
     * Returns the port number on which this request was received.
     * Same as the CGI variable SERVER_PORT.
     */
    public int getServerPort() {
	String portstr = (String) env_vars.get("SERVER_PORT");
	Integer port;

	if      (portstr == null)
	    return -1;

	try {
	    port = new Integer(portstr);
	}
	catch(NumberFormatException e) {
	    return -1;
	}

	return port.intValue();
    }

    /**
     * Returns the IP address of the agent that sent the request.
     * Same as the CGI variable REMOTE_ADDR.
     */
    public String getRemoteAddr() {
	return (String) env_vars.get("REMOTE_ADDR");
    }

    /**
     * Returns the fully qualified host name of the agent that sent the
     * request. Same as the CGI variable REMOTE_HOST.
     */
    public String getRemoteHost() {
	return (String) env_vars.get("REMOTE_HOST");
    } 
    
    /**
     * Applies alias rules to the specified virtual path and returns
     * the corresponding real path, or null if the translation can not
     * be performed for any reason.  For example, an HTTP servlet would
     * resolve the path using the virtual docroot, if virtual hosting
     * is enabled, and with the default docroot otherwise.  Calling
     * this method with the string "/" as an argument returns the
     * document root.
     * @param path The virtual path to be translated to a real path.
     */
    public String getRealPath(String path) {
	// FIXME: Make this somehow talk to Apache, do a subrequest
	// and get the real filename. Until then, we just tack the path onto
	// the doc root and hope it's right. *sigh*

	// DOCUMENT_ROOT is not a standard CGI var, although Apache always
	// gives it. So we allow for it to be not present.
	String doc_root = (String) env_vars.get("DOCUMENT_ROOT");

	if     (doc_root == null)
	    return null;

	return doc_root + path;
    } 

    /**
     * Returns an input stream for reading binary data in the request body.
     * @exception IOException other I/O related errors.
     */
    public ServletInputStream getInputStream() throws IOException {
	got_input = true;
	return servlet_in;
    }			
    
    
    // Parameter stuff
    private boolean parseParams() {
	// Have we already done it?
	if (params != null)
	    return false;

	try {
	    //Is this a post or a get
	    String method = getMethod();
	    if ( method.equals( "GET" ) ) {
		params = HttpUtils.parseQueryString(getQueryString());	    
	    } else if ( method.equals( "POST" ) ) {
		// Hmm... must have given the stream through getInputStream()
		if (got_input)
		    return true;
		
		got_input = true;
		params = HttpUtils.parsePostData(getContentLength(), 
						 servlet_in);
		return false;
	    } else {
	        //Unknown method
	        return true;
	    }
	} catch(IllegalArgumentException e) { 
	    return true;
	}
	return false;
    }

    /**
     * Returns a string containing the lone value of the specified
     * parameter, or null if the parameter does not exist. For example,
     * in an HTTP servlet this method would return the value of the
     * specified query string parameter. Servlet writers should use
     * this method only when they are sure that there is only one value
     * for the parameter.  If the parameter has (or could have)
     * multiple values, servlet writers should use
     * getParameterValues. If a multiple valued parameter name is
     * passed as an argument, the return value is implementation
     * dependent.
     * @param name the name of the parameter whose value is required.
     */
    public String getParameter(String name) {
	if (parseParams())
	    return null;

	Object val = params.get(name);

	if     (val == null)
	    return null;

	if     (val instanceof String[]) {
	    // It's an array, return the first element
	    return ((String[])val)[0];
	} return (String) val;
    }

    /**
     * added for the <servlet> tag support - RZ.
     * provides the ability to add to the request object
     * the parameter of an embedded servlet.
     */
    public void setParameter(String name, String value)
    {
	//add the parameter in the hashtable, overrides any previous value
	parseParams();
	if(params != null)
	    {
		params.put(name, value);
	    }
    }

    /**
     * Returns the values of the specified parameter for the request as
     * an array of strings, or null if the named parameter does not
     * exist. For example, in an HTTP servlet this method would return
     * the values of the specified query string or posted form as an
     * array of strings.
     * @param name the name of the parameter whose value is required.
     */
    public String[] getParameterValues(String name) {
	if (parseParams())
	    return null;

	Object val = params.get(name);

	if     (val == null)
	    return null;

	if     (val instanceof String) {
	    // It's a string, convert to an array and return
	    String va[] =
	    {(String) val};
	    return va;
	} return (String[]) val;
    }

    /**
     * Returns the parameter names for this request as an enumeration
     * of strings, or an empty enumeration if there are no parameters
     * or the input stream is empty.  The input stream would be empty
     * if all the data had been read from the stream returned by the
     * method getInputStream.
     */
    public Enumeration getParameterNames() {
	// an "empty Enumeration"... *sigh*
	if (parseParams())
	    return (new Vector()).elements();
	return params.keys();
    } 
    
    /**
     * Returns the value of the named attribute of the request, or
     * null if the attribute does not exist.  This method allows
     * access to request information not already provided by the other
     * methods in this interface.  Attribute names should follow the
     * same convention as package names. 
     * The following predefined attributes are provided.
     *
     * <TABLE BORDER>
     * <tr>
     *        <th>Attribute Name</th>
     *        <th>Attribute Type</th>
     *        <th>Description</th>
     *        </tr>
     *
     * <tr>
     *        <td VALIGN=TOP>javax.net.ssl.cipher_suite</td>
     *        <td VALIGN=TOP>string</td>
     *        <td>The string name of the SSL cipher suite in use, if the
     *                request was made using SSL</td>
     *        </tr>
     *
     * <tr>
     *        <td VALIGN=TOP>javax.net.ssl.peer_certificates</td>
     *        <td VALIGN=TOP>array of java.security.cert.X509Certificate</td>
     *        <td>The chain of X.509 certificates which authenticates the client.
     *                This is only available when SSL is used with client
     *                authentication is used.</td>
     *        </tr>
     *
     * <tr>
     *        <td VALIGN=TOP>javax.net.ssl.session</td>
     *        <td VALIGN=TOP>javax.net.ssl.SSLSession</td>
     *        <td>An SSL session object, if the request was made using SSL.</td>
     *        </tr>
     *
     * </TABLE>
     *
     * <BR>
     * <P>The package (and hence attribute) names beginning with java.*,
     * and javax.* are reserved for use by Javasoft. Similarly, com.sun.*
     * is reserved for use by Sun Microsystems.
     * 
     * <p><b>Note</b> The above attributes are not yet implemented by
     * JServ.
     * <p>On the other hand, attribute named "apache.jserv.&lt;variable&gt;"
     * returns the content of the environment (CGI) variable 
     * "&lt;variable&gt;".
     */
    public Object getAttribute(String name) {
	// We return "apache.jserv.<variable>" as the contents
	// of environment (CGI) variable "<variable>"
	if (!name.startsWith("apache.jserv."))
	    return null;

	return env_vars.get(name.substring("apache.jserv.".length()));
    }

    //------------------------------------ Implementation of HttpServletRequest
    /**
     * Gets the HTTP method (for example, GET, POST, PUT) with which
     * this request was made. Same as the CGI variable REQUEST_METHOD.
     * @return the HTTP method with which this request was made.
     */
    public String getMethod() {
	return (String) env_vars.get("REQUEST_METHOD");
    }

    /**
     * Gets this request's URI as a URL.
     * @return this request's URI as a URL.
     */
    public String getRequestURI() {
	if (getPathInfo() != null)
	    return getServletPath() + getPathInfo();
	else
	    return getServletPath();
    }

    /**
     * Gets the part of this request's URI that refers to the servlet
     * being invoked. Analogous to the CGI variable SCRIPT_NAME.
     * @return the servlet being invoked, as contained in this
     * request's URI.
     */
    public String getServletPath() {
	return (String) env_vars.get("SCRIPT_NAME");
    }

    /**
     * Gets any optional extra path information following the servlet
     * path of this request's URI, but immediately preceding its query
     * string. Same as the CGI variable PATH_INFO.
     *
     * @return the optional path information following the servlet
     * path, but before the query string, in this request's URI; null
     * if this request's URI contains no extra path information.
     */
    public String getPathInfo() {
	return (String) env_vars.get("PATH_INFO");
    }

    /**
     * Gets any optional extra path information following the servlet
     * path of this request's URI, but immediately preceding its query
     * string, and translates it to a real path.  Same as the CGI
     * variable PATH_TRANSLATED.
     *
     * @return extra path information translated to a real path or null
     * if no extra path information is in the request's URI.
     */
    public String getPathTranslated() {
	return (String) env_vars.get("PATH_TRANSLATED");
    }

    /**
     * Gets any query string that is part of the servlet URI.  Same as
     * the CGI variable QUERY_STRING.
     * @return query string that is part of this request's URI, or null
     * if it contains no query string.
     */
    public String getQueryString() {
	return (String) env_vars.get("QUERY_STRING");
    }

    /**
     * Gets the name of the user making this request.  The user name is
     * set with HTTP authentication.  Whether the user name will
     * continue to be sent with each subsequent communication is
     * browser-dependent.  Same as the CGI variable REMOTE_USER.
     *
     * @return the name of the user making this request, or null if not
     * known.
     */
    public String getRemoteUser() {
	return (String) env_vars.get("REMOTE_USER");
    }

    /**
     * Gets the authentication scheme of this request.  Same as the CGI
     * variable AUTH_TYPE.
     * 
     * @return this request's authentication scheme, or null if none.
     */
    public String getAuthType() {
	return (String) env_vars.get("AUTH_TYPE");
    }

    /**
     * Gets the value of the requested header field of this request.
     * The case of the header field name is ignored.
     * @param name the String containing the name of the requested
     * header field.
     * @return the value of the requested header field, or null if not
     * known.
     */
    public String getHeader(String name) {
	return (String) headers_in.get(name.toLowerCase());
    }

    /**
     * Gets the value of the specified integer header field of this
     * request.  The case of the header field name is ignored.  If the
     * header can't be converted to an integer, the method throws a
     * NumberFormatException.
     * @param name  the String containing the name of the requested
     * header field.
     * @return the value of the requested header field, or -1 if not
     * found.
     */
    public int getIntHeader(String name) {
	String hdrstr = (String) headers_in.get(name.toLowerCase());
	if (hdrstr == null)
	    return -1;
	
	return Integer.parseInt( hdrstr );
    }

    /**
     * Gets the value of the requested date header field of this
     * request.  If the header can't be converted to a date, the method
     * throws an IllegalArgumentException.  The case of the header
     * field name is ignored.
     *
     * <PRE>  From RFC2068:
     *  3.3.1 Full Date
     *
     *
     *   HTTP applications have historically allowed three different formats
     *   for the representation of date/time stamps:
     *
     *    Sun, 06 Nov 1994 08:49:37 GMT  ; RFC 822, updated by RFC 1123
     *    Sunday, 06-Nov-94 08:49:37 GMT ; RFC 850, obsoleted by RFC 1036
     *    Sun Nov  6 08:49:37 1994       ; ANSI C's asctime() format
     *
     *   The first format is preferred as an Internet standard and
     *   represents a fixed-length subset of that defined by RFC 1123
     *   (an update to RFC 822).  The second format is in common use,
     *   but is based on the obsolete RFC 850 [12] date format and
     *   lacks a four-digit year.  HTTP/1.1 clients and servers that
     *   parse the date value MUST accept all three formats (for
     *   compatibility with HTTP/1.0), though they MUST only generate
     *   the RFC 1123 format for representing HTTP-date values in
     *   header fields
     * <pre>
     * @param name  the String containing the name of the requested
     * header field.
     * @return the value the requested date header field, or -1 if not
     * found.
     */
    public long getDateHeader(String name) {
	String val = (String) headers_in.get(name.toLowerCase());

	if ( val == null )
	    return -1;

	SimpleDateFormat sdf =
	    new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
	try {
	    Date date = sdf.parse(val);
	    return date.getTime();
	}
	catch(ParseException e) {
	    ;			// Try another format

	}

	sdf = new SimpleDateFormat("EEEEEE, dd-MMM-yy HH:mm:ss zzz");
	try {
	    Date date = sdf.parse(val);
	    return date.getTime();
	}
	catch(ParseException e) {
	    // Try another format
	}

	sdf = new SimpleDateFormat("EEE MMMM d HH:mm:ss yyyy");
	try {
	    Date date = sdf.parse(val);
	    return date.getTime();
	} catch(ParseException e) {
	    throw new IllegalArgumentException( val );
	}
    }

    /**
     * Gets the header names for this request.
     * @return an enumeration of strings representing the header names
     * for this request. Some server implementations do not allow
     * headers to be accessed in this way, in which case this method
     * will return null.
     */
    public Enumeration getHeaderNames() {
	return headers_in.keys();
    }


    //--------------------------------------- Implementation of ServletResponse
    /**
     * Sets the content length for this response.
     * @param len the content length.
     */
    public void setContentLength(int len) {
	Integer length = new Integer(len);

	headers_out.put("Content-Length", length.toString());
    }

    /**
     * Sets the content type for this response.  This type may later
     * be implicitly modified by addition of properties such as the MIME
     * <em>charset=&lt;value&gt;</em> if the service finds it necessary,
     * and the appropriate media type property has not been set.  This
     * response property may only be assigned one time.
     * @param type the content's MIME type
     */
    public void setContentType(String type) {
	headers_out.put("Content-Type", type);
    }

    /**
     * Returns an output stream for writing binary response data.
     * @exception IOException if an I/O exception has occurred.
     */
    public ServletOutputStream getOutputStream() throws IOException {
	return servlet_out;
    }

    //--------------------------------- Implementation of HttpServletResponse
    /**
     * Checks whether the response message header has a field with
     * the specified name.
     * @param name the header field name.
     * @return true if the response message header has a field with
     * the specified name; false otherwise.
     */
    public boolean containsHeader(String name) {
	return headers_out.contains(name.toLowerCase());
    }

    /**
     * Sets the status code and message for this response.  If the
     * field had already been set, the new value overwrites the
     * previous one.  The message is sent as the body of an HTML
     * page, which is returned to the user to describe the problem.
     * The page is sent with a default HTML header; the message
     * is enclosed in simple body tags (&lt;body&gt;&lt;/body&gt;).
     * @param sc the status code.
     * @param sm the status message.
     */
    public void setStatus(int sc, String sm) {
	status = sc;
	status_string = sm;
    }

    /**
     * 
     * Sets the status code for this response.  This method is used to
     * set the return status code when there is no error (for example,
     * for the status codes SC_OK or SC_MOVED_TEMPORARILY).  If there
     * is an error, the <code>sendError</code> method should be used
     * instead.
     * @param sc the status code
     * @see sendError
     */
    public void setStatus(int sc) {
	setStatus(sc, null);
    }

    /**
     * Adds a field to the response header with the given name and value.
     * If the field had already been set, the new value overwrites the
     * previous one.  The <code>containsHeader</code> method can be
     * used to test for the presence of a header before setting its
     * value.
     * @param name the name of the header field
     * @param value the header field's value
     * @see containsHeader
     */
    public void setHeader(String name, String value) {
        int offset_of_newline;
        // We need to make sure no newlines are present in the header:
        if ((offset_of_newline = value.indexOf((int)'\n')) > 0) {
            char msgAsArray[] = value.toCharArray();
            msgAsArray[offset_of_newline] = ' ';
            while ((offset_of_newline =
                    value.indexOf((int)'\n',offset_of_newline+1)) > 0) {
                 msgAsArray[offset_of_newline] = ' ';
            }
            value = new String(msgAsArray);
        }
	headers_out.put(name, value);
    }

    /**
     * Adds a field to the response header with the given name and
     * integer value.  If the field had already been set, the new value
     * overwrites the previous one.  The <code>containsHeader</code>
     * method can be used to test for the presence of a header before
     * setting its value.
     * @param name the name of the header field
     * @param value the header field's integer value
     * @see containsHeader
     */
    public void setIntHeader(String name, int value) {
	Integer val = new Integer(value);

	headers_out.put(name, val.toString());
    }

    /**
     * Adds a field to the response header with the given name and
     * date-valued field.  The date is specified in terms of
     * milliseconds since the epoch.  If the date field had already
     * been set, the new value overwrites the previous one.  The
     * <code>containsHeader</code> method can be used to test for the
     * presence of a header before setting its value.
     * @param name the name of the header field
     * @param value the header field's date value
     * @see containsHeader
     */
    public void setDateHeader(String name, long date) {
	SimpleDateFormat sdf =
	    new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
	TimeZone tz = TimeZone.getTimeZone("GMT");

	sdf.setTimeZone(tz);

	headers_out.put(name, sdf.format(new Date(date)));
    }

    /**
     * Sends an error response to the client using the specified status
     * code and descriptive message.  If setStatus has previously been
     * called, it is reset to the error status code.  The message is
     * sent as the body of an HTML page, which is returned to the user
     * to describe the problem.  The page is sent with a default HTML
     * header; the message is enclosed in simple body tags
     * (&lt;body&gt;&lt;/body&gt;).
     * @param sc the status code
     * @param msg the detail message
     */
    public void sendError(int sc, String msg) {
	try {
	    // Tell Apache to send an error
	    status = sc;
	    setHeader("Servlet-Error", msg);
	    sendHttpHeaders();

	    // Flush and close, so the error can be returned right
	    // away, and so any additional data sent is ignored
	    out.flush();
	    client.close();
	} catch (IOException e) {
	    // Not much more we can do...
	    JServDebug.trace( e );
	}
    }

    /**
     * Sends an error response to the client using the specified
     * status code and a default message.
     * @param sc the status code
     */
    public void sendError(int sc) {
	sendError(sc, findStatusString( sc ) );
    }

    /**
     * JServSendError method. This sends an error message to Apache
     * when an exception occur in the ServletEngine. 
     */
    public void sendError(Throwable e) {
	sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
		  e.toString() + ": " + e.getMessage());
	JServDebug.trace( e );
    }

    /**
     * Sends a temporary redirect response to the client using the
     * specified redirect location URL.  The URL must be absolute (for
     * example, <code><em>https://hostname/path/file.html</em></code>).
     * Relative URLs are not permitted here.
     * @param location the redirect location URL
     * @exception IOException If an I/O error has occurred.
     */
    public void sendRedirect(String location) throws IOException {
	// We use Apache's internal status mechanism: set the status to
	// 200, with a Location: header and no body.

	setStatus(SC_OK);
	setHeader("Location", location);
	sendHttpHeaders();
    }

    //Stream are inner classes
    // ServletInputStream implementation
    private class JServInputStream extends ServletInputStream {

	//Default construcot
	public JServInputStream() {}

	// methods from InputStream
	public int read() throws IOException {
	    return in.read();
	}

	public int read(byte b[]) throws IOException {
	    return read(b, 0, b.length);
	}

	public int read(byte b[], int off, int len) throws IOException {
	    return in.read(b, off, len);
	}

	public long skip(long n) throws IOException {
	    return in.skip(n);
	}
  
	public void close() throws IOException {
	    //Ignore closing of the input stream since it also
	    //close the output stream.
	    //conn.in.close();
	}
    }

    // ServletOutputStream implementation
    class JServOutputStream extends ServletOutputStream {
    
	//Default constructor
	public JServOutputStream() {}
  
	// methods from OutputStream
	public void write(int b) throws IOException {
	    sendHttpHeaders();
	    out.write(b);
	}

	public void write(byte b[], int off, int len) throws IOException {
	    sendHttpHeaders();
	    out.write( b, off, len);
	}

	public void flush() throws IOException {
	    sendHttpHeaders();
	    out.flush();
	}

	public void close() throws IOException {
	    sendHttpHeaders();
	    out.close();
	}
    }
}

