// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// JServHandler.java:
// - org.apache.jserv.JServDebug.java

package org.apache.jserv;

import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * Class that implements the debugging policy for the JServ module.
 *
 * @author Francis J. Lacoste
 * @version $Revision: 1.1 $
 */
public final class JServDebug implements JServDebug.DebugConstants {
    /**
     * Set this constants to false to have an optimized version of
     * JServ that doesn't do tracing.
     */
    private static final boolean TRACING_COMPILED_IN = true;

    /**
     * Interface describing the different logging modules.
     */
    public static interface DebugConstants {
	/**
	 * Constant used to turn on tracing of everything.
	 */
	public static final int ALL = Integer.MAX_VALUE;

	/**
	 * Constant used to turn off tracing altogether.
	 */
	public static final int NONE = 0;

	/**
	 * Constant used when a message should always be logged.
	 */
	public static final int ALWAYS = 0;

	/**
	 * Constant used for tracing the initialization of the
	 * JServ module.
	 */ 
	public static final int INIT = 1;

	/**
	 * Constant used to trace the request servicing.
	 */
	public static final int SERVICE_REQUEST = 2;

	/**
	 * Constant used to trace the request data reading.
	 */
	public static final int REQUEST_DATA = 4;

	/**
	 * Constant used to trace the response headers sent to the client.
	 */
	public static final int RESPONSE_HEADERS = 8;

	/**
	 * Constant to trace signal handling.
	 */
	public static final int SIGNAL = 16;

	/**
	 * Constant reserved for future use.
	 */
         /*private static final int RESERVED_FOR_FUTURE_USE = 32;*/

	/**
	 * Constant used when outputting stack trace.
	 */
	public static final int EXCEPTION_TRACING = 64;

	/**
	 * Constant used for tracing the ServletManager.
	 */
	public static final int SERVLET_MANAGER = 128;

	/**
	 * Constant used for tracing class loading by the class loader.
	 */
	public static final int CLASSLOADER = 256;

	/**
	 * Constant used for tracing class loading by the class loader.
	 */
	public static final int CLASSLOADER_VERBOSE = 512;

    }
    
    /**
     * What is the current debugging policy.
     */
    private static int debugFlags = 
    Integer.getInteger( "jserv.tracing", NONE ).intValue();

    /**
     * The timestamp formatter.
     */
    private static SimpleDateFormat formatter = 
    new SimpleDateFormat("[dd/MM/yyyy HH:mm:ss zz]: ");

    /**
     * Log a message to stderr, if the tracing is on for the following module.
     * @param msg The message to log.
     * @param module The module in which this message should be log.
     */
    public static void trace( String msg, int module ) {
	if ( TRACING_COMPILED_IN ) {
	    if ( ( debugFlags & module ) == module ) {
		System.err.println( formatter.format( new Date() ) + msg );
	    }
	}
    }

    /**
     * Trace an exception if exception tracing is turn on.
     */
    public static void trace( Throwable t ) {
	System.err.println( formatter.format( new Date() ) + 
			    t.toString() + ": " + t.getMessage() );
	if ( ( debugFlags & EXCEPTION_TRACING ) == EXCEPTION_TRACING ) {
	    t.printStackTrace();
	}
    }

    /**
     * Set the tracing global policy. This should the addition of
     * all the tracing constants that should be turn on.
     */
    public static void setTracing( int policy ) {
	debugFlags = policy;
    }

    /**
     * Set the tracing for a module.
     */
    public static void setTracing( int module, boolean state ) {
	if ( state ) {
	    debugFlags |= module;
	} else {
	    debugFlags ^= module;
	}
    }

}
