// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

// JServHandler.java:
// - org.apache.jserv.JServHandler

package org.apache.jserv;

import java.util.Hashtable;
import java.util.Properties;
import java.util.Vector;
import java.util.Enumeration;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.File;
import java.io.IOException;

import java.net.Socket;
import java.net.ServerSocket;


/**
 * <CODE>JServHandler</CODE> is the entry point to the Java part of
 * mod_jserv.
 *
 * It sets up the server, initalizes everything, and listens on a TCP
 * port for requests for the server. When it gets a request, it
 * launches a JServConnection thread.
 *
 * @author Alexei Kosut
 **/
public class JServHandler extends Thread 
implements JServSendError, JServDebug.DebugConstants 
{
    public final static String version = "JServ/0.9.8";

    public final static int DEFAULT_PORT = 8007;
    protected static int port;
    protected static JServSignals signal_handler;

    protected ServerSocket listen_socket;

    /**
     * Table that contains one servlet manager per host that 
     * serves servlet.  */
    private static Hashtable servletMgrTable = new Hashtable();

    protected static String auth = null;
    protected static boolean localhostcheck = true;

    /**
     * The Reader that wraps around System.in. This pattern is used because
     * <ol>
     * <li>We need readLine() from BufferedReader to parse the configuration
     * and that the one in DataInputStream is deprecated.
     * <li>Even when setting the buffer size to 1 in the BufferedReader it
     * sometimes happen that we lose the signal data sent after the 
     * configuration.
     * </ol>
     */
    static BufferedReader in = new BufferedReader( new InputStreamReader( System.in ) );

    // Start the server up, listening on the specified port
    public static void main(String[]args) {
	String auth = null;
	boolean noauth = false;
	int argn, port = 0;
	File configFile = null;
 	JServDebug.trace( "Initializing JServ", INIT );
	try {
	    for (argn = 0; argn < args.length; argn++) {
		if (args[argn].equals("-m")) {
		    noauth = true;
		    argn++;	//Next option is the configuration file.

		    configFile = new File(args[argn]);
		} else if (args[argn].equals("-t")) {
 		    JServDebug.setTracing( EXCEPTION_TRACING, true );
		} else {
		    break;
		}
	    }

	    try {
		port = Integer.parseInt(args[argn++]);
	    } catch(NumberFormatException e) {
		throw new ArrayIndexOutOfBoundsException();
	    }
	} catch(ArrayIndexOutOfBoundsException e) {
	    System.err.println("JServHandler: illegal arguments");
	    System.err.println("usage: java apache.jserv.JServHandler" +
			       " [-m configFile] [-t] port");
	    System.exit(1);
	}

	if (!noauth) {
	    // Read the auth from the first line of "stdin"
	    try {
		// Read until newline
		byte a[] = new byte[25];
		int i;

		for (i = 0; i < 25; i++) {
		    int b = in.read();

		    if (b == -1 || b == '\n')
			break;

		    a[i] = (byte) b;
		}

		auth = new String(a, 0, i);
	    } catch(IOException e) {
		fail(e, "Exception while reading authentication");
	    }
	}
	//Read configuration via stdin or property file
	if (noauth) {
	    readConfigByFile(configFile);
	} else {
	    readConfigByStdin();
	}
	//Read the server configuration.
	new JServHandler(port, auth);
    }

    /**
     * Reads in the host configuration via Stdin - This occurs when
     * Apache starts the server.
     */
    private static void readConfigByStdin() {
	try {
	    while (true) {
		String configLine = in.readLine();
		//Check for end of transmission
		if (configLine.equals("end")) {
		    break;
                }		
		//Parse host information
		StringTokenizer stok = new StringTokenizer(configLine, "\t");
		try {
		    JServDebug.trace( "Creating ServletManager", INIT );
		    String hostname = stok.nextToken();
		    JServDebug.trace( "    Host: " + hostname, INIT );
		    String propName = stok.nextToken();
		    JServDebug.trace( "    Property file: " + propName, INIT );

		    //We need at least one servlet repository
		    Vector servletRep = new Vector();
		    String rep = stok.nextToken();
		    JServDebug.trace( "    Servlet Repository: " + rep, INIT);
		    servletRep.addElement( rep );
		    while (stok.hasMoreElements()) {
			//Add additional
			rep = stok.nextToken();
			servletRep.addElement(stok.nextToken());
			JServDebug.trace( "    Servlet Repository: " + rep,
					  INIT);
		    }

		    JServServletManager mgr = new
			JServServletManager(hostname, propName, servletRep);
		    servletMgrTable.put(hostname, mgr);
		} catch(NoSuchElementException badformat) {
		    fail(badformat, "Format of host configuration is " +
			 "hostname propertyFile (servlet_alias)+");
		}
	    }
	} catch(IOException ioe) {
	    fail(ioe, "Exception while reading host configuration.");
	}
    }

    /**
     * Read the host configuration via a property file.
     * This occurs in manual mode.
     */
    private static void readConfigByFile(File file) {
	Properties prop = new Properties();
	try {
	    prop.load(new BufferedInputStream(new FileInputStream(file)));
	} catch(IOException ioe) {
	    fail(ioe, "IOException while reading configuration");
	    return;
	}
	String host = prop.getProperty("jserv.hosts");
	if (host == null) {
	    fail(new IllegalArgumentException(), "No hosts in configuration file");
	}
	StringTokenizer hostStok = new StringTokenizer(host, ":");
	while (hostStok.hasMoreElements()) {
	    JServDebug.trace( "Creating ServletManager", INIT );
	    //Get properties file for that host.
	    String hostname = hostStok.nextToken();
	    JServDebug.trace( "    Host: " + hostname, INIT );
	    String propFile = prop.getProperty("jserv." + hostname + ".properties");
	    if (propFile == null) {
		fail(new IllegalArgumentException(),
		     "No property file for host " + hostname);
	    }
	    JServDebug.trace( "    Property file: " + propFile, INIT );

	    //Load servlet repository for that host
	    Vector servletRep = new Vector();
	    String repStr = prop.getProperty("jserv." + hostname + ".repository");
	    if (repStr == null) {
		fail(new IllegalArgumentException(),
		     "No servlet repository for host " + hostname);
	    }
	    StringTokenizer repStok = new StringTokenizer(repStr, ":");
	    while (repStok.hasMoreElements()) {
		String rep = repStok.nextToken();
		servletRep.addElement( rep );
		JServDebug.trace( "    Servlet Repository: " + rep, INIT );
	    }

	    //Create servlet manager for this host
	    JServServletManager mgr = new
		JServServletManager(hostname, propFile, servletRep);
	    servletMgrTable.put(hostname, mgr);
	}
        if ("false".equals(prop.getProperty("jserv.security.localhostonly")))
            JServHandler.localhostcheck = false;
        else
            JServHandler.localhostcheck = true;
    }

    /** 
     * Check whether an authorization string is correct.
     *
     * @return true if authorization cookies are not in use or if
     * the string is correct; otherwise false.
     **/
    static boolean checkAuthString(String toCheck) {
	if (auth == null  ||  auth.equals(toCheck))
	    return true;
	else
	    return false;
    }

    // Exit with an error message, when an exception occurs.
    public static void fail(Exception e, String msg) {
	System.err.println(msg + ": " + e);
	System.exit(1);
    }				
    //Implementation of the SendHandler interface

    /** Report a problem encountered while initializing. **/
    public void sendError(int sc, String msg) {
	JServDebug.trace( JServConnection.findStatusString( sc ) + ": " + msg,
			  ALWAYS );
    }

    /** Report an exception or error encountered while loading a servlet. **/
    public void sendError(Throwable e) {
	JServDebug.trace( e );
    }

    // Create a ServerSocket to listen for connections on;  start the thread.
    public JServHandler(int port, String auth) {
	// Auth string
	this.auth = auth;

	if (port == 0)
	    port = DEFAULT_PORT;
	this.port = port;

	// Wait for signals
	this.signal_handler = new JServSignals(servletMgrTable);

	try {
	    listen_socket = new ServerSocket(port);
	} catch(IOException e) {
	    fail(e, "Exception creating server socket");
	}

	this.start();

	//Load all startup servlets in another Thread.
	Runnable startupLauncher = new Runnable()
	    {
		public void run()
		    {
		 	Enumeration mgrEnum = 
			    JServHandler.this.servletMgrTable.elements();
			while (mgrEnum.hasMoreElements()) {
			    JServServletManager mgr = 
				(JServServletManager) mgrEnum.nextElement();
			    JServDebug.trace( "Initializing " + mgr.getName(),
					      INIT );
			    mgr.init(JServHandler.this);
			}
		    }
	    };
	new Thread(startupLauncher, "Startup Servlet Launcher").start();
    }
  

    /**
     * The body of the server thread.  Loop forever, listening for and
     * accepting connections from clients.  For each connection, 
     * create a JServConnection object to handle communication through the
     * new Socket.
     */
    public void run() {
        JServDebug.trace( "Listening for connections on port " + port, 
			  INIT );
        try {
	    while (true) {
		Socket client_socket = listen_socket.accept();
		JServDebug.trace( "Connection from " + 
				  client_socket.getInetAddress(),
				  SERVICE_REQUEST );
		JServConnection c = 
		    new JServConnection(client_socket, servletMgrTable);
	    }
	} catch(IOException e) {
	    fail(e, "Exception while listening for connections");
	}
    }
    
}

