// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// JServLock.java:
// - org.apache.jserv.JServLock

package org.apache.jserv;

import javax.servlet. *;
import javax.servlet.http. *;

/**
 *
 * This is a reader/writer lock object. The code is vaugely based on
 * an example from <i>_Concurrent Programming in Java_</i>, by Doug Lea.
 *
 * <p>As implemented, JServLock works as an actual "lock" object, which
 * another object will set up and lock and unlock. It blocks incoming
 * readers if there are waiting writers.
 *
 * <p>There are method that wait only at specified amount of time before
 * failing with TimeoutException.
 */
public class JServLock {
    private int activeReadLocks = 0;
    private int waitingReadLocks = 0;

    private int activeWriteLocks = 0;
    private int waitingWriteLocks = 0;

    /**
     * Exception thrown when the timeout specified is reached without
     * suceeding in obtaining the lock.
     */
    public static class TimeoutException extends Exception {
	
	/**
	 * TimeoutException constructor.
	 */
	public TimeoutException() { }
    }

    // We only allow a reader to lock if there are no write locks,
    // and no waiting writer locks.
    private boolean allowReadLock() {
	return activeWriteLocks == 0 && waitingWriteLocks == 0;
    }				
    
    // We only allow a writer to lock if there are no active
    // locks
     private boolean allowWriteLock() {
	return activeReadLocks == 0 && activeWriteLocks == 0;
    } 

    /**
     * Wait for a read lock. This will wait for all write lock to
     * be removed before returning.
     *
     * @exception InterruptedException if the wait is interrupted. Calling
     * thread must consider the operation to have failed and stop its 
     * processing.
     */
    public synchronized void readLock() throws InterruptedException {
	// Register our intent
	waitingReadLocks++;

	try {
	    // Wait for a chance to read
	    if ( allowReadLock() == false ) {
		wait();
	    }			
	    // Bingo! Switch from a waiting lock to an active lock
	    activeReadLocks++;
	} finally {
	    //Remove intent to read
	    waitingReadLocks--;
	}
    }

    /**
     * Wait for a read lock. This will wait for all write lock to
     * be removed before returning.
     *
     * @param timeout the number of millisecond before giving up and failing 
     * with a TimeoutException.

     * @exception TimeoutException if the lock isn't acquired after the
     * specified amount of time.
     * @exception InterruptedException if the wait is interrupted. Calling
     * thread must consider the operation to have failed and stops its 
     * processing.
     */
    public synchronized void readLock( long timeout ) 
	 throws InterruptedException, TimeoutException
    {
	// Register our intent
	waitingReadLocks++;
	try {
	    // Wait for a chance to read
	    if ( allowReadLock() == false ) {
		wait( timeout );
		//Check to see if was have the lock
		if ( allowReadLock() == false ) {
		    //Timeout without obtaining lock.
		    throw new TimeoutException();
		}
	    }			
	    // Bingo! Switch from a waiting lock to an active lock
	    activeReadLocks++;
	} finally {
	    //Remove intent to read.
	    waitingReadLocks--;
	}
    }

    /**
     * Unlocks a previously acquired read lock.
     */
    public synchronized void readUnlock() {
	// We're gone
	activeReadLocks--;
	// Wake other threads up
	notifyAll();
    } 

    /**
     * Wait for a read lock. This will wait until all read lock have been
     * removed and no other write lock are active.
     *
     * @exception InterruptedException if the wait is interrupted. Calling
     * thread must consider the operation to have failed and stops its 
     * processing.
     */
    public synchronized void writeLock() throws InterruptedException {
	// Register our intent
	waitingWriteLocks++;

	try {
	    // Wait for a chance to write
	    if ( allowWriteLock() == false ) {
		wait();
	    }

	    // Bingo! Switch from a waiting lock to an active lock
	    waitingWriteLocks--;
	} finally {
	    //Remove intent lock
	    activeWriteLocks++;
	}
    }
    
    /**
     * Wait for a read lock. This will wait until all read lock have been
     * removed and no other write lock are active.
     *
     * @param timeout the number of millisecond before giving up and failing 
     * with a TimeoutException.

     * @exception TimeoutException if the lock isn't acquired after the
     * specified amount of time.
     * @exception InterruptedException if the wait is interrupted. Calling
     * thread must consider the operation to have failed and stops its 
     * processing.
     */
    public synchronized void writeLock( long timeout ) 
	 throws InterruptedException, TimeoutException
    {
	// Register our intent
	waitingWriteLocks++;
	
	try {
	    // Wait for a chance to write
	    if ( allowWriteLock() == false ) {
		wait( timeout );
		if ( allowWriteLock() == false ) {
		    //Timeout, we failed to acquire the lock
		    throw new TimeoutException();
		}
	    }
	    // Bingo! Switch from a waiting lock to an active lock
	    activeWriteLocks++;
	} finally {
	    waitingWriteLocks--;
	}
    }

    /**
     * Unlock a previously acquired write lock.
     */
    public synchronized void writeUnlock() {
	// We're gone
	activeWriteLocks--;
	// Wake other threads up
	notifyAll();
    }
}
