// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================

// TODO:
// (from Tim Williams, williams@ugsolutions.com:)
// - Fix problem with PARAM settings accumulating with each next servlet
// - Allow unused servlet tag attributes act as init parameters of the
//   named servlet, e.g.: <SERVLET xyz=123> 
// - Support "CODEBASE" servlet tag, allowing remote loading of servlets
// - Make "NAME" parameter work for symbolic names, not just "CODE"

package org.apache.jserv;

import java.io.*;
import java.util.*;
import javax.servlet.*;
import javax.servlet.http.*;


/** 
 * <CODE>JServSSI</CODE> provides support for including dynamic servlet
 * output from within HTML documents via the <CODE>&lt;SERVLET&gt;</CODE> tag.
 * 
 * <h3>Configuring the Apache Server</h3>
 * <blockquote>
 *
 * <p>In order to take advantage of the capabilities provided by this
 * class, you must first configure your Apache web server so that it
 * knows how to pass documents which will contain
 * <CODE>&lt;SERVLET&gt;</CODE> tags to this class.
 *
 * <p>Typically, this is done by naming files with <CODE>SERVLET</CODE> 
 * tags with a ".jhtml" extension, and adding the following lines to 
 * your Apache server configuration file:
 *
 *   <code><pre>
 *   # enable &lt;servlet&gt; tag in .jhtml files
 *   AddHandler jhtml-parser .jhtml
 *   Action jhtml-parser /<i>servletDirectory</i>/org.apache.jserv.JServSSI
 *   </pre></code>
 *
 * where <i>servletDirectory</i> is the URI from which your servlets
 * are accessed, as designated by the "<code>ServletAlias</code>" directive.
 * (This is often "<code>/servlets</code>".)
 *
 * </blockquote>
 * <h3>Including Servlets in HTML Pages</h3>
 * <blockquote>
 *
 * <h4>The <CODE>&lt;SERVLET&gt;</CODE> 
 * 	and <CODE>&lt;/SERVLET&gt;</CODE> tags</h4>
 *
 * <p>Servlet output may be included in an HTML document by use of
 * the <CODE>&lt;SERVLET&gt;</CODE> tag.  For example, to embed the
 * output of the demo "HelloWorldServlet" servlet in an HTML page, you 
 * might write the following:
 *
 *   <code><pre>
 *   <i>... (some HTML code) ...</i>
 *   &lt;SERVLET CODE="HelloWorldServlet.class"&gt;
 *   Your web server has not been configured to support servlet tags.
 *   &lt;/SERVLET&gt;
 *   <i>... (more HTML code) ...</i>
 *   </pre></code>
 *
 * <p>When this page is served from the web server, the code above
 * will be replaced with the output of HelloWorldServlet.  
 * If you see the message between the tags instead, there is a problem 
 * with your server configuration.  If this happens, check to make sure 
 * your file has a ".jhtml" extension and that the Apache server is 
 * configured as described above.
 *
 * <p>Two attributes are used by the <CODE>SERVLET</CODE> tag, 
 * and they are roughly equivalent:
 * <ol>
 * <p><li>The <CODE>CODE</CODE> attribute may be set to the name of a class file
 *     (the ".class" extension is optional) containing the servlet
 *     to be run.  Currently, this servlet must be installed in the
 *     same directory as the other servlets, not in the directory
 *     where your HTML resides.
 * <p><li>The <CODE>NAME</CODE> attribute may also be set to the name of the
 *     servlet to be run (with no class extension).
 * </ol>
 * <p>In some implementations of <CODE>SERVLET</CODE> tags, if 
 * both <CODE>NAME</CODE> and <CODE>CODE</CODE> attributes are set, the 
 * servlet designated by <CODE>CODE</CODE> will then become available for 
 * future use under the symbolic named designated by the <CODE>NAME</CODE> 
 * attribute.  This is not currently supported.
 *
 * <p>Note that both the <CODE>&lt;SERVLET&gt;</CODE> and 
 * <CODE>&lt;/SERVLET&gt;</CODE> tags must be present.
 *
 * <h4>The <CODE>&lt;PARAM&gt;</CODE> tag</h4>
 *
 * <p>You may send parameters to a servlet via the <CODE>PARAM</CODE> tag, which
 * should be placed between the <CODE>&lt;SERVLET ... &gt;</CODE> and
 * <CODE>&lt;/SERVLET&gt;</CODE> tags, like so:
 *
 *   <code><pre>
 *   &lt;SERVLET CODE="MyServlet.class"&gt;
 *   &lt;PARAM NAME="param1" VALUE="valueOfParam1"&gt;
 *   &lt;PARAM NAME="anotherParam" VALUE="valueOfAnotherParam"&gt;
 *   &lt;/SERVLET&gt;
 *   </pre></code>
 *
 * <p>You could then access these parameters from your
 * servlet as follows:
 *
 *   <code><pre>
 *   public void doGet(HttpServletRequest req, HttpServletResponse res)
 *       throws ServletException, IOException
 *   {
 *       String param1 = req.getParameter("param1");
 *       String anotherParam = req.getParameter("anotherParam");
 *   }
 *   </pre></code>
 *
 * </blockquote>
 * <h3>Notes</h3>
 * <blockquote>
 *
 * <dl>
 * <p><dt><b>Attribute values</b> 
 * <dd>Attribute values, <CODE>NAME</CODE> and <CODE>VALUE</CODE> 
 * in the <CODE>PARAM</CODE> tag may be a single word 
 * (<CODE>NAME=xyz value=1</CODE>) or must be enclosed in quotes 
 * if they contain whitespace 
 * (<CODE>NAME=xyz VALUE="This is a test"</CODE>).
 *
 * <p><dt><b>Case sensitivity and SGML</b>
 * <dd>This class does not care about case when reading
 * SGML tags or their attributes, so uppercase, or lowercase, or any
 * combination thereof may be used.  The text in attribute values
 * is not translated to upper- or lowercase but is passed on intact.
 *
 * <p><dt><b>Error handling</b> 
 * <dd>To simplify error detection and correction,
 * exceptions thrown by JServSSI or called servlets are printed,
 * enclosed in comments ("&lt!-- ... --&gt"), in the HTML output.
 *
 * <p><dt><b>Known bugs</b>
 * <dd>Currently, the parameters set for the first 
 * servlet on a page will still be set in the next servlet.
 * </dl>
 * </blockquote>
 *
 * @author Roger Zeng
 * @author Tim Williams
 * @see javax.servlet.http.HttpServlet
 */

public class JServSSI 
    extends HttpServlet 
{
    // cache parsed pages and last modification time for such
    private Hashtable pages = null, times = null;

    /**
     * Initialize JServSSI.
     * @param config servlet configuration, stored by superclass
     * @exception ServletException passed on from superclass
     */
    public void init(ServletConfig config)
	throws ServletException
    {
	super.init(config);
	if (pages == null) {
	    pages = new Hashtable();
	    times = new Hashtable();
	}
    }

    /**
     * Handle POST the same as GET.
     * This method is simply a call to doGet().
     * @param req encapsulates the request to the servlet 
     * @param resp encapsulates the response from the servlet 
     * @see javax.servlet.http.HttpServletRequest#getPathTranslated
     * @exception ServletException will be passed on from included servlets
     * @see #doGet
     */
    public void doPost(HttpServletRequest req, HttpServletResponse res)
	throws ServletException, IOException
    {
	doGet(req, res);
    }

    /**
     * Process named page HTML passed found via getPathTranslated().
     * @param req encapsulates the request to the servlet 
     * @param resp encapsulates the response from the servlet 
     * @see javax.servlet.http.HttpServletRequest#getPathTranslated
     * @exception ServletException will be passed on from included servlets
     */
    public void doGet(HttpServletRequest req, HttpServletResponse res)
	throws ServletException, IOException
    {
	// parse file contents
	String path = req.getPathTranslated();
	File file = new File(path);

	// retrieve and interpret page
	Vector parts = getPageParts(file);
	res.setContentType("text/html");
	interpretPage(parts, 
		(HttpServletRequest) req, (HttpServletResponse) res);
    }

    /**
     * Clean up servlet when finished.
     */
    public void destroy()
    {
	super.destroy();
	pages = times = null;
    }

    /**
     * If HTML file has changed or has not yet been loaded, load
     * file and chop into sections, storing result for future use.
     * Otherwise, return stored preprocessed page.
     * @param file file for which we want page section list
     * @return list of page sections, as described in parsePage().
     * @see #parsePage
     */
    private Vector getPageParts(File file)
	throws IOException
    {
	// first, check to see if we have cached version
	Vector parts = (Vector) pages.get(file);
	if (parts == null || 
		((Long)times.get(file)).longValue() < file.lastModified()) {
	    // if no cached version, or modified, load
	    times.put(file, new Long(file.lastModified()));
	    parts = parsePage(file);
	    pages.put(file, parts);
	}
	return parts;
    }

    /**
     * Scan through vector of page sections, either printing them
     * or invoking the servlets to which they refer.
     * @param parts page sections, as provide by parsePage()
     * @see #parsePage
     */
    private void interpretPage(Vector parts,
	    HttpServletRequest req, HttpServletResponse res)
	throws ServletException, IOException
    {
	ServletOutputStream out = res.getOutputStream();
	for (int i = 0; i < parts.size(); i++) {
	    Object part = parts.elementAt(i);
	    if (part instanceof SGMLTag) 
		runServlet((SGMLTag) part, 
			(Hashtable) parts.elementAt(++i),
			req, res);
	    else 
		out.print(part.toString());
	}
    }

    /**
     * Invoke servlet for this page section.
     */
    private void runServlet(SGMLTag tag, Hashtable params, 
	    HttpServletRequest req, HttpServletResponse res)
	throws ServletException, IOException
    {
	try {
	    // figure which servlet we want
	    String className = (String) tag.getAttribute("CODE", null);
	    if (className != null) {
		if (className.endsWith(".class"))
		    className = className.substring(0, 
			    className.length() - new String(".class").length());
	    }
	    else {
		className = (String) tag.getAttribute("NAME", null);
		if (className == null)
		    throw new ServletException("no class named in servlet tag");
	    }

	    // set parameters 
	    // FIXME: this is still wrong from previous versions
	    // Parameters set for this servlet will still be present
	    // in next servlet on page!
	    if (params.size() > 0) {
		JServConnection jsc = (JServConnection) req;
		Enumeration e = params.keys();
		ServletOutputStream out = res.getOutputStream();
		while (e.hasMoreElements()) {
		    String key = (String) e.nextElement();
		    jsc.setParameter(key, (String) params.get(key));
		}
	    }

	    // load and execute servlet
	    HttpServlet servlet = (HttpServlet) getServletContext()
		    .getServlet(className);
	    if (servlet == null)
		throw new ClassNotFoundException(
			"can't find servlet named \"" + className + "\"");
	    servlet.service((ServletRequest) req, (ServletResponse) res);
	}
	catch (Exception e) {
	    // display error messages
	    PrintWriter out = new PrintWriter(res.getOutputStream());
	    out.print("<!-- ");
	    e.printStackTrace(out);
	    out.print("-->");
	    out.flush();

	    // pass servlet problems along 
	    if (e instanceof ServletException || e instanceof IOException)
		throw (ServletException) e;
	}
    }

    /**
     * Open and read file, returning list of contents.
     * The returned vector will contain several types of object:
     * <ul>
     * <li>Objects of class java.lang.String which are parts of
     *     the page which are simply to be echoed verbatim
     * <li>Objects of class SGMLTag which represent any &lt;servlet&gt;
     *     tags encountered and their attributes
     * <li>After each SGMLTag is a java.util.Hashtable, containing
     *     PARAM values set for the preceeding SERVLET tag.
     * </ul>
     * @param file file to open and process
     * @return Vector containing page elements.
     */
    private Vector parsePage(File file)
	throws IOException
    {
	// read file contents
	FileReader fin = new FileReader(file);
	BufferedReader br = new BufferedReader(fin);
	String line, text = "";
	while ((line = br.readLine()) != null)
	    text += line + "\n";

	// scan through page parsing servlet statements
	Vector sections = new Vector();
	int start = 0;
	for (   SGMLTag tag = new SGMLTag(text, 0);
		!tag.finished();
		tag = new SGMLTag(text, tag.end))
	{
	    if (tag.isNamed("SERVLET") && tag.isWellFormed()) {
		SGMLTag servlet = tag;
		Hashtable params = new Hashtable();
		while (!tag.finished()) 
		{
		    // set parameter?
		    tag = new SGMLTag(text, tag.end);
		    if (tag.isNamed("PARAM") && tag.isWellFormed()) {
			Hashtable attrs = tag.getAttributes();
			if (attrs.get("NAME") != null
				&& attrs.get("VALUE") != null) 
			    params.put(attrs.get("NAME"), attrs.get("VALUE"));
		    }

		    // handle end servlet tag?
		    else if (tag.isNamed("/SERVLET") && tag.isWellFormed()) {
			sections.addElement( 
				text.substring(start, servlet.start));
			sections.addElement(servlet);
			sections.addElement(params);
			start = tag.end;
			break;
		    }
		}
	    }
	}
	sections.addElement(text.substring(start, text.length()));
	return sections;
    }

    /**
     * Return information about JServSSI to caller.
     * @return string explaining this what this servlet does
     * and referring the user to the javadocs.
     */
    public String getServletInfo()
    {
	return "JServSSI is a servelet which process <SERVLET> tags "
	    + "in .jhtml files and replaces such tags with the "
	    + "output of the servlets to which they refer. "  
	    + "See the documentation on org.apache.jserv.JServSSI for "
	    + "more information";
    }
}
