// ====================================================================
// Copyright (c) 1997,1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// JServSession.java:
// - org.apache.jserv.JServSession.java

package org.apache.jserv;

import java.util.Hashtable;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionContext;
import javax.servlet.http.HttpSessionBindingListener;
import javax.servlet.http.HttpSessionBindingEvent;

/**
 * This is the object that encapsulates a session.
 *
 * @author Francis J. Lacoste
 */
public class JServSession implements HttpSession {
    //The session data
    Hashtable sessionData;
  
    //The session context
    JServServletManager context;

    //The session id
    String id;

    //The time at which this session has been created
    long creationTime;

    //The last time the session was accessed.
    long lastAccessTime;

    //Is this session valid
    private boolean valid;

    //Is this session new
    private boolean isNew;
  
    /**
     * Creates a new session.
     * @param id The id of the session.
     * @param context The context of the session.
     */
    public JServSession( String id, JServServletManager context ) {
	this.valid = true;
	this.id = id;
	this.context = context;
	this.sessionData = new Hashtable();
	this.creationTime = System.currentTimeMillis();
	this.lastAccessTime = creationTime;
	this.isNew = true;
    }

    /**
     * Returns the identifier assigned to this session. An HttpSession's 
     * identifier is a unique string that is created and maintained by 
     * HttpSessionContext.
     * @return the identifier assigned to this session
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized String getId() {
	checkState();
	return id;
    }

    /**
     * Returns the context in which this session is bound.
     * @return the name of the context in which this session is bound
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized HttpSessionContext getSessionContext() {
	checkState();
	return context;
    }

    /**
     * Returns the time at which this session representation was created,
     * in milliseconds since midnight, January 1, 1970 UTC.
     * @return the time when the session was created.
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized long getCreationTime() {
	checkState();
	return creationTime;
    }

    /**
     * Returns the last time the client sent a request carrying the identifier
     * assigned to the session. Time is expressed
     * as milliseconds since midnight, January 1, 
     * 1970 UTC. 
     * Application level operations, such as getting or setting a value
     * associated with the session, does not affect the access time.
     *
     * <P> This information is particularly useful in session management
     * policies.  For example,
     * <UL>
     * <LI>a session manager could leave all sessions
     * which have not been used in a long time 
     * in a given context.
     * <LI>the sessions can be sorted according to age to optimize some task.
     * </UL>
     * @return the last time the client sent a request carrying the identifier 
     * assigned to the session.
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized long getLastAccessedTime() {
	checkState();
	return lastAccessTime;
    }

    /**
     * Causes this representation of the session to be invalidated and removed 
     * from its context.
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized void invalidate() {
	checkState();
	valid = false;
	context.removeSession( this );
    }

    /**
     * Binds the specified object into the session's application layer data
     * with the given name.  Any existing binding with the same name is
     * replaced.  New (or existing) values that implement the
     * HttpSessionBindingListener interface will call its  
     * valueBound() method.
     * @param name the name to which the data object will be bound.  This
     * parameter cannot be null.
     * @param value the data object to be bound.  This parameter cannot be null.
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */ 
    public synchronized void putValue( String name, Object value ) {
	checkState();
	sessionData.put( name, value );
	if ( value instanceof HttpSessionBindingListener ) {
	    HttpSessionBindingListener listener = 
		(HttpSessionBindingListener)value;
	    listener.valueBound( new HttpSessionBindingEvent( this, name ) );
	}
    }

    /**
     * Returns the object bound to the given name in the session's
     * application layer data.  Returns null if there is no such binding.
     * @param name the name of the binding to find
     * @return the value bound to that name, or null if the binding does
     * not exist.
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized Object getValue( String name ) {
	checkState();
	return sessionData.get( name );
    }

    /**
     * Removes the object bound to the given name in the session's
     * application layer data.  Does nothing if there is no object
     * bound to the given name.  The value that implements the
     * HttpSessionBindingListener interface will call its
     * valueUnbound() method.
     * @param name the name of the object to remove
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized void removeValue( String name ) {
	checkState();
	Object value = sessionData.remove( name );
	if ( value instanceof HttpSessionBindingListener ) {
	    HttpSessionBindingListener listener = 
		(HttpSessionBindingListener)value;
	    listener.valueUnbound( new HttpSessionBindingEvent( this, name ) );
	}
    }
    /**
     * Returns an array of the names of all the application layer
     * data objects bound into the session. For example, if you want to delete
     * all of the data objects bound into the session, use this method to 
     * obtain their names.
     * @return an array containing the names of all of the application layer 
     * data objects bound into the session
     * @exception IllegalStateException if an attempt is made to access  
     * session data after the session has been invalidated
     */
    public synchronized String[] getValueNames() {
	checkState();
	Vector buf = new Vector();
	Enumeration namesEnum = sessionData.keys();
	while (namesEnum.hasMoreElements() ) {
	    buf.addElement( namesEnum.nextElement() );
	}
	String[] names = new String[ buf.size() ];
	buf.copyInto( names );
	return names;
    }

    /**
     * A session is considered to be "new" if it has been created by the server, 
     * but the client has not yet acknowledged joining the session. For example,
     * if the server supported only cookie-based sessions and the client had 
     * completely disabled the use of cookies, then calls to
     * HttpServletRequest.getSession() would 
     * always return "new" sessions.
     * @return true if the session has been created by the server but the 
     * client has not yet acknowledged joining the session; false otherwise
     */ 
    public synchronized boolean isNew() {
	checkState();
	return isNew;
    }

    /**
     * Has the session been invalidated.
     */
    public boolean isValid() {
	return valid;
    }

    /**
     * Tells the session that it has been accessed
     */
    public synchronized void access() {
	lastAccessTime = System.currentTimeMillis();
	isNew = false;
    }
  
    /**
     * Throws an IllegalStateException when the session is no longer valid.
     */
    private void checkState() {
	if ( !valid ) 
	    throw new IllegalStateException( "Session " + id +
					     "has been invalidated." );
    }
    
}

	
