// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

// JServSignals.java:
// - org.apache.jserv.JServSignals

package org.apache.jserv;

import java.util.Hashtable;
import java.util.Enumeration;

import java.io.IOException;

/**
 *
 *
 * This class does all the JServ signal-handling stuff. It reads from
 * System.in, and looks for lf-terminated lines that look like "S%d\n",
 * where %d is the number of the signal.
 * 
 * <p>When it receives a SIGHUP or SIGTERM, it calls destroyServlets() in
 * all the servlet manager and then exit.
 * 
 * @see JServServletManager#destroyServlets
 *
 * @author Alexei Kosut
 * @author Francis J. Lacoste
 */
class JServSignals extends Thread 
implements JServSendError, JServDebug.DebugConstants 
{
    private static final int SIGHUP = 1;
    private static final int SIGTERM = 15;

    Hashtable servletMgrTable;

    /**
     * Creates the signal handler thread and starts it.
     * @param table A table of servlet manager to call back when 
     * we receive a signal.
     */
    protected JServSignals(Hashtable table) {
	this.servletMgrTable = table;

	setDaemon(true);
	start();
    }

    // This thread just waits for something to appear on in.
    public void run() {
	String line;
	char sig[] = new char[3];

	try {
	    for (;;) {
		try {
		//XXX: We are doing this because with green_threads
		//jdk implementation, blocking on IO blocks all the
		//threads and not just the reading one.
		    while ( ! JServHandler.in.ready() )
			sleep( 300 );
		}
		catch(InterruptedException e) {
		    ;
		}

		//Normally, we should block until there is data
		if (JServHandler.in.read( sig ) < 3)
		    break;

		line = new String(sig);

		if (line.equals("S01")) {
		    restartServlets();  // SIGHUP
		}
		else if (line.equals("S15")) {
		    destroyServlets();	// SIGTERM

		}
	    }
	}
	catch(IOException e) {
	}

	// We reached eof (null) or got an IO exception. Either way,
	// something happened to our parent. Maybe it got killed.
	// We should get killed, too.
	//
	// It's okay, though. If it's an accident, mod_jserv will
	// notice that we've died, and will reactivate us.
	JServDebug.trace( "Lost connection to parent: shutting JServ down.",
			  SIGNAL );
	destroyServlets();
    }

    // This method will be called when a SIGTERM is caught.
    protected synchronized void destroyServlets() {
	JServDebug.trace("Received SIGTERM : shutting JServ down.",
			 SIGNAL);
	// Destroy all the servlets and exit
	clearServlets();
	System.exit(0);
    }
    
    // This method will be called when a SIGHUP is caught.
    protected synchronized void restartServlets() {
	JServDebug.trace("Received SIGHUP : shutting JServ down.",
			 SIGNAL);
	// Clear the servlets
	clearServlets();
	
	// Restart every servlet manager.
	Enumeration mgrEnum = servletMgrTable.elements();
	while (mgrEnum.hasMoreElements()) {
	    JServServletManager mgr = (JServServletManager)mgrEnum.nextElement();
	    mgr.init( this );
	}
    }

    // This clears the servlets
    private void clearServlets() {
	//Tell each manager to destroy its servlet.
	Enumeration mgrEnum = servletMgrTable.elements();
	while ( mgrEnum.hasMoreElements() ) {
	    JServServletManager mgr = (JServServletManager) mgrEnum.nextElement();
	    mgr.destroyServlets();
	}
    }

    /** Report a problem encountered while initializing. **/
    public void sendError(int sc, String msg) {
	JServDebug.trace( JServConnection.findStatusString( sc ) + ": " + msg,
			  ALWAYS );
    }

    /** Report an exception or error encountered while loading a servlet. **/
    public void sendError(Throwable e) {
	JServDebug.trace( e );
    }

}
