// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// JServUtils.java:
// - org.apache.jserv.JServUtils

package org.apache.jserv;

import java.net.URLEncoder;

import java.util.Date;
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.Locale;
import java.util.TimeZone;
import java.util.NoSuchElementException;

import java.text.SimpleDateFormat;

import javax.servlet.http.Cookie;
/**
 * Various utility methods used by the servlet engine.
 *  
 * @author Francis J. Lacoste
 * @author Ian Kluft
 * @verstion $Revision: 1.1 $
 */
public final class JServUtils {

  private JServUtils() { }

  /**
     * This method urlencodes the given string. This method is here for
     * symmetry and simplicity reasons and just calls URLEncoder.encode().
     *
     * @param  str the string
     * @return the url-encoded string
     */
    public final static String URLEncode(String str)
    {
	if (str == null)  return  null;

	return URLEncoder.encode(str);
    }

    /**
     * This method decodes the given urlencoded string.
     * 
     * @param  str the url-encoded string
     * @return the decoded string
     * @exception IllegalArgumentException If a '%' is not followed by a valid
     *                           2-digit hex number.
     */
    public final static String URLDecode(String str) throws IllegalArgumentException
    {
	if (str == null)  return  null;

	StringBuffer dec = new StringBuffer();	// decoded string output
	int strPos = 0;
	int strLen = str.length();

	dec.ensureCapacity(str.length());
	while ( strPos < strLen ) {
	    int	laPos;		// lookahead position

	    // look ahead to next URLencoded metacharacter, if any
	    for ( laPos = strPos; laPos < strLen; laPos++) {
		char laChar = str.charAt(laPos);
		if ( laChar == '+' || laChar == '%' )
		    break;
	    }

	    // if there were non-metacharacters, copy them all as a block
	    if ( laPos > strPos ) {
	        dec.append(str.substring(strPos,laPos));
		strPos = laPos;
	    }

	    // shortcut out of here if we're at the end of the string
	    if ( strPos >= strLen ) {
		break;
	    }

	    // process next metacharacter
	    char metaChar = str.charAt(strPos);
	    if ( metaChar == '+' ) {
		dec.append(' ');
		strPos++;
		continue;
	    } else if ( metaChar == '%' ) {
		try {
		    dec.append((char) Integer.parseInt(
			str.substring(strPos+1,strPos+3),16));
		} catch (NumberFormatException e) {
		    throw new IllegalArgumentException("invalid hexadecimal "+
			str.substring(strPos+1,strPos+3) +
			" in URLencoded string" );
		}
		strPos += 3;
	    }
	}

	return dec.toString();
    }

    /**
     * Parse a cookie header into an array of cookies as per the 
     * Netscape Cookie specification.
     *
     * @param cookieHdr The Cookie header value.
     */
    public static Cookie[] parseCookieHeader( String cookieHdr ) {
	Vector cookieJar = new Vector();
	if ( cookieHdr != null ) {
	    StringTokenizer stok = new StringTokenizer( cookieHdr, "= ;" );
	    while ( stok.hasMoreTokens() ) {
		try {
		    String name = URLDecode( stok.nextToken() );
		    String value = URLDecode( stok.nextToken() );
		    cookieJar.addElement( new Cookie( name, value ) );
		} catch ( IllegalArgumentException badcookie ) {
		} catch ( NoSuchElementException badcookie ) {
		}
	    }
	}
	Cookie[] cookies = new Cookie[ cookieJar.size() ];
	cookieJar.copyInto( cookies );
	return cookies;
    }

    private static SimpleDateFormat cookieDate = 
    new SimpleDateFormat( "EEEE, dd-MMM-yy kk:mm:ss zz", Locale.US );

    static {
	cookieDate.setTimeZone( TimeZone.getTimeZone( "GMT" ) );
    }
	 
    /**
     * Encode a cookie as per the Netscape Cookies specification. The 
     * resulting string can be used in a Set-Cookie header.
     * @param cookie The cookie to encode.
     * @return A string following Netscape Cookies specification.
     */
    public static String encodeCookie( Cookie cookie ) {
	StringBuffer buf = new StringBuffer( cookie.getName() );
	buf.append( '=' );
	buf.append( cookie.getValue() );
	int age = cookie.getMaxAge();
	if ( cookie.getMaxAge() > 0 ) {
	    buf.append( "; expires=" );
	    buf.append( cookieDate.format(new Date(System.currentTimeMillis() +
						   age * 1000 )
					  )
			);
	} else if ( cookie.getMaxAge() == 0 ) {
	    buf.append( "; expires=" );
	    //Set expiration to the epoch to delete the cookie
	    buf.append( cookieDate.format( new Date( 0 ) ) );
	} 
	if ( cookie.getDomain() != null ) {
	    buf.append( "; domain=" );
	    buf.append( cookie.getDomain() );
	}
	if ( cookie.getPath() != null ) {
	    buf.append( "; path=" );
	    buf.append( cookie.getPath() );
	}
	if ( cookie.getSecure() ) {
	    buf.append( "; secure" );
	}
	return buf.toString();
    }

    /**
     * Parse a content-type header for the character encoding. If the
     * content-type is null or there is no explicit character encoding,
     * ISO-8859-1 is returned.
     * @param contentType a content type header.
     */
    public static String parseCharacterEncoding( String contentType ) {
	int start;
	if ( contentType == null || 
	     ( start = contentType.indexOf( "charset=" ) ) == -1 
	       )
	    {
		return "ISO-8859-1";
	    }
	String encoding = contentType.substring( start + 8 );
	//Is there another attribute ?
	int end;
	if ( ( end = encoding.indexOf( ";" ) ) > -1 ) {
	    return encoding.substring( 0, end );
	} else {
	    return encoding;
	}
    }

}
