 /* ====================================================================
 * Copyright (c) 1997 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/*
 * jserv.c: Implements part of Java Server API for Apache
 * by Alexei Kosut <akosut@apache.org>
 *
 * This is a standalone C program that controls the Java process
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>

#include <sys/wait.h>

/* Name of the class to run - not configurable */
#define JSERV_CLASS "apache.jserv.JServHandler"

/* Child #2 - This is created by child #1 */

static int jserv_child (char *args[], char *envp[]) {
    if (execve(args[0], args, envp) < 0) {
	perror("jserv exec failed");
    }

    return 0;
}

/* Make java arguments */
char **create_args(int argc, char *argv[]) {
    char **args = malloc((argc + 3) * (sizeof (char *)));
    int i;

    /* java apache.jserv.JServHandler -m <args> */
    args[0] = argv[1];

    /* Pass -J options to Java */
    for (i = 1; !strncmp(argv[i + 1], "-J", 2); i++) {
	args[i] = 2 + argv[i + 1];
    }

    args[i++] = JSERV_CLASS;
    args[i++] = "-m";
   
    for (; (i - 1) < argc; i++) {
	args[i] = argv[i - 1];
    }

    args[++i] = NULL;

    return args;
}

/* These is a signal handler for child #1. When it gets a signal,
 * it passes it on to Apache.
 */

static FILE *jserv_in = NULL;

static void jserv_signal(int sig) {
    fprintf(jserv_in, "S%02d", sig);
    fflush(jserv_in);

    if (sig == SIGTERM) {
	fclose(jserv_in);
	waitpid(-1, NULL, 0);
	exit(0);
    }
}

/* Child #1 - This is created by Apache */

int main (int argc, char *argv[], char *envp[]) {
    int i, pid;
    int in_fds[2];

    char **args = create_args(argc, argv);

    if (argc < 3) {
	fprintf(stderr, "jserv: Illegal arguments\n");
	fprintf(stderr, "usage: jserv javabin [-Jopt1 -Jopt2...] [-t] propfile port\n");
	exit(1);
    }

    do {
	if (jserv_in) {
	    fclose(jserv_in);
 	    free(jserv_in);
	}
	
	/* Spawn child */

	if (pipe(in_fds) < 0) {
	    fprintf(stderr, "Could not open JServ pipes\n");
	    exit(1);
	}

	if ((pid = fork()) < 0) {
	    close (in_fds[0]);
	    close (in_fds[1]);
	    fprintf(stderr, "Could not fork JServ child process\n");
	    exit(1);
	}

	if (!pid) {
 	    /* Child process */

	    close (in_fds[1]);
	    dup2(in_fds[0], STDIN_FILENO);
	    close (in_fds[0]);

	    signal (SIGCHLD, SIG_DFL);

	    return jserv_child(args, envp);
	}

	/* Parent */

	close(in_fds[0]);
	jserv_in = fdopen(in_fds[1], "w");
    
	/* Set up signal handlers */
	signal(SIGTERM, jserv_signal);
	signal(SIGHUP, jserv_signal);
	    
	/* Poll for a change. Every second, check the following items
	 * to see if we should do something:
	 */
	for (;;) {
	    /* Wait for a sec */
	    sleep(1);
	    
	    /* Check on the child pid. If the status changed, it means
	     * that JServ has exited, and we should reinstall it.
	     */
	    if (waitpid(pid, NULL, 0) == pid)
		break;
	}

	/* Sleep for a few secs before reinstalling Java */
	sleep(3);
    } while (1);
}
