// ====================================================================
// Copyright (c) 1997, 1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

// JServClassLoader.java:
// - org.apache.jserv.JServClassLoader

package org.apache.jserv;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;

import java.net.URL;

import java.util.Date;
import java.util.Hashtable;
import java.util.Vector;
import java.util.Enumeration;

import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipException;

import java.text.SimpleDateFormat;
/*
 * TODO: A more obvious explanation of the _point_ of the classloader
 * design, and why it achieves autoreloading. <mbp 19971217>
 */

/**
 * A class loader that loads classes from directories and/or zip-format
 * file such as JAR file. It tracks the modification time of the classes
 * it loads to permit reloading through re-instantiation.
 *
 * <P>The classloader is managed by a <CODE>JServServletManager</CODE>.
 * When the classloader reports that one of the classes it has loaded
 * has changed on disk, the servlet manager discards the classloader
 * and creates a new instance using <CODE>reinstantiate</CODE>.
 * The classes are then reloaded into the new classloader as required.
 *
 * <P>The classloader can also load resources, which are a means
 * for packaging application data such as images within a jar file
 * or directory.
 *
 * <P>The classloader always first tries to load classes and resources
 * from its own path, and uses the system classloader if that fails.
 *
 * <P><B>How autoreload works:</B></P>
 *
 * <P>The Java VM considers two classes the same if they have the same
 * fully-qualified name <B>and</B> if they were loaded from the same
 * <CODE>ClassLoader</CODE>.
 *
 * <P>There is no way for a classloader to 'undefine' a class once it
 * has been loaded.  However, the servlet engine can discard a
 * classloader and the classes it contains, causing the
 *
 * <P>The <CODE>JServServletManager</CODE> creates a new instance of
 * the classloader each time it detects that any of the loaded classes
 * have changed.
 *
 * <P>Before terminating, all servlets are destroyed.
 *
 * <P>According to the Java Language Specification (JLS), classes may
 * be garbage-collected when there are no longer any instances of that
 * class and the <CODE>java.lang.Class</CODE> object is finalizable.
 * It is intended that this be the case when a <CODE>JServClassLoader</CODE>
 * is discarded.
 *
 * <P>Many VM releases did not implement class garbage collection
 * properly.  In such a VM, the memory usage will continue to grow if
 * autoreloading is enable.  Running the VM with
 * <CODE>-verbosegc</CODE> (or the corresponding option for
 * non-Javasoft VMs) may give some debugging information.
 *
 * <P>It is important that the <CODE>destroy</CODE> method be
 * implemented properly, as servlets may be destroyed and
 * reinitialized several times in the life of a VM.
 *
 * @see java.lang.ClassLoader
 *
 * @author Francis J. Lacoste
 * @author Martin Pool
 * @version $Revision: 1.5 $ $Date: 1998/02/13 13:45:03 $
 **/
public class JServClassLoader extends ClassLoader 
implements JServDebug.DebugConstants 
{
    /**
     * Cache of the loaded classes. This contains ClassCacheEntry keyed 
     * by class names.
     */
    private Hashtable cache;

    /**
     * Private class used to maintain information about the classes that
     * we loaded.
     */
    private static class ClassCacheEntry {
	Class loadedClass;	//The actual loaded class

	/** The file from which this class was loaded; or null if
	 * it was loaded from the system. **/
	File origin;

	/** The time at which the class was loaded from the origin
	 * file, in ms since the epoch. **/
	long lastModified;

	/** Check whether this class was loaded from the system. **/
	public boolean isSystemClass() {
	    return origin == null;
	}
    }
    
    /**
     * The classpath which this classloader searches for class definitions.
     * Each element of the vector should be either a directory, a .zip
     * file, or a .jar file.
     **/
    private Vector repository;

    //------------------------------------------------------- Constructors

    /**
     * Creates a new class loader that will load classes from the
     * file specified as argument. 
     * @param classRepository This can either be a directory or a zip/jar file.
     * @throw java.lang.IllegalArgumentException if the file is not a directory 
     *           or a zip/jar file.
     */
    public JServClassLoader(File classRepository) {
	this(new File[] {
	    classRepository
		});
    }

    /**
     * Creates a new class loader that will load classes from a specified
     * path.
     *
     * @param classRepository An array of directories and/or a zip/jar file.
     *
     * @throw java.lang.IllegalArgumentException if the file is not a
     * directory or a zip/jar file.
     */
    public JServClassLoader(File[] classRepository)
    {
	 cache = new Hashtable();
	 repository = new Vector(classRepository.length);

	 //Verify that all the repository are valid.
	 for (int i = 0; i < classRepository.length; i++) {
	     File file = classRepository[i];
	     //Check to see if we have proper access.
	     if (!file.canRead()) {
		 throw new IllegalArgumentException("Don't have read access for " +
						    "file " + file.getName());
	     }

	     // Check that it is a directory or zip/jar file
	     // FIXME: Should probably check format rather than name
	     if (file.isDirectory() ||
		 file.getName().endsWith(".jar") ||
		 file.getName().endsWith(".zip")) {
		 repository.addElement(file);
	     }
	     else {
		 throw new IllegalArgumentException(file.getName() + " is not a " +
						    "directory or zip/jar file.");
	     }
	 }

	 // Increment and store generation counter
	 this.generation = generationCounter++;
     }
    

    /**
     * Check to see if a given class should be reloaded because of a 
     * modification to the original class.
     * @param className The name of the class to check for modification.
     */
    public synchronized boolean shouldReload(String classname) {
	ClassCacheEntry entry = (ClassCacheEntry) cache.get(classname);
	debugTrace("Should we reload " + classname + "?" );
	if (entry == null) {
	    debugTrace( classname + " wasn't even loaded." );
	    return false;
	} else {
	    if (entry.isSystemClass()) {
		//System classes cannot be reloaded
		debugTrace( classname + " is from system : can't reload." );
		return false;
	    }
	    boolean reload = entry.origin.lastModified() > entry.lastModified;
	    debugTrace(entry.origin.lastModified() + " > " + 
		       entry.lastModified + " : " + 
		       (reload ? " yes." : " no.") );
	    return reload;
	}
    }

    /** Check whether the classloader should be reinstantiated.
     *
     * <P>The classloader must be replaced if there is any class whose origin
     * file has changed since it was last loaded.
     *
     * @since mod_jserv 0.9.9
     **/
    public synchronized boolean shouldReload()
    {
	// Check whether any class has changed
	Enumeration en = cache.elements();
	while (en.hasMoreElements()) {
	    ClassCacheEntry entry = (ClassCacheEntry) en.nextElement();
	    
	    if (entry.isSystemClass())
		continue;

	    // XXX: Because we want the classloader to be an accurate
	    // reflection of the contents of the repository, we also
	    // reload if a class origin file is now missing.  This
	    // probably makes things a bit more fragile, but is OK in
	    // a development situation. <mbp@pharos.com.au>
	    
	    long msOrigin = entry.origin.lastModified();

	    if (msOrigin == 0) {
		trace( entry.origin.toString()+
		       " no longer exists, should reload" );
		return true;
	    }
	    
	    if (msOrigin > entry.lastModified) {
		trace( entry.origin.toString() + " is new, should reload" );
		return true;
	    }
	}

	// No changes, no need to reload
	return false;
    }

    /**
     * Re-instantiate this class loader. This method creates a new instance
     * of the class loader that will load classes form the same path
     * as this one.
     **/
    public JServClassLoader reinstantiate() {
	File[] classRepository = new File[repository.size()];
	repository.copyInto(classRepository);
	return new JServClassLoader(classRepository);
    }
 
    /**
     * Date/time format for trace messages
     */
    static private final SimpleDateFormat traceDateFormat =
    new SimpleDateFormat("[dd/MM/yyyy HH:mm:ss zz]");
 
 
    /** Generation number of the classloader, used to distinguish between
     * different instances. **/
    private int generation;
 
    /** Generation counter, incremented for each classloader as they are
     * created. **/
    static private int generationCounter = 0;      
     
    private void trace(String s) {
	JServDebug.trace("cl#" + Integer.toString(generation) +
			 ": " + s, CLASSLOADER );
    }
  
    private void debugTrace(String s) {
	JServDebug.trace("cl#" + Integer.toString(generation) +
			 ": " + s, CLASSLOADER_VERBOSE );
    }
     
   //------------------------------------ Implementation of Classloader

    /* XXX: The javadoc for java.lang.ClassLoader says that the
     * ClassLoader should cache classes so that it can handle repeated
     * requests for the same class.  On the other hand, the JLS seems
     * to imply that each classloader is only asked to load each class
     * once.  Is this a contradiction?
     *
     * Perhaps the second call only applies to classes which have been
     * garbage-collected?
     */

    /** Resolves the specified name to a Class. The method loadClass()
     * is called by the virtual machine.  As an abstract method,
     * loadClass() must be defined in a subclass of ClassLoader.
     *
     * @param	   name     the name of the desired Class.
     *
     * @param      resolve  true if the Class needs to be resolved; false
     *             if the virtual machine just wants to determine
     *             whether the class exists or not
     *
     * @return	   the resulting Class, or null if it was not found.
     *
     * @exception  ClassNotFoundException  if the class loader cannot find
     *               a definition for the class.
     **/
    protected synchronized Class loadClass(String name, boolean resolve)
	 throws ClassNotFoundException
    {
	Class c = null;		//The class object that will be returned.

	debugTrace( "Asked to load " + name );

	// Use the cached value, if this class is already loaded into
	// this classloader.
	ClassCacheEntry entry = (ClassCacheEntry) cache.get(name);
	if (entry != null) {
	    c = entry.loadedClass;
	    debugTrace( "Loaded class " + name + " from cache." );
	    if (resolve)
		resolveClass(c);
	    return c;
	}


	if (!securityAllowsClass(name)) {
	    debugTrace( name + " must be loaded by system" );
	    return loadSystemClass(name, resolve);
	}

	//Try to load it from each repository
	Enumeration repEnum = repository.elements();
	
	//Cache entry.
	ClassCacheEntry classCache = new ClassCacheEntry();
	while (repEnum.hasMoreElements()) {
	    byte[]classData;
	    
	    File file = (File) repEnum.nextElement();
	    try {
		if (file.isDirectory()) {
		    classData = loadClassFromDirectory(file, name, classCache);
		}
		else {
		    classData = loadClassFromZipfile(file, name, classCache);
		}
	    } catch(IOException ioe) {
		JServDebug.trace( ioe );//Error while reading in data
		
		classData = null;	//Consider as not found
		
	    }
	    if (classData != null) {
		if (file.isFile()) {
		    debugTrace( "Loaded class " + name +
				" from zip file " + file.getName() );
		}
		else {
		    debugTrace( "Loaded class " + name + 
				" from directory " + file.getName() );
		}

		//Define the class
		c = defineClass(name, classData, 0, classData.length);
		debugTrace( "Defined class " + name );
		
		//Cache the result;
		classCache.loadedClass = c;
		//Origin is set by the specific loader
		classCache.lastModified = classCache.origin.lastModified();
		cache.put(name, classCache);

		//Resolve it if necessary
		if (resolve) {
		    resolveClass(c);
		    debugTrace( "Resolved class " + name );
		}
		return c;
	    }
	}

	debugTrace( name + " not found in repository" );
	return loadSystemClass(name, resolve);
    }

    /** Load a class using the system classloader. **/
    private Class loadSystemClass(String name, boolean resolve)
	 throws NoClassDefFoundError, ClassNotFoundException
    {
	Class c = findSystemClass(name);
	// Throws if not found.

	// Add cache entry
	ClassCacheEntry cacheEntry = new ClassCacheEntry();
	cacheEntry.origin = null;
	cacheEntry.loadedClass = c;
	cacheEntry.lastModified = Long.MAX_VALUE;
	
	cache.put(name, cacheEntry);
	debugTrace("Loaded class " + name + " from system.");

	if (resolve)
	    resolveClass(c);

	return c;
    }

    /** Checks whether a classloader is allowed to define a given class,
     * within the security manager restrictions. **/
    private boolean securityAllowsClass(String className) {
	try {
	    SecurityManager security = System.getSecurityManager();
	    if (security == null)
		return true;

	    int lastDot = className.lastIndexOf('.');
	    String pkgName =
		(lastDot > -1) ? className.substring(0, lastDot) : "";
	    security.checkPackageDefinition(pkgName);
	    // Throws if not allowed
	    return true;
	}
	catch (SecurityException e) {
	    return false;
	}
    }

    /**
     * Tries to load the class from a directory.
     * @param dir The directory that contains classes.
     * @param name The classname
     * @param cache The cache entry to set the file if successful.
     */
    private byte[] loadClassFromDirectory(File dir, String name,
					  ClassCacheEntry cache)
	 throws IOException
    {
	//Translate class name to file name
	String classFileName =
	    name.replace('.', File.separatorChar) + ".class";

	//Check for garbage input at beginning of file name i.e. ../ or similar
	if     (!Character.isJavaIdentifierStart(classFileName.charAt(0))) {
	    //Find real beginning of class name
	    int start = 1;
	    while (!Character.isJavaIdentifierStart(classFileName.charAt(start))) {
		start++;
	    }
	    classFileName = classFileName.substring(start);
	}

	File classFile = new File(dir, classFileName);

	if (classFile.exists()) {
	    cache.origin = classFile;
	    InputStream in = new FileInputStream(classFile);
	    try {
		return loadBytesFromStream(in, (int)classFile.length());
	    }
	    finally {
		in.close();
	    }
	}
	else {

	    //Not found
	    return null;
	}
    }

    /**
     * Tries to load the class from a zip file.
     * @param file The zipfile that contains classes.
     * @param name The classname
     * @param cache The cache entry to set the file if successful.
     */
    private byte[] loadClassFromZipfile(File file, String name,
					ClassCacheEntry cache)
	 throws IOException
    {
	//Translate class name to file name
	String classFileName = name.replace('.', '/') + ".class";

	ZipFile zipfile = new ZipFile(file);

	try {
	  ZipEntry entry = zipfile.getEntry(classFileName);
	  if (entry != null) {
	    cache.origin = file;
	    return loadBytesFromStream(zipfile.getInputStream(entry),
                                       (int)entry.getSize());
	  } else {
	    //Not found
	    return null;
	  }
	} finally {
	  zipfile.close();
	}
    }

    /**
     * Loads all the bytes of an InputStream.
     */
    private byte[] loadBytesFromStream(InputStream in, int length)
         throws IOException
     {
	byte[] buf = new byte[length];
	int nRead, count = 0;
	while ((length > 0) && ((nRead = in.read(buf,count,length)) != -1)) {
            count += nRead;
            length -= nRead;
	}
        return buf;
    }

    /**
     * Get an InputStream on a given resource.  Will return null if no
     * resource with this name is found.
     *
     * <P>The JServClassLoader translate the resource's name to a file
     * or a zip entry. It looks for the resource in all its repository
     * entry.
     *
     * @see     java.lang.Class#getResourceAsStream(String)
     * @param	name	the name of the resource, to be used as is.
     * @return	an InputStream on the resource, or null if not found.
     */
    public InputStream getResourceAsStream(String name) {
	//Try to load it from the system class
	InputStream s = getSystemResourceAsStream(name);
	if          (s == null) {
	    //Try to find it from every repository
	    Enumeration repEnum = repository.elements();
	    while       (repEnum.hasMoreElements()) {
		File file = (File) repEnum.nextElement();
		if   (file.isDirectory()) {
		    s = loadResourceFromDirectory(file, name);
		}
		else {
		    s = loadResourceFromZipfile(file, name);
		}
		if (s != null) {
		    break;	//Terminate search

		}
	    }
	} else {
	    debugTrace( "Loaded resource " + name + " from system." );
	}
	return s;
    }

    /**
     * Loads resource from a directory.
     */
    private InputStream loadResourceFromDirectory(File dir, String name) {
	//Name of resources are always separated by /
	String fileName = name.replace('/', File.separatorChar);
	File resFile = new File(dir, fileName);
	if (resFile.exists()) {
	    try {
		debugTrace( "Loaded resource " + name +
			    " from directory" + dir.getName() );
		return new FileInputStream(resFile);
	    } catch (FileNotFoundException shouldnothappen) {
		shouldnothappen.printStackTrace();
		return null;
	    }
	} else {
	    return null;
	}
    }

    /**
     * Loads resource from a zip file
     */
    private InputStream loadResourceFromZipfile(File file, String name) {
	try {
	    ZipFile zipfile = new ZipFile(file);
	    ZipEntry entry = zipfile.getEntry(name);
	    if (entry != null) {
		debugTrace("Loaded resource " + name + " from zip file" +
		      file.getName() );
		return zipfile.getInputStream(entry);
	    } else {
		return null;
	    }
	} catch(IOException ioe) {
	    JServDebug.trace( ioe );
	    return null;
	}
    }

    /**
     * Find a resource with a given name.  The return is a URL to the resource.
     * Doing a getContent() on the URL may return an Image, an AudioClip,
     * or an InputStream.<p>
     *
     * The JServClassLoader looks for the resource only in the directory 
     * repository for this resource.
     *
     * @param	name	the name of the resource, to be used as is.
     * @return	an URL on the resource, or null if not found.
     * @since   JDK1.1
     */
    public java.net.URL getResource(String name) {
	URL u = getSystemResource(name);
	if  (u != null) {
	    debugTrace( "Loaded resource " + name + " from system." );
	    return u;
	}			
	//Load for it only in directories since no URL can point into 
	//a zip file.

	Enumeration repEnum = repository.elements();
	while (repEnum.hasMoreElements()) {
	    File file = (File) repEnum.nextElement();
	    if (file.isDirectory()) {
		String fileName = name.replace('/', File.separatorChar);
		File resFile = new File(file, fileName);
		if (resFile.exists()) {
		    debugTrace( "Loaded resource " + name + 
				" from directory " + file.getName() );
		    //Build a file:// URL form the file name
		    try {
			return new URL("file://" + resFile.getAbsolutePath());
		    } catch(java.net.MalformedURLException badurl) {
			badurl.printStackTrace();
			return null;
		    }
		}
	    }
	}
	//Not found
	return null;
    }

}
