// ====================================================================
// Copyright (c) 1997,1998 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

// JServHandler.java:
// - org.apache.jserv.JServHandler

package org.apache.jserv;

import java.util.Properties;
import java.util.Vector;
import java.util.Hashtable;
import java.util.StringTokenizer;
import java.util.Enumeration;
import java.util.NoSuchElementException;

import java.io.File;
import java.io.BufferedInputStream;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.File;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpUtils;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.http.HttpSessionContext;

/**
 * Class that encapsulates the loading and managing of servlets per
 * virtual server.
 *
 * <p><font size="+2">Configuration options taken from system properties
 *    </font> :
 *
 * <ul>
 * <li><var>jserv.timeout</var> - the number of millisecond to wait on
 * finishing connections before giving up on destroying a servlet. Default to
 * 7000 (7 seconds)
 * <li><var>jserv.autoreload.classes</var> - 
 * Whether to reload servlets whenever one of the loaded class is modified.
 * Default to true.
 * <li><var>jserv.autoreload.file</var> - Whether to reload servlets when
 * the servlet configuration file changes.
 * </ul>
 * 
 * <P><b>Note about synchronization</b> :
 *
 * <p>All the method that modifies the servlet table are synchronized
 * on the JServServletManager. Since this table is private there no needs
 * to synchronized on it anymore. 
 *
 * @author Alexei Kosut
 * @author Francis J. Lacoste
 * @author Martin Pool
 *
 * @version $Revision: 1.8 $ $Date: 1998/02/11 23:46:15 $
 */
public class JServServletManager implements HttpSessionContext, 
    JServDebug.DebugConstants 
{
    /**
     * The name of the session parameter.
     */
    static final String SESSION_IDENTIFIER = "JServSessionId";
    
    /**
     * The amount of time to wait before giving up on lock when destroying
     * servlet. This is taken from the system property 
     * <var>jserv.timeout</var> and default to 7000 ( 7 seconds ).
     */
    static final long timeout = 
    Long.parseLong( System.getProperty( "jserv.timeout", "7000" ) );

    /**
     * Determines wheter to check the property file for changes.
     */
    static final boolean checkFile = 
    Boolean.valueOf( System.getProperty( "jserv.autoreload.file", 
					 "true" ) ).booleanValue();


    /**
     * Determines wheter to check the classes for changes.
     */
    static final boolean checkClasses = 
    Boolean.valueOf( System.getProperty( "jserv.autoreload.classes", 
					 "true" ) ).booleanValue();

    /**
     * The file that contains the servlet properties.
     */
    protected File propertyFile;

    /**
     * The time of the last initialization.
     */
    protected long lastInitialization;

    /**
     * The properties containing configuration information
     * for these servlets.
     */
    protected Properties properties;

    /**
     * The default init arguments for all the servlet
     * in this name space.
     */
    protected Properties defaultArgs;

    /**
     * The class loader used for loading new servlet.
     */
    protected JServClassLoader loader;

    /**
     * The cache of the loaded servlet.
     */
    private Hashtable servletContexts;

    /**
     * The ThreadGroup in which the servlets are run.
     */
    protected ThreadGroup tGroup;

    /**
     * The name of this ServletManager.
     */
    protected String name;

    /**
     * The servlet to load on startup.
     */
    protected String startups;

    /**
     * The names of all the named servlet.
     */
    protected Vector servletNames;

    /**
     * The sessions in this manager.
     */
    protected Hashtable sessions;

    /**
     * Creates a new servlet manager.
     * @param name The name of this ServletManager.
     * @param propFile The name of the property file to use.
     * @param servletRep An array of zip/files and directory used
     *        to load the servlet from.
     */
    JServServletManager(String name, String propFile, Vector servletRep) {
	this.name = name;
	servletContexts = new Hashtable();

	//Creates thread group.
	tGroup = new ThreadGroup(name + "-Servlets");

	//Build the class repository
	File[] repository = new File[servletRep.size()];
	for (int i = 0; i < servletRep.size(); i++) {
	    repository[i] = new File((String) servletRep.elementAt(i));
	}			//Build the class loader
	loader = new JServClassLoader(repository);

	this.propertyFile = new File( propFile );
    }

    /**
     * Load the configuration from the property file and load the
     * startup servlets.
     *
     * @param JServSendError An object that can handle errors.
     */
    public synchronized void init( JServSendError errorHandler ) {
	JServDebug.trace( "ServletManager " + name + 
			  " reading configuration file", SERVLET_MANAGER );
	//Load the properties
	properties = new Properties();
	try {
	  InputStream propIn = new FileInputStream( propertyFile );
	  try {
	    properties.load(new BufferedInputStream(propIn));
	  } finally {
	    propIn.close();
	  }
	} catch(IOException ioe) {
	    properties.clear();
	}

	String defaultInitArgsStr =
	    properties.getProperty("servlets.default.initArgs", "");
	defaultArgs = parseInitArgs(defaultInitArgsStr);

	startups = properties.getProperty("servlets.startup", "");
	
	//Get all the named servlet in the property file
	servletNames = new Vector();
	Enumeration names = properties.keys();
	while ( names.hasMoreElements() ) {
	    String prop = (String)names.nextElement();
	    //Servlet name are property of the form 
	    //servlet.<name>.code
	    if ( prop.startsWith( "servlet." ) && prop.endsWith( ".code" ) )
	        {
		    String name = prop.substring( 8, prop.length() - 5 );
		    JServDebug.trace( "Servlet name: " + name, SERVLET_MANAGER
				      );
		    servletNames.addElement( name );
		}
	}
	lastInitialization = propertyFile.lastModified();

	//FIXME: Should we really clear the sessions table after
	//a servlet reload ?
	sessions = new Hashtable();
	
	//Load startup servlets
	loadStartupServlets( errorHandler );
    }

    
    /** 
     * Reinstantiate the classloader if necessary.  Check if any of 
     * the classes has changed or if the property file has been modified.
     * If this is the case, this method does the following :
     * <ol>
     * <li>Destroy all the loaded servlets.
     * <li>Re-read its configuration file.
     * <li>Reload the startup servlets.
     * </ol>
     * 
     * @param errorHandler The object that knows what to do with errors.
     **/
    public synchronized void checkReload( JServSendError errorHandler ) {
	if ( ( checkClasses && loader.shouldReload() ) ||
	     ( checkFile && propertyFile.lastModified() > lastInitialization )
	       )
	    {
		JServDebug.trace( "Re-initing ServletManager " + name, 
				  SERVLET_MANAGER );
		destroyServlets();

		// Create a new class loader so that the class
		// definitions for this servlet and all the
		// classes it depends upon, aside from system
		// classes, are reloaded.
		loader = loader.reinstantiate();

		//Reread configuration and load servlets
		init( errorHandler );
	    }
    }

    /**
     * Get all the name that are defined in this ServletManager
     */
    public Enumeration getServletNames() {
	return servletNames.elements();
    }

    /**
     * Get an enumeration of all the servlets that have been loaded.
     */
    public synchronized Enumeration getLoadedServlets() {
	Vector servlets = new Vector();
	Enumeration loadedServlets = servletContexts.elements();
	while ( loadedServlets.hasMoreElements() ) {
	    JServContext context = (JServContext)loadedServlets.nextElement();
	    servlets.addElement( context.servlet );
	}
	return servlets.elements();
    }
     
    /**
     * Loads and initialize a servlet.  If the servlet is already
     * loaded and initialized, a reference to the existing context
     * is returned.
     *
     * @return the ServletContext object for the servlet.
     *
     * @param servletName The name of the servlet to load.
     * @param errorHandler The error handler to call back if there is an error.
     * @exception ServletException If there is an error while initializing the
     * servlet.
     */
    public synchronized JServContext loadServlet(String name, JServSendError se) 
	 throws ServletException
    {
	// Check whether the servlet is already loaded, initialized,
	// and cached.
	JServContext context = (JServContext) servletContexts.get(name);
	Servlet servlet;
	if (context != null) 
	    return context;

	// Otherwise, load, init, and cache the servlet

	// Find the servlet's full name if an alias
	String sdname = "servlet." + name;
	String classname = properties.getProperty(sdname + ".code");
	boolean isAlias = true;
	if (classname == null) {
	    classname = name;
	    isAlias = false;
	}

	// Load the servlet
	try {
	    servlet = (Servlet) loader.loadClass(classname).newInstance();
	} catch(NoClassDefFoundError e) {
	    se.sendError( HttpServletResponse.SC_NOT_FOUND, classname);
	    return null;
	} catch(ClassNotFoundException e) {
	    se.sendError( HttpServletResponse.SC_NOT_FOUND, classname);
	    return null;
	} catch(ClassFormatError e) {
	    se.sendError(e);
	    return null;
	} catch(IllegalAccessException e) {
	    se.sendError(e);
	    return null;
	} catch(InstantiationException e) {
	    se.sendError(e);
	    return null;
	}

	// Setup the init parameters
	Properties initargs;
	if (isAlias) {
	    String argsline = properties.getProperty(sdname + ".initArgs", null);
	    initargs = parseInitArgs(argsline);
	}
	else {
	    initargs = new Properties(defaultArgs);
	    // Try to load a property file classname.initArgs
	    try {
		InputStream argsIn =
		    loader.getResourceAsStream(classname + ".initArgs");
		if (argsIn != null) {
		    try {
			initargs.load(new BufferedInputStream(argsIn));
		    } finally {
			argsIn.close();
		    }
		}
	    } catch(IOException ioe) {
	    }
	}

	// Init the servlet
	try {
	    context = new JServContext(servlet, this, initargs);
	    JServDebug.trace( "Initializing servlet " + name, 
			      SERVLET_MANAGER );
	    servlet.init(context);
	} catch (ServletException initError ) {
  	    throw initError;
	} catch(Exception e) {
	    // Something happened.
	    se.sendError(e);
	    return null;
	} catch(Error e) {
	    // Something really bad happened...
	    se.sendError(e);
	    throw (Error)e.fillInStackTrace();
	}

	// Add the servlet to the cache, so we can use it again
	servletContexts.put(name, context);
	return context;
    }

    /**
     * Loads and initialize all the startup servlets.
     * @param se The sendError handler to call back in case of error.
     */
    private void loadStartupServlets(JServSendError se) {
	if (startups == null) {
	    return;
	} 

	StringTokenizer st = new StringTokenizer(startups, ", ");

	while (st.hasMoreTokens()) {
	    String servname = st.nextToken();

	    if (servname == null)
		continue;
	    try {
		loadServlet(servname, se);
	    } catch ( ServletException initError ) {
		se.sendError( initError );
	    }
	}
    }


    /**
     * Get the name of this ServletManager.
     */
    public String getName() {
	return name;
    }

    /**
     * Destroy all the servlets and servlet contexts.
     **/
    public synchronized void destroyServlets() {
	JServDebug.trace("destroying Servlets", SERVLET_MANAGER);
	Enumeration contextEnum = servletContexts.keys();
	try {
	    while (contextEnum.hasMoreElements()) {
		String servletName = (String)contextEnum.nextElement();
		JServContext context = 
		    (JServContext) servletContexts.get( servletName );
		
		// Wait until all pending requests have completed.
		//or timeout expires
		try {
		    context.lock.writeLock( timeout );
		} catch ( InterruptedException stop ) {
		    JServDebug.trace( "Caught interrupted exception while " +
				      "waiting for " + servletName + 
				      "Skipping remaing destroy(). ", 
				      SERVLET_MANAGER );
		    return;
		} catch ( JServLock.TimeoutException stilllocked ) {
		    JServDebug.trace( "Timeout for servlet " + servletName + 
				      "expired. Probable deadlock. " +
				      "Skipping destroy()", SERVLET_MANAGER );
		    continue;
		}
		
		// Destroy the servlet
		try {
		    JServDebug.trace( "Destroying servlet " + servletName,
				      SERVLET_MANAGER );
		    context.servlet.destroy();
		} catch(Exception e) {
		    JServDebug.trace( e );
		} catch(Error e) {
		    JServDebug.trace( e );
		    throw e;	// too bad to continue
		}
	    }
	} finally {
	    //Make sure that the servlet tables is empty
	    servletContexts.clear();
	}
    }

    private static final int NAME = 0;
    private static final int VALUE = 1;
    // Convert initArgs line into Properties
    // "<name=value>,<name=value>"
    // \ escape following character, ie \= -> = \, -> ,
    private Properties parseInitArgs(String argsline) {
	Properties initargs;
	if ( defaultArgs != null ) {
	    //Creates with defaults.
	    initargs = new Properties(defaultArgs);
	}
	else {
	    initargs = new Properties();
	}

	if (argsline == null) {
	    return initargs;
	}
	int state = NAME;
	StringBuffer name = new StringBuffer();
	StringBuffer value = new StringBuffer();
	for ( int i=0; i < argsline.length(); i++ ) {
	    char c = argsline.charAt( i );
	    switch ( c ) {
	    case '=':
		state = VALUE;
		break;
	    case ',':
		initargs.put( name.toString(), value.toString() );
		name.setLength(0);
		value.setLength(0);
		state = NAME;
		break;
	    case '\\':
		if ( ++i < argsline.length() ) {
		    c = argsline.charAt( i );
		} else {
		    //Trailing \ ignore.
		    continue;
		}
		//Append the next character
	    default:
		if ( state == NAME ) {
		    name.append( c );
		} else {
		    value.append( c );
		}
		break;
	    }
	}
	//Add last pair
	if ( name.length() > 0 && value.length() > 0 ) {
	  initargs.put( name.toString(), value.toString() );
	}
	return initargs;
    }

    //-----------------------------Static utility methods for session handling

    /**
     * Get the session identifier in a query  string.
     * @param queryStr The query string that came in from the url.
     * @return The session identifier encoded in the url, or null if there
     * is no session identifier in the url.
     */
    public static final String getUrlSessionId( String queryStr ) {
	if ( queryStr == null ) return null;
	try {
	    Hashtable params = HttpUtils.parseQueryString( queryStr );
	    Object o = params.get( SESSION_IDENTIFIER );
	    if ( o == null ) 
		return null;
	    else if ( o instanceof String ) 
		return (String)o;
	    else 
		return ( (String[])o)[0];
	} catch ( IllegalArgumentException badquerystr ) {
	    return null;
	}
    }

    /**
     * Get the session identifier set in cookies.
     * @param cookies The cookies to search for a session identifier.
     * @return The session identifier found in the cookies, or null if there
     * is none.
     */
    public static final String getCookieSessionId( Cookie[] cookies )
    {
	if ( cookies == null || cookies.length == 0 ) return null;
	for ( int i=0; i < cookies.length; i++ ) {
	    if ( cookies[i].getName().equals( SESSION_IDENTIFIER ) ) {
		return cookies[i].getValue();
	    }
	}
	return null;
    }
    
    /**
     * Encode a URL with a session identifier.
     * @param url The url to encode.
     * @param id The session identifier to encode with the url.
     */
    public static String encodeUrl( String url, String id ) {
	//Is there a query string 
	if ( url.indexOf( '?' ) == -1 ) {
	    return url + '?' + SESSION_IDENTIFIER + '=' + id;
	} else {
	    return url + '&' + SESSION_IDENTIFIER + '=' + id;
	}
    }

    /**
     * Get a session identifier. These should be unique identifier.
     */
    public static final String getIdentifier() {
	//Unless this takes less than 1 millis second that shouldn't
	//cause collision.
	return Long.toString( System.currentTimeMillis() );
    }
    
    //----------------------------------- Implementation of HttpSessionContext


    /**
     * Returns the session bound to the specified session ID.
     * @param sessionID the ID of a particular session object.
     * @return the session name. Returns null if the session ID does not refer
     * to a valid session.
     */
    public synchronized HttpSession getSession( String sessionId ) {
	return (HttpSession)sessions.get( sessionId );
    }

    /**
     * Returns an enumeration of all of the session IDs in this context.
     * @return an enumeration of all session IDs in this context.
     */
    public synchronized Enumeration getIds() {
	Vector ids = new Vector();
	Enumeration idEnum = sessions.keys();
	while ( idEnum.hasMoreElements() ) {
	    ids.addElement( idEnum.nextElement() );
	}
	return ids.elements();
    }


    /**
     * Creates a new session.
     * @param response The response used to send a cookie to the
     * client.
     * @return A new session.
     */
    public synchronized JServSession createSession( HttpServletResponse response )
    {
	JServSession s = new JServSession( getIdentifier(), this );
	sessions.put( s.id, s );
	Cookie c = new Cookie( SESSION_IDENTIFIER, s.id );
	c.setDomain( name );
	c.setPath( "/" );
	response.addCookie( c );
	return s;
    }

    /**
     * Remove a session from the context. This is called by the session
     * when it is invalidated.
     * @param s The session to remove from this context.
     */
    public synchronized void removeSession( JServSession s ) {
	sessions.remove( s.id );
    }
}




