// ====================================================================
// Copyright (c) 1997 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServHandler.java - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

package apache.jserv;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

import javax.servlet.*;
import javax.servlet.http.*;

public class JServHandler extends Thread {
    public final static int DEFAULT_PORT = 8007;
    protected int port;
    protected ServerSocket listen_socket;

    protected Properties properties;
    protected Hashtable servlets;
    
    protected String auth;

    // Exit with an error message, when an exception occurs.
    public static void fail(Exception e, String msg) {
	System.err.println(msg + ": " +  e);
        System.exit(1);
    }
    
    // Create a ServerSocket to listen for connections on;  start the thread.
    public JServHandler(String propfile, int port, String auth) {
	// Read in the server properties
	properties = new Properties();

	File propf = new File(propfile);

	if (propf.exists() && propf.canRead()) {
	    try {
		properties.load(new FileInputStream(propf));
	    } catch (IOException e) {
		properties.clear();	/* Reset as empty */
	    }
	}

	// Auth string
	this.auth = auth;

	// Set up the servlet hashtable
	servlets = new Hashtable();

        if (port == 0) port = DEFAULT_PORT;
        this.port = port;
        try { listen_socket = new ServerSocket(port); }
        catch (IOException e) { fail(e, "Exception creating server socket"); }
        this.start();
    }
    
    // The body of the server thread.  Loop forever, listening for and
    // accepting connections from clients.  For each connection, 
    // create a JServConnection object to handle communication through the
    // new Socket.
    public void run() {
        try {
            while(true) {
                Socket client_socket = listen_socket.accept();
                JServConnection c = new
		    JServConnection(client_socket, properties, servlets, auth);
            }
        }
        catch (IOException e) { 
            fail(e, "Exception while listening for connections");
        }
    }
    
    // Start the server up, listening on the specified port
    public static void main(String[] args) {
        int port = 0;
	String propfile = null;
	String auth;

        if (args.length == 2) {
	    propfile = args[0];
            try { port = Integer.parseInt(args[1]);  }
            catch (NumberFormatException e) { port = 0; }
        }

	// Read the auth from the first line of "stdin"

	BufferedReader in =
	    new BufferedReader(new InputStreamReader(System.in));

	try {
	    auth = in.readLine();
	} catch (IOException e) {
	    auth = null;
	}
	
        new JServHandler(propfile, port, auth);
    }
}

// ServletInputStream implementation

class JServInputStream extends ServletInputStream {
    private JServConnection conn;

    // constructors

    public JServInputStream(JServConnection conn) {
	this.conn = conn;
    }

    // methods from ServletInputStream
    public int readLine(byte b[], int off, int len) throws IOException {
	int i;

	for (i = 0; i <= len; i++) {
	    b[off + i] = (byte)conn.in.read();
	    if (b[off + i] == '\n')
		break;
	}

	return i;
    }

    // methods from InputStream

    public int read() throws IOException {
	return conn.in.read();
    }

    public int read(byte b[]) throws IOException {
	return read(b, 0, b.length);
    }

    public int read(byte b[], int off, int len) throws IOException {
	return conn.in.read(b, off, len);
    }

    public long skip(long n) throws IOException {
	return conn.in.skip(n);
    }

    public void close() throws IOException {
	conn.in.close();
    }
}

// ServletOutputStream implementation

class JServOutputStream extends ServletOutputStream {
    private JServConnection conn;

    // constructor
    public JServOutputStream(JServConnection conn) {
	this.conn = conn;
    }

    // methods from ServletOutputStream

    public void print(String s) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(s);
    }

    public void print(boolean b) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(b);
    }

    public void print(char c) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(c);
    }

    public void print(int i) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(i);
    }

    public void print(long l) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(l);
    }

    public void print(float f) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(f);
    }

    public void print(double d) throws IOException {
	conn.sendHttpHeaders();
	conn.out.print(d);
    }

    public void println() throws IOException {
	conn.sendHttpHeaders();
	conn.out.println();
    }

    public void println(String s) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(s);
    }

    public void println(boolean b) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(b);
    }

    public void println(char c) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(c);
    }

    public void println(int i) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(i);
    }

    public void println(long l) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(l);
    }

    public void println(float f) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(f);
    }

    public void println(double d) throws IOException {
	conn.sendHttpHeaders();
	conn.out.println(d);
    }

    // methods from OutputStream

    public void write(int b) throws IOException {
	conn.sendHttpHeaders();
	conn.out.write(b);
    }

    public void write(byte b[]) throws IOException {
	conn.sendHttpHeaders();
	conn.out.write(new String(b));
    }

    public void write(byte b[], int off, int len) throws IOException {
	conn.sendHttpHeaders();
	conn.out.write(new String(b), off, len);
    }

    public void flush() throws IOException {
	conn.sendHttpHeaders();
	conn.out.flush();
    }

    public void close() throws IOException {
	conn.sendHttpHeaders();
	conn.out.close();
    }
}

// This class is the thread that handles all communication with a client
class JServConnection extends Thread
    implements HttpServletRequest, HttpServletResponse,
               ServletConfig, ServletContext
{
    protected Socket client;
    protected DataInputStream in;
    protected PrintWriter out;

    protected Properties properties;
    protected Hashtable servlets;

    protected String master_auth;

    // Servlet setup stuff

    protected Servlet servlet;
    protected String classname;

    protected String auth;

    protected JServInputStream servlet_in;
    protected JServOutputStream servlet_out;

    protected Properties initargs;

    // HTTP Stuff
    protected Hashtable headers_in;
    protected Hashtable headers_out;
    protected Hashtable env_vars;

    protected int status;
    protected String status_string;

    protected Vector log_messages;

    private Hashtable params;
    private boolean got_input;

    private boolean sent_header;

    // Initialize the streams and start the thread
    public JServConnection(Socket client_socket, Properties properties,
			   Hashtable servlets, String auth) {
        client = client_socket;
	this.properties = properties;
	this.servlets = servlets;
	master_auth = auth;

        try {
	    in = new DataInputStream(client.getInputStream());
	    out = new PrintWriter(client.getOutputStream());
        }
        catch (IOException e) {
            try { client.close(); } catch (IOException e2) { ; }
            System.err.println("Exception while getting socket streams: " + e);
            return;
        }
        this.start();
    }
    
    // Provide the service.
    public void run() {

	// Set up hash tables
	headers_in = new Hashtable();
	headers_out = new Hashtable();
	env_vars = new Hashtable();

	// Initalize log messages
	log_messages = new Vector();

	// Set up HTTP defaults
	status = SC_OK;
	status_string = null;

	sent_header = false;
	got_input = false;
       
	// Read in the data

        try {
	    // Read data, and check to make sure it read ok
	    boolean readok = readData();

	    if (!readok) {
		try {
		    sendError(SC_INTERNAL_SERVER_ERROR,
			      "Malformed data send to JServ");
		    out.flush();
		    client.close();
		} catch (IOException e) {;}
		return;
	    }

	    // Check the authorization string
	    if (auth == null || master_auth == null
		|| !auth.equals(master_auth)) {
		try {
		    sendError(SC_FORBIDDEN,
		        "Illegal access to JServ: incorrect auth string");
		    out.flush();
		    client.close();
		} catch (IOException e) {;}
		return;
	    }

        }
        catch (IOException e) {
	    try {client.close();} catch (IOException e2) {;}
	    return;
	}

	// Look for the servlet

	synchronized (servlets) {
	    servlet = (Servlet)servlets.get(classname);
	    
	    if (servlet == null) {
		// Load the servlet
		try {
		    servlet = (Servlet)Class.forName(classname).newInstance();
		} catch (ClassNotFoundException e) {
		    try {
			sendError(SC_NOT_FOUND, "Class " + classname +
				  " could not be found");
			out.flush();
			client.close();
		    } catch (IOException ioe) {;}
		    return;
		} catch (IllegalAccessException e) {
		    try {
			sendError(SC_INTERNAL_SERVER_ERROR,
				  "Caught IllegalAccessException when loading "
				  + classname);
			out.flush();
			client.close();
		    } catch (IOException ioe) {;}
		    return;
		} catch (InstantiationException e) {
		    try {
			sendError(SC_INTERNAL_SERVER_ERROR,
				  "Caught IllegalAccessException when loading "
				  + classname);
			out.flush();
			client.close();
		    } catch (IOException ioe) {;}
		    return;
		}
		
		// Setup the init parameters
	    
		String argsline = properties.getProperty(classname
							 + ".initArgs");
		initargs = parseInitArgs(argsline);
		
		// Init the servlet
		try {
		    servlet.init(this);
		} catch (ServletException se) {
		    try {
			if (se.getMessage() != null)
			    sendError(SC_INTERNAL_SERVER_ERROR,
				      "ServletException caught in " +
				      classname + ".init(): " +
				      se.getMessage());
			else
			    sendError(SC_INTERNAL_SERVER_ERROR,
				      "ServletException caught in " +
				      classname + ".init()");
			out.flush();
			client.close();
		    }
		    catch (IOException e) {;} 
		    return;
		}

		// Add the servlet to servlets, so we can use it again
		servlets.put(classname, servlet);
	    }
	}

	// Set up the servlet's I/O streams
	servlet_in = new JServInputStream(this);
	servlet_out = new JServOutputStream(this);

	// Start up the servlet
	try {
	    servlet.service(this, this);
	} catch (ServletException se) {
	    try {
		if (se.getMessage() != null)
		    sendError(SC_INTERNAL_SERVER_ERROR,
			      "ServletException caught in " + classname +
			      ".service():" + se.getMessage());
		else
		    sendError(SC_INTERNAL_SERVER_ERROR,
			      "ServletException caught in " + classname +
			      ".service");
	    } catch (IOException e) {;}
	} catch (IOException e) {;}

	// All done; close the connection
	try {
	    out.flush();
	    client.close();
	}
	catch (IOException e) {;} 
	return;
    }

    // Convert initArgs line into Properties
    // "<name=value>,<name=value>"
    private Properties parseInitArgs(String argsline) {
	Properties initargs = new Properties();
	if (argsline == null)
	    return initargs;

	StringTokenizer stc = new StringTokenizer(argsline, ",");

	try {
	    while (stc.hasMoreTokens()) {
		StringTokenizer ste =
		    new StringTokenizer(stc.nextToken(), "=");
		String key, val;
		
		key = ste.nextToken();
		val = ste.nextToken();
		
		initargs.put(key, val);
	    }
	} catch (NoSuchElementException e) {;}
	
	return initargs;
    }

    // Read in a line from the input

    private String readHexLine() throws IOException {
	Integer len;
	byte hex[] = new byte[4];
	byte str[];

	try {
	    // Read four bytes from the input stream
	    in.readFully(hex);

	    // Convert those four bytes from hex into an int
	    len = Integer.valueOf(new String(hex), 16);

	    // Read len bytes from the input stream
	    str = new byte[len.intValue()];
	    in.readFully(str);
	} catch (EOFException e) {
	    throw new IOException("Got EOFException");
	} catch (NumberFormatException e) {
	    return null;
	}
       
	// Return str as an String
	return new String(str);
    }

    // Read in all the data

    private boolean readData() throws IOException{
	for (;;) {
	    String line = readHexLine();

	    // Oops..
	    if (line == null)
		return false;

	    // Alldone
	    if (line.length() == 0)
		return true;

	    // Get the identifier from the first character
	    char id = line.charAt(0);
	    line = line.substring(1);

	    // All ids want two peices of data, seperated by a tab.

	    StringTokenizer tokens = new StringTokenizer(line, "\t");
	    String token1, token2;

	    try {
		token1 = tokens.nextToken();
	    } catch (NoSuchElementException e) {
		return false;
	    }

	    try {
		token2 = tokens.nextToken();
	    } catch (NoSuchElementException e) {
		token2 = new String("");
	    }

	    // Switch, depending on what the id is
	    switch (id) {
	    case 'A':	// Auth data
		auth = token1;
		classname = token2;
		break;
	    case 'E':	// Env variable
		env_vars.put(token1, token2);
		break;
	    case 'H':	// Header
		headers_in.put(token1.toLowerCase(), token2);
		break;
	    default:	// What the heck is this?
		return false;
	    }
	}
    }

    // Find a status string from a HTTP status code

    private String findStatusString(int sc) {
	switch (sc) {
	case SC_ACCEPTED:
	    return "Accepted";
	case SC_BAD_GATEWAY:
	    return "Bad Gateway";
	case SC_BAD_REQUEST:
	    return "Bad Request";
	case SC_CREATED:
	    return "Created";
	case SC_FORBIDDEN:
	    return "Forbidden";
	case SC_INTERNAL_SERVER_ERROR:
	    return "Internal Server Error";
	case SC_MOVED_PERMANENTLY:
	    return "Moved Permanently";
	case SC_MOVED_TEMPORARILY:
	    return "Moved Temporarily";
	case SC_NO_CONTENT:
	    return "No Content";
	case SC_NOT_FOUND:
	    return "Not Found";
	case SC_NOT_IMPLEMENTED:
	    return "Method Not Implemented";
	case SC_NOT_MODIFIED:
	    return "Not Modified";
	case SC_OK:
	    return "OK";
	case SC_SERVICE_UNAVAILABLE:
	    return "Service Temporarily Unavailable";
	case SC_UNAUTHORIZED:
	    return "Authorization Required";
	default:
	    return "Response";
	}
    }

    // Send the HTTP headers, prepare to send response

    protected void sendHttpHeaders() {
	if (sent_header)
	    return;

	sent_header = true;

	// Send logging data

	for (Enumeration e = log_messages.elements(); e.hasMoreElements(); ) {
	    out.print("Servlet-Log: " + e.nextElement() + "\r\n");
	}
	
	// Send the status info
	Integer status_code = new Integer(status);

	if (status_string == null) {
	    status_string = findStatusString(status);
	}

	out.print("Status: " + status_code.toString() + " " +
		  status_string + "\r\n");

       	// Send the headers
	for (Enumeration e = headers_out.keys(); e.hasMoreElements(); ) {
	    String key = (String)e.nextElement();

	    out.print(key + ": " + headers_out.get(key) + "\r\n");
	}

	// Send a terminating blank line
	out.print("\r\n");
    }

    // Methods from ServletConfig

    public ServletContext getServletContext() {
	return this;
    }

    public String getInitParameter(String name) {
	return initargs.getProperty(name);
    }

    public Enumeration getInitParameterNames() {
	return initargs.propertyNames();
    }

    // Methods from ServletContext

    public Servlet getServlet(String name) throws ServletException {
	synchronized (servlets) {
	    return (Servlet)servlets.get(name);
	}
    }

    public Enumeration getServlets() {
	synchronized (servlets) {
	    return servlets.keys();
	}
    }

    public void log(String msg) {
	log_messages.addElement(msg);
    }

    public String getRealPath(String path) {
	// FIXME: Make this somehow talk to Apache, do a subrequest
	// and get the real filename. Until then, we just tack the path onto
	// the doc root and hope it's right. *sigh*
	
	return new String((Servlet)env_vars.get("DOCUMENT_ROOT") + path);
    }

    public String getMimeType(String file) {
	// FIXME: See above. This should talk to Apache, do a subrequest
	// and return the content type. But we can't do that right now,
	// so we test a few common types, and return them. *sigh*

	if (file.endsWith(".html") || file.endsWith(".htm"))
	    return "text/html";
	if (file.endsWith(".txt"))
	    return "text/plain";
	if (file.endsWith(".gif"))
	    return "image/gif";
	if (file.endsWith(".jpg") || file.endsWith(".jpeg"))
	    return "image/jpeg";
	if (file.endsWith(".class"))
	    return "application/octet-stream";

	// Return plain text as the default
	return "text/plain";
    }

    public String getServerInfo() {
	return (String)env_vars.get("SERVER_SOFTWARE");
    }

    public Object getAttribute(String name) {
	// We don't have any attributes, either for the request
	// or the network service.
	return null;
    }

    // Methods from HttpServletRequest + ServletRequest

    public int getContentLength() {
	String lenstr = (String)env_vars.get("CONTENT_LENGTH");
	Integer len;

	if (lenstr == null)
	    return -1;

	try {
	    len = new Integer(lenstr);
	} catch (NumberFormatException e) {
	    return -1;
	}
	
	return len.intValue();
    }

    public String getContentType() {
	return (String)env_vars.get("CONTENT_TYPE");
    }

    public String getProtocol() {
	return (String)env_vars.get("SERVER_PROTOCOL");
    }

    public String getScheme() {
	// FIXME: We just don't have this information available. We'll
	// look at the port, and return https if it's 443, but that's
	// not a real solution.
	if (getServerPort() == 443)
	    return "https";

	return "http";
    }

    public String getServerName() {
	return (String)env_vars.get("SERVER_NAME");
    }

    public int getServerPort() {
	String portstr = (String)env_vars.get("SERVER_PORT");
	Integer port;

	if (portstr == null)
	    return -1;

	try {
	    port = new Integer(portstr);
	} catch (NumberFormatException e) {
	    return -1;
	}

	return port.intValue();
    }

    public String getRemoteAddr() {
	return (String)env_vars.get("REMOTE_ADDR");
    }

    public String getRemoteHost() {
	return (String)env_vars.get("REMOTE_HOST");
    }

    // getRealPath() defined in ServletContext

    public ServletInputStream getInputStream() throws IOException {
	got_input = true;
	return servlet_in;
    }

    // Parameter stuff

    private boolean parseParams() {
	// Have we already done it?
	if (params != null)
	    return false;

	// Hmm... must have given the stream through getInputStream()
	if (got_input)
	    return true;

	got_input = true;

	// If we have a content length, we have an entity, otherwise
	// we should read from the query string
	try {
	    HttpUtils parser = new HttpUtils();

	    if (getContentLength() > 0) {
		params = parser.parsePostData(getContentLength(), servlet_in);
	    }
	    else {
		params = parser.parseQueryString(getQueryString());
	    }
	} catch (IllegalArgumentException e) {
	    return true;
	}

	return false;
    }

    public String getParameter(String name) {
	if (parseParams()) return null;

	Object val = params.get(name);

	if (val == null)
	    return null;

	if (val instanceof String[]) {
	    // It's an array, return the first element
	    return ((String[])val)[0];
	}

	return (String)val;
    }

    public String[] getParameterValues(String name) {
	if (parseParams()) return null;
	
	Object val = params.get(name);

	if (val == null)
	    return null;

	if (val instanceof String) {
	    // It's a string, convert to an array and return
	    String va[] = { (String)val };
	    return va;
	}

	return (String[])val;
    }

    public Enumeration getParameterNames() {
	// an "empty Enumeration"... *sigh*
	if (parseParams()) return (new Vector()).elements();
	return params.keys();
    }

    // getAttribute() defined in ServletContext

    public String getMethod() {
	return (String)env_vars.get("REQUEST_METHOD");
    }

    public String getRequestURI() {
	if (getPathInfo() != null)
	    return getServletPath() + getPathInfo();
	else
	    return getServletPath();
    }

    public String getServletPath() {
	return (String)env_vars.get("SCRIPT_NAME");
    }

    public String getPathInfo() {
	return (String)env_vars.get("PATH_INFO");
    }

    public String getPathTranslated() {
	return (String)env_vars.get("PATH_TRANSLATED");
    }

    public String getQueryString() {
	return (String)env_vars.get("QUERY_STRING");
    }

    public String getRemoteUser() {
	return (String)env_vars.get("REMOTE_USER");
    }

    public String getAuthType() {
	return (String)env_vars.get("AUTH_TYPE");
    }

    public String getHeader(String name) {
	return (String)headers_in.get(name.toLowerCase());
    }

    public int getIntHeader(String name) {
	String hdrstr = (String)headers_in.get(name.toLowerCase());
	Integer hdr;

	if (hdrstr == null)
	    return -1;

	try {
	    hdr = new Integer(hdrstr);
	} catch (NumberFormatException e) {
	    return -1;
	}
	
	return hdr.intValue();
    }

    public long getDateHeader(String name) {
	SimpleDateFormat sdf =
	    new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
	String val = (String)headers_in.get(name.toLowerCase());

	if (val == null)
	    return -1;

	try {
	    Date date = sdf.parse(val);
	    return date.getTime();
	} catch (ParseException e) {
	    return -1;
	}
    }

    public Enumeration getHeaderNames() {
	return headers_in.keys();
    }
    
    // Methods from HttpServletResponse + HttpServletResponse

    public void setContentLength(int len) {
	Integer length = new Integer(len);

	headers_out.put("Content-Length", length.toString());
    }

    public void setContentType(String type) {
	headers_out.put("Content-Type", type);
    }

    public ServletOutputStream getOutputStream() throws IOException {
	return servlet_out;
    }

    public boolean containsHeader(String name) {
	return headers_out.contains(name.toLowerCase());
    }

    public void setStatus(int sc, String sm) {
	status = sc;
	status_string = sm;
    }
   
    public void setStatus(int sc) {
	status = sc;
	status_string = null;
    }

    public void setHeader(String name, String value) {
	headers_out.put(name, value);
    }

    public void setIntHeader(String name, int value) {
	Integer val = new Integer(value);

	headers_out.put(name, val.toString());
    }

    public void setDateHeader(String name, long date) {
	SimpleDateFormat sdf =
	    new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
	TimeZone tz = TimeZone.getTimeZone("GMT");

	sdf.setTimeZone(tz);

	headers_out.put(name, sdf.format(new Date(date)));
    }

    public void sendError(int sc, String msg) throws IOException {
	// Tell Apache to send an error
	status = sc;
	setHeader("Servlet-Error", msg);
	sendHttpHeaders();
    }

    public void sendError(int sc) throws IOException {
	sendError(sc, "Servlet sent error");
    }

    public void sendRedirect(String location) throws IOException {
	// We use Apache's internal status mechanism: set the status to
	// 200, with a Location: header and no body.
	
	setStatus(SC_OK);
	setHeader("Location", location);
	sendHttpHeaders();
    }
}
