 /* ====================================================================
 * Copyright (c) 1997 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/*
 * mod_jserv.c: Implements part of Java Server API for Apache
 * by Alexei Kosut <akosut@apache.org>
 * parts based on mod_alias.c and mod_cgi.c
 */

#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"

/* Name of the class to run - not configurable */
#define JSERV_CLASS "apache.jserv.JServHandler"

/* Some defaults */
#define JSERV_DEFAULT_PORT 8007
#define JSERV_DEFAULT_PROPERTIES "conf/jserv.properties"

typedef struct {
    char *real;
    char *fake;
    int is_dir;
} jserv_entry;

typedef struct {
    array_header *dirs;
    char *classpath;
    char *binary;
    char *properties;
    int port;

    char auth[10];
} jserv_conf;

module jserv_module;

void *create_jserv_config (pool *p, server_rec *s)
{
    jserv_conf *j = (jserv_conf *)pcalloc (p, sizeof(jserv_conf));

    j->dirs = make_array (p, 20, sizeof(jserv_entry));
    j->classpath = NULL;
    j->binary = NULL;
    j->properties = NULL;
    j->port = JSERV_DEFAULT_PORT;

    /* Create an auth string */
    srand(time(NULL) | getpid());
    ap_snprintf(j->auth, sizeof(j->auth), "%x", rand());

    return j;
}

void *merge_jserv_config (pool *p, void *basev, void *overridesv)
{
    jserv_conf *j = (jserv_conf *)pcalloc (p, sizeof(jserv_conf));
    jserv_conf *base = (jserv_conf *)basev,
	*overrides = (jserv_conf *)overridesv;

    j->dirs = append_arrays (p, overrides->dirs, base->dirs);

    strcpy(j->auth, base->auth);

    /* This is odd, because we want the base server to end up with all the
     * data, because it launches the Java server, not the virtual server.
     */

    if (base->classpath && overrides->classpath) {
	base->classpath =
	    pstrcat(p, overrides->classpath, ":", base->classpath, NULL);
    } else {
	base->classpath =
	    overrides->classpath ? overrides->classpath : base->classpath;
    }

    return j;
}

const char *add_jserv_dir(cmd_parms *cmd, void *dummy, char *f, char *r)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
	(jserv_conf *)get_module_config(s->module_config,&jserv_module);
    jserv_entry *new = push_array (conf->dirs);
    struct stat finfo;

    if (stat(r, &finfo) == -1)
	return pstrcat(cmd->pool, "\"", r, "\" does not exist", NULL);

    new->fake = f;
    new->real = r;
    new->is_dir = S_ISDIR(finfo.st_mode);

    /* Add the dir to the classpath */
    if (conf->classpath)
	conf->classpath = pstrcat(cmd->pool, r, ":", conf->classpath, NULL);
    else
	conf->classpath = r;

    return NULL;
}

const char *set_jserv_classpath(cmd_parms *cmd, void *dummy, char *path) {
    server_rec *s = cmd->server;
    jserv_conf *conf =
	(jserv_conf *)get_module_config(s->module_config,&jserv_module);

    if (conf->classpath)
	conf->classpath = pstrcat(cmd->pool, path, ":", conf->classpath, NULL);
    else
	conf->classpath = path;

    return NULL;
}

const char *set_jserv_bin (cmd_parms *cmd, void *dummy, char *bin) {
    server_rec *s = cmd->server;
    jserv_conf *conf =
	(jserv_conf *)get_module_config(s->module_config,&jserv_module);

    conf->binary = bin;

    return NULL;
}

const char *set_jserv_properties (cmd_parms *cmd, void *dummy, char *props) {
    server_rec *s = cmd->server;
    jserv_conf *conf =
	(jserv_conf *)get_module_config(s->module_config,&jserv_module);

    conf->properties = server_root_relative(cmd->pool, props);

    return NULL;
}

const char *set_jserv_port (cmd_parms *cmd, void *dummy, char *port) {
    server_rec *s = cmd->server;
    jserv_conf *conf =
	(jserv_conf *)get_module_config(s->module_config,&jserv_module);

    conf->port = atoi(port);

    return NULL;
}

command_rec jserv_cmds[] = {
{ "ServletAlias", add_jserv_dir, NULL, RSRC_CONF, TAKE2, 
  "a URL path prefix and a directory which stores class files" },
{ "ServletClassPath", set_jserv_classpath, NULL, RSRC_CONF, TAKE1,
  "a Java class path to search" },
{ "ServletBinary", set_jserv_bin, NULL, RSRC_CONF, TAKE1,
  "The path to the Java runtime binary" },
{ "ServletProperties", set_jserv_properties, NULL, RSRC_CONF, TAKE1,
  "the filename that holds the Java servlet properties" },
{ "ServletPort", set_jserv_port, NULL, RSRC_CONF, TAKE1,
  "the port to run the Java servlet server on" },
{ NULL }
};

typedef struct {
    jserv_conf *conf;
    server_rec *s;
    pool *p;
} jserv_child_conf;

static int jserv_child (void *data) {
    jserv_conf *conf = ((jserv_child_conf *)data)->conf;
    server_rec *s = ((jserv_child_conf *)data)->s;
    pool *p = ((jserv_child_conf *)data)->p;

    char port[10];
    char *env[4];

    ap_snprintf(port, sizeof(port), "%d", conf->port);
    
    env[0] = pstrcat(p, "CLASSPATH=", conf->classpath, NULL);
    env[1] = pstrcat(p, "PATH=", getenv("PATH"), NULL);
    env[2] = pstrcat(p, "TZ=", getenv("TZ"), NULL);
    env[3] = NULL;

    /* Change uid to the server's User, if we're running as root */
    if (!geteuid() && setuid(s->server_uid) == -1) {
	log_unixerr("setuid", NULL, "unable to change uid for JServ", s);
	exit(1);
    }

    error_log2stderr(s);
    cleanup_for_exec();

    execle(conf->binary, conf->binary, JSERV_CLASS, conf->properties,
	   port, NULL, env);

    return 1;
}

void init_jserv (server_rec *s, pool *p)
{
    jserv_conf *conf =
	(jserv_conf *)get_module_config(s->module_config,&jserv_module);
    jserv_entry *entries = (jserv_entry *)conf->dirs->elts;
    jserv_child_conf *jcc;
    FILE *jserv_in;
    int i;

    /* Only for the main server */
    if (s->is_virtual)
	return;

    /* Are we on? */
    if (!conf->binary || !conf->classpath)
	return;

    if (!conf->properties)
	conf->properties = server_root_relative(p, JSERV_DEFAULT_PROPERTIES);

    /* Setup child data */
    jcc = pcalloc(p, sizeof(jserv_child_conf));
    jcc->conf = conf;
    jcc->s = s;
    jcc->p = p;

    /* Spawn child */
    if (!(spawn_child(p, jserv_child, (void *)jcc, kill_always, &jserv_in,
		      NULL))) {
	fprintf(stderr, "httpd: could not spawn JServ child process");
	exit(1);
    }

    /* Send the auth data to JServ */
    fputs(conf->auth, jserv_in);
    fputs("\n", jserv_in);

    fflush(jserv_in);
    pfclose(p, jserv_in);
}

static int jserv_alias_matches (char *uri, char *alias_fakename)
{
    char *end_fakename = alias_fakename + strlen (alias_fakename);
    char *aliasp = alias_fakename, *urip = uri;

    while (aliasp < end_fakename) {
	if (*aliasp == '/') {
	    /* any number of '/' in the alias matches any number in
	     * the supplied URI, but there must be at least one...
	     */
	    if (*urip != '/') return 0;
	    
	    while (*aliasp == '/') ++ aliasp;
	    while (*urip == '/') ++ urip;
	}
	else {
	    /* Other characters are compared literally */
	    if (*urip++ != *aliasp++) return 0;
	}
    }

    /* Check last alias path component matched all the way */

    if (aliasp[-1] != '/' && *urip != '\0' && *urip != '/')
	return 0;

    /* Return number of characters from URI which matched (may be
     * greater than length of alias, since we may have matched
     * doubled slashes)
     */

    return urip - uri;
}

static char *try_jserv_list (request_rec *r, array_header *dirs)
{
    jserv_entry *entries = (jserv_entry *)dirs->elts;
    char *found = NULL;
    int i;
    
    for (i = 0; i < dirs->nelts; ++i) {
        jserv_entry *e = &entries[i];
	int l = jserv_alias_matches(r->uri, e->fake);

	if (l > 0) {
	    /* Note some info for later use */
	    table_set(r->notes, "jserv-dirname", e->real);
	    if (e->is_dir) table_set(r->notes, "jserv-isdir", "true");
	    r->handler = pstrdup(r->pool, "jserv-servlet");

	    return pstrcat(r->pool, e->real, r->uri + l, NULL);
	}
    }

    return NULL;
}

int translate_jserv(request_rec *r)
{
    void *sc = r->server->module_config;
    jserv_conf *conf = (jserv_conf *)get_module_config(sc, &jserv_module);
    char *ret;

#if defined(__EMX__) || defined(WIN32)
    /* Add support for OS/2 drive names */
    if ((r->uri[0] != '/' && r->uri[0] != '\0') && r->uri[1] != ':')
#else    
    if (r->uri[0] != '/' && r->uri[0] != '\0') 
#endif    
        return DECLINED;
    
    if ((ret = try_jserv_list (r, conf->dirs))) {
        r->filename = ret;
        return OK;
    }

    return DECLINED;
}

/* Send a line in the format JServ expects; four hex characters indicating
 * length, then the string.
 */

static void send_hexline(FILE *f, const char *line) {
    fprintf(f, "%04lx%s", strlen(line), line);
}

int jserv_handler(request_rec *r) {
    void *s = r->server->module_config;
    jserv_conf *conf = (jserv_conf *)get_module_config(s, &jserv_module);
    char *p, *classname, *dirname = table_get(r->notes, "jserv-dirname");
    char *hdr, *is_dir = table_get(r->notes, "jserv-isdir");
    int ret;

    struct sockaddr_in addr;
    FILE *j;
    int i, sock;

    /* Make sure we're set up correctly */
    if (!conf->binary) {
	log_reason("ServletBinary not configured", r->uri, r);
	return SERVER_ERROR;
    }
    if (!conf->classpath) {
	log_reason("ServletClassPath not configured", r->uri, r);
	return SERVER_ERROR;
    }

    /* Make sure we have a dir name - we must be called with
     * a matching ServletAlias directive, found when translating.
     */
    if (!dirname) {
	log_reason("Servlet class name unknown (illegal access?)", r->uri, r);
	return FORBIDDEN;
    }

    /* If it's a directory, then this is simple. If not, this is more
     * complex, since we need to identify the class name from the path info
     */
    
    if (is_dir) {
	classname = pstrdup(r->pool, r->filename + strlen(dirname) + 1);

	/* Translate slash (/) to period (.), so you can use physical
	 * names to identify classes.
	 */
	for (p = classname; *p; p++) {
	    if (*p == '/') *p = '.';
	}
    }
    else if (!r->path_info) {
	/* We should have some path info... if not, it means we didn't
	 * get a class to match!
	 */
return NOT_FOUND;
    }
    else {
	char *path = r->path_info;
	char *slash;

	/* If there's a leading /, strip it off */
	if (path[0] == '/')
	    path++;

	slash = strchr(path, '/');

	if (!slash) {
	    /* No extra path info */
	    classname = pstrdup(r->pool, path);
	    r->path_info = "";
	}
	else {
	    classname = pstrndup(r->pool, r->path_info, slash - r->path_info);
	    r->path_info = slash;
	}
    }
    
    /* Make sure we have a class name. */

    if (!classname || !*classname) {
	log_reason("Class name could not be identified", r->uri, r);
	return NOT_FOUND;
    }

    /* Open the connection to the server */

    addr.sin_addr.s_addr = inet_addr("127.0.0.1");
    addr.sin_port = htons(conf->port);

    if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
	log_reason("Could not create JServ socket", r->uri, r);
	return SERVER_ERROR;
    }
    note_cleanups_for_fd(r->pool, sock);

    hard_timeout("proxy connect", r);

    do {
	i = connect(sock, (struct sockaddr *)&addr,
		    sizeof(struct sockaddr_in));
#ifdef WIN32
        if(i == SOCKET_ERROR)
            errno = WSAGetLastError() - WSABASEERR;
#endif /* WIN32 */
    }
    while (i == -1 && errno == EINTR);
    if (i == -1) {
	log_reason("JServ connect failed", r->uri, r);
	return SERVER_ERROR;
    }

    kill_timeout(r);
    
    j = pfdopen(r->pool, sock, "rb+");

    /* Send the data */

    add_cgi_vars(r);
    add_common_vars(r);

    if ((ret = setup_client_block(r, REQUEST_CHUNKED_ERROR)))
	return ret;

    hard_timeout("jserv send", r);

    /* We don't yet have auth code, so send "auth" */
    send_hexline(j, pstrcat(r->pool, "A", conf->auth, "\t", classname, NULL));

    /* Send environment vars, except for the header ones */
    if (r->subprocess_env) {
	array_header *env_arr = table_elts(r->subprocess_env);
	table_entry *elts = (table_entry *)env_arr->elts;

	for (i = 0; i < env_arr->nelts; ++i) {
	    if (!elts[i].key) continue;
	    if (!strncmp(elts[i].key, "HTTP_", 5)) continue;
	    send_hexline(j, pstrcat(r->pool, "E", elts[i].key, "\t",
				    elts[i].val, NULL));
	}
    }

    /* Send the request headers */

    if (r->headers_in) {
	array_header *hdr_arr = table_elts(r->headers_in);
	table_entry *elts = (table_entry *)hdr_arr->elts;

	for (i = 0; i < hdr_arr->nelts; ++i) {
	    if (!elts[i].key) continue;
	    send_hexline(j, pstrcat(r->pool, "H", elts[i].key, "\t",
				    elts[i].val, NULL));
	}
    }

    /* Send the terminating entry */
    send_hexline(j, "");

    /* If there is a request entity, send it */
    
    if (should_client_block(r)) {
	long len_read;
	char buffer[HUGE_STRING_LEN];

	while ((len_read = get_client_block(r, buffer, HUGE_STRING_LEN)) > 0) {
	    reset_timeout(r);
	    if (fwrite(buffer, sizeof(char), len_read, j) < len_read) {
		/* Something happened, soak up the rest */
		while (get_client_block(r, buffer, HUGE_STRING_LEN) > 0);
		break;
	    }
	}
    }

    fflush(j);
    kill_timeout(r);

    /* Receive the response */

    hard_timeout("jserv receive", r);

    if ((ret = scan_script_header(r, j))) {
	kill_timeout(r);
	return ret;
    }

    /* Check for our special headers */

    if ((hdr = table_get(r->err_headers_out, "Servlet-Log"))) {
	table_unset(r->err_headers_out, "Servlet-Log");
	log_error(hdr, r->server);
    }

    if ((hdr = table_get(r->err_headers_out, "Servlet-Error"))) {
	int status = r->status;

	table_unset(r->err_headers_out, "Servlet-Error");
	log_reason(hdr, r->uri, r);

	/* Zap the existing status, or Apache will do odd things */
	r->status = HTTP_OK;
	r->status_line = NULL;

	return status;
    }
    
    /* Do CGI-style Location-handling */

    if (table_get(r->headers_out, "Location") && r->status == 200) {
	kill_timeout(r);

	r->status = HTTP_OK;
	r->status_line = NULL;

	return REDIRECT;
    }
    
    send_http_header(r);
    if (!r->header_only)
	send_fd(j, r);

    kill_timeout(r);
    pfclose(r->pool, j);

    return OK;
}

handler_rec jserv_handlers[] = {
    { "jserv-servlet", jserv_handler },
    { NULL }
};

module jserv_module = {
   STANDARD_MODULE_STUFF,
   init_jserv,			/* initializer */
   NULL,			/* dir config creater */
   NULL,			/* dir merger --- default is to override */
   create_jserv_config,		/* server config */
   merge_jserv_config,		/* merge server configs */
   jserv_cmds,			/* command table */
   jserv_handlers,	       	/* handlers */
   translate_jserv,		/* filename translation */
   NULL,			/* check_user_id */
   NULL,			/* check auth */
   NULL,			/* check access */
   NULL,			/* type_checker */
   NULL,			/* fixups */
   NULL,			/* logger */
   NULL				/* header parser */
};
