/*
 * Copyright (c) 1997-1998 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */

/*****************************************************************************
 * Description: mod_jserv.c is used by Apache to talk to Apache JServ.       *
 *              It provides a common entry point for protocols and runs      * 
 *              configuration, initialization and request tasks.             *
 * Author:      Pierpaolo Fumagalli <ianosh@iname.com>                       *
 * Version:     $Revision: 1.23 $                                                 *
 *****************************************************************************/
#include "jserv.h"

/*****************************************************************************
 * Shared variables                                                          *
 *****************************************************************************/
jserv_config *jserv_servers=NULL;
pool *jserv_pool=NULL;

/*****************************************************************************
 * Configuration Procedures                                                  *
 *****************************************************************************/

/* ========================================================================= */
/* Check our mount default config */
static void jserv_mount_config_default(pool *p, jserv_config *cfg) {
    jserv_mount *cur=cfg->mount;

    while (cur!=NULL) {
        /* Check mount point */
        if (cur->mountpoint==NULL) jserv_error_exit(JSERV_LOG_EMERG,cfg,
                        "Mountpoint not defined in mount structure");
        /* Update configuration pointer */
        cur->config=cfg;

        /* Check defaults */
        if (cur->protocol==NULL) cur->protocol=cfg->protocol;
        if (cur->host==NULL) {
            cur->host=cfg->host;
            cur->hostaddr=cfg->hostaddr;
        }
        if (cur->port==JSERV_DEFAULT) cur->port=cfg->port;
        if (cur->secretfile==NULL) {
            cur->secretfile=cfg->secretfile;
            cur->secret=cfg->secret;
            cur->secretsize=cfg->secretsize;
        }
        cur=cur->next;
    }
}

/* ========================================================================= */
/* Check our server default config */
static void jserv_server_config_default(pool *p, jserv_config *cfg) {

    /* Check ApJServManual */
    if (cfg->manual==JSERV_DEFAULT) cfg->manual=JSERV_DEFAULT_MANUAL;

    /* Check ApJServProperties */
    if (cfg->properties==NULL)
        cfg->properties=ap_pstrdup(p,JSERV_DEFAULT_PROPERTIES);

    /* Check ApJServDefaultProtocol */
    if (cfg->protocol==NULL) {
        cfg->protocol=jserv_protocol_getbyname(JSERV_DEFAULT_PROTOCOL);
        if (cfg->protocol==NULL) jserv_error_exit(JSERV_LOG_EMERG,cfg,
            "Cannot find ApJServDefaultProtocol %d", JSERV_DEFAULT_PROTOCOL);
    }

    /* Check ApJServDefaultPort */
    if (cfg->port==JSERV_DEFAULT) cfg->port=cfg->protocol->port;

    /* Check ApJServDefaultHost */
    if (cfg->host==NULL) {
        cfg->host=ap_pstrdup(p,JSERV_DEFAULT_HOST);
        cfg->hostaddr=JSERV_DEFAULT;
    } 

    /* Check address for ApJServDefaultHost */
    if (cfg->hostaddr==JSERV_DEFAULT) cfg->hostaddr=jserv_resolve(cfg->host);
    if (cfg->hostaddr==0) {
        jserv_error_exit(JSERV_LOG_EMERG,cfg,
            "Error setting defaults: ApJServDefaultHost name \"%s\" can't be resolved",
            cfg->host);
    }

    /* Check ApJServMountCopy */
    if (cfg->mountcopy==JSERV_DEFAULT) cfg->mountcopy=JSERV_DEFAULT_MOUNTCOPY;

    /* Check ApJServLogFile */
    if (cfg->logfile==NULL) {
        cfg->logfile=ap_pstrdup(p,JSERV_DEFAULT_LOGFILE);
        cfg->logfilefd=JSERV_DEFAULT;
    }

    /* Check file descriptor for ApJServLogFile */
    if (cfg->logfilefd==JSERV_DEFAULT) {
        const char *buf=jserv_openfile(p, cfg->logfile, JSERV_TRUE, 
                    &cfg->logfilefd, JSERV_LOGFILE_FLAGS, JSERV_LOGFILE_MODE);
        if (buf!=NULL) jserv_error_exit(JSERV_LOG_EMERG,cfg,
                            "Error setting defaults: ApJServLogFile: %s", buf);
    }

    /* Fill our ApJServMount structures */ 
    jserv_mount_config_default(p, cfg);
}


/* ========================================================================= */
/* Create our server config */
static void *jserv_server_config_create(pool *p, server_rec *s) {
    jserv_config *cfg=(jserv_config *)ap_pcalloc(p, sizeof(jserv_config));

    /* Our server entry */
    cfg->server=s;

    /* DEFAULT values setup */
    cfg->manual=JSERV_DEFAULT;
    cfg->properties=NULL;
    cfg->protocol=NULL;
    cfg->host=NULL;
    cfg->hostaddr=JSERV_DEFAULT;
    cfg->port=JSERV_DEFAULT;
    cfg->mount=NULL;
    cfg->mountcopy=JSERV_DEFAULT;
    cfg->logfile=NULL;
    cfg->logfilefd=JSERV_DEFAULT;
    cfg->secretfile=NULL;
    cfg->secret=NULL;
    cfg->secretsize=JSERV_DEFAULT;
    cfg->actions=ap_make_table(p,5);


    /* Add our server to server chain (if it's not virtual) */
    if (!(s->is_virtual)) {
        cfg->next=NULL;
        jserv_servers=cfg;
    }

    /* We created the server config */
    return(cfg);
}

/* ========================================================================= */
/* Merge two different server configs */
static void *jserv_server_config_merge(pool *p, void *vbase, void *voverride) {
    jserv_config *cfg=(jserv_config *)ap_pcalloc(p, sizeof(jserv_config));
    jserv_config *base=(jserv_config *) vbase;
    jserv_config *override=(jserv_config *) voverride;
    int copy;

    /* Setup base server defaults */
    jserv_server_config_default(p, base);

    /* Our server entry */
    cfg->server=override->server;

    /* Things not handled inside VIRTUALHOST */
    cfg->manual=base->manual;
    cfg->properties=base->properties;

    /* Configuration duplication */
    cfg->protocol=override->protocol?
                  override->protocol:
                  base->protocol;
    cfg->port=(override->port!=JSERV_DEFAULT)?
               override->port:
               cfg->protocol->port;
    cfg->mountcopy=(override->mountcopy!=JSERV_DEFAULT)?
                    override->mountcopy:
                    base->mountcopy;

    /* ApJServDefaultHost merging */
    if (override->host!=NULL) {
        cfg->host=override->host;
        cfg->hostaddr=override->hostaddr;
    } else {
        cfg->host=base->host;
        cfg->hostaddr=base->hostaddr;
    }

    /* ApJServLogFile merging */
    if (override->logfile!=NULL) {
        cfg->logfile=override->logfile;
        cfg->logfile=override->logfile;
    } else {
        cfg->logfile=base->logfile;
        cfg->logfilefd=base->logfilefd;
    }

    /* ApJServSecretKey merging */
    if (override->secretfile!=NULL) {
        cfg->secretfile=override->secretfile;
        cfg->secret=override->secret;
        cfg->secretsize=override->secretsize;
    } else {
        cfg->secretfile=base->secretfile;
        cfg->secret=base->secret;
        cfg->secretsize=base->secretsize;
    }

    /* Check whether to copy or not base mounts */
    copy=JSERV_FALSE;
    if (override->mountcopy==JSERV_FALSE) copy=JSERV_FALSE;
    if (override->mountcopy==JSERV_TRUE) copy=JSERV_TRUE;
    if (override->mountcopy==JSERV_DEFAULT) {
        if (base->mountcopy==JSERV_FALSE) copy=JSERV_FALSE;
        if (base->mountcopy==JSERV_TRUE) copy=JSERV_TRUE;
    }
    /* Copy mounts if nedeed */
    cfg->mount=override->mount;
    if (copy==JSERV_TRUE) {
        if (cfg->mount==NULL) cfg->mount=base->mount;
        else {
            jserv_mount *cur=cfg->mount;

            while (cur->next!=NULL) cur=cur->next;
            cur->next=base->mount;
        }
    }

    /* Fill mount defaults */
    jserv_mount_config_default(p, cfg);

    /* Merge action tables */
    cfg->actions=ap_overlay_tables(p, override->actions, base->actions);

    /* Add our server to server chain */
    if (jserv_servers==NULL) {
        /* Should never happen but I want to be sure */
        cfg->next=NULL;
        jserv_servers=cfg;
    } else {
        jserv_config *cur=jserv_servers;

        /* Append configuration to list */
        while(cur->next!=NULL) cur=cur->next;
        cur->next=cfg;
        cfg->next=NULL;
    }

    /* All done */
    return (cfg);
}

/* ========================================================================= */
/* Handle JServManual directive (FLAG) */
static const char *jserv_cfg_manual(cmd_parms *cmd, void *dummy, int flag) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);

    /* Check if we already processed JServManual directives */
    if (cfg->manual!=JSERV_DEFAULT)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Check if we specified this in a virtual server */
    if (cfg->server!=NULL)
        if (cfg->server->is_virtual)
            return ap_pstrcat(cmd->pool, cmd->cmd->name,
                              ": cannot be specified inside <VirtualHost>",
                              NULL);

    /* Set up our value */
    cfg->manual=flag?JSERV_TRUE:JSERV_FALSE;

    return NULL;
}

/* ========================================================================= */
/* Handle JServProperties directive (TAKE1) */
static const char *jserv_cfg_properties(cmd_parms *cmd, void *dummy,
                                        char *value) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);

    /* Check if we already processed JServProperties directives */
    if (cfg->properties!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Check if we specified this in a virtual server */
    if (cfg->server!=NULL)
        if (cfg->server->is_virtual)
            return ap_pstrcat(cmd->pool, cmd->cmd->name,
                              ": cannot be specified inside <VirtualHost>",
                              NULL);

    /* Set up our value */
    cfg->properties=ap_server_root_relative(cmd->pool,value);
    return NULL;
}

/* ========================================================================= */
/* Handle JServDefaultProtocol directive (TAKE1) */
static const char *jserv_cfg_protocol(cmd_parms *cmd, void *dummy, 
                                      char *value) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);

    /* Check we did not provided status or wrapper protocols */
    if ((strcasecmp(value,"status")==0) | (strcasecmp(value,"wrapper")==0))
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                      ": protocol name cannot be '",value,"'",NULL);

    /* Check if we already processed ApJServProperties directives */
    if (cfg->protocol!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Set up our value */
    cfg->protocol=jserv_protocol_getbyname(value);
    if (cfg->protocol==NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot find protocol '", value, "'", NULL);
    else return NULL;
}

/* ========================================================================= */
/* Handle ApJServDefaultHost directive (TAKE1) */
static const char *jserv_cfg_host(cmd_parms *cmd, void *dummy, char *value) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);
    unsigned long address;

    /* Check if we already processed ApJServDefaultHost directives */
    if (cfg->host!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Copy our value in host */
    cfg->host=ap_pstrdup(cmd->pool,value);

    /* Resolve current host address */
    address=jserv_resolve(value);
    if (address==0)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot resolve host name '", value, "'", NULL);

    cfg->hostaddr=address;
    return NULL;
}

/* ========================================================================= */
/* Handle ApJServDefaultPort directive (TAKE1) */
static const char *jserv_cfg_port(cmd_parms *cmd, void *dummy, char *value) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);

    /* Check if we already processed ApJServManual directives */
    if (cfg->port!=JSERV_DEFAULT)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Set up our value */
    cfg->port=atoi(value);
    return NULL;
}

/* ========================================================================= */
/* Handle ApJServMount directive (TAKE123) */
static const char *jserv_cfg_mount(cmd_parms *cmd, void *dummy, char *value1,
                                   char *value2, char *value3) {
    server_rec *s=cmd->server;
    pool *p=cmd->pool;
    jserv_config *cfg=jserv_server_config_get(s);
    jserv_mount *mnt=(jserv_mount *) ap_pcalloc(p,sizeof(jserv_mount));
    char *buf;
    char *tmp;
    int x,y;

    /* We check if mountpoint has a valid value */
    if (value1==NULL) 
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": the first field (directory) must be specified",
                          NULL);

    /* Check if we have already some defined mounts. If we already have a mount
       we insert current mount into the mount list */
    if (cfg->mount!=NULL) {
        jserv_mount *cur=cfg->mount;

        while (cur->next!=NULL) cur=cur->next;
        cur->next=mnt;
    } else cfg->mount=mnt;

    /* Setup default (empty) values */
    mnt->mountpoint=NULL;
    mnt->config=cfg;
    mnt->protocol=NULL;
    mnt->host=NULL;
    mnt->hostaddr=JSERV_DEFAULT;
    mnt->port=JSERV_DEFAULT;
    mnt->secretfile=NULL;
    mnt->secret=NULL;
    mnt->secretsize=JSERV_DEFAULT;
    mnt->zone=NULL;

    /* Remove double slash and add starting/trailing slash in mountpoint */
    /* NOTE: temporary buffer tmp is allocated under temporary pool */
    tmp=ap_pstrcat(cmd->temp_pool,"/",value1,NULL);
    buf=(char *)ap_pcalloc(cmd->temp_pool,strlen(tmp)+1);
    x=0; y=0; buf[y++]=tmp[x++];
    while(tmp[x]!='\0')
        if (tmp[x]=='/') (buf[y-1]!='/')?(buf[y++]=tmp[x++]):x++;
        else buf[y++]=tmp[x++];
    if (buf[y-1]!='/') (buf[y++]='/');
    buf[y]='\0';

    /* Map this mountpoint */
    mnt->mountpoint=ap_pstrdup(p,buf);

    /* We check if our mounted uri has a valid value */
    if (value2!=NULL) {
        char *buf;
        char *tmp;
        char *protocol=NULL;
        char *host=NULL;
        char *port=NULL;
        char *zone=NULL;
        int x;

        /* We create a copy of value2 in temporary pool to keep original
           value safe */
        buf=ap_pstrdup(cmd->temp_pool,value2);
        tmp=buf;

        /* Try to find if we have a protocol://host */
        for (x=0; (x<128) & (buf[x]!='\0'); x++) {
            if ((buf[x]==':') & (buf[x+1]=='/') & (buf[x+2]=='/')) {
                if (x!=0) {
                    protocol=buf;
                    buf[x]='\0';
                }
                tmp=&buf[x+3];
            }
        }

        /* Check what was found after determining protocol */
        if (tmp[0]=='/') {
            zone=&tmp[1];
            host=NULL;
            tmp[0]='\0';
        } else if (tmp[0]==':') {
            port=&tmp[1];
            host=NULL;
            tmp[0]='\0';
        } else if (tmp[0]!='\0') host=tmp;
        tmp++;

        /* Check others (if we already got a zone we finished) */
        if (zone==NULL) {
            for (x=0; tmp[x]!='\0'; x++) {
                if (tmp[x]==':') {
                    port=&tmp[x+1];
                    tmp[x]='\0';
                } else if(tmp[x]=='/') {
                    zone=&tmp[x+1];
                    tmp[x]='\0';
                }
            }
        }

        /* Set protocol into structure */
        if (protocol!=NULL) {
            mnt->protocol=jserv_protocol_getbyname(protocol);
            if (strcasecmp(protocol,"status")==0)
                return ap_pstrcat(cmd->pool, cmd->cmd->name,
                                  ": mounted URL (2nd field): protocol name "
                                  "cannot be '",protocol,"'",NULL);
            if (mnt->protocol==NULL) 
                return ap_pstrcat(cmd->pool, cmd->cmd->name,
                                  ": mounted URL (2nd field): protocol '",
                                  protocol, "' cannot be found", NULL);

        }

        /* Set zone into structure */
        if (zone!=NULL) mnt->zone=ap_pstrdup(p,zone);

        /* Set port into structure */
        if (port!=NULL) mnt->port=atoi(port);

        /* Set host name and address into structure */
        if (host!=NULL) {
            unsigned long address;

            mnt->host=ap_pstrdup(p,host);
            address=jserv_resolve(mnt->host);
            if (address==0)
                return ap_pstrcat(cmd->pool, cmd->cmd->name,
                                  ": mounted URL (2nd field): cannot resolve ",
                                  "host name '", host, "'", NULL);
            mnt->hostaddr=address;
        }
    }

    /* Check if our secret file field is valid */
    if (value3!=NULL) {
        const char *ret;

        /* Get the secret key file contents and length */
        ret=jserv_readfile(cmd->pool, value3, JSERV_TRUE, &mnt->secret,
                           &mnt->secretsize);

        /* If ret is not null, an error occourred and ret points to message */
        if (ret!=NULL)
            return ap_pstrcat(cmd->pool, cmd->cmd->name,
                              ": secret file (3rd field): ", ret, NULL);
    }
    return NULL;
}

/* ========================================================================= */
/* Handle ApJServMountCopy directive (FLAG) */
static const char *jserv_cfg_mountcopy(cmd_parms *cmd, void *dummy, int flag) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);

    /* Check if we already processed ApJServMountCopy directives */
    if (cfg->mountcopy!=JSERV_DEFAULT)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Set up our value */
    cfg->mountcopy=flag?JSERV_TRUE:JSERV_FALSE;
    return NULL;
}

/* ========================================================================= */
/* Handle ApJServLogfile directive (TAKE1) */
static const char *jserv_cfg_logfile(cmd_parms *cmd, void *dummy, 
                                     char *value) {
    server_rec *s = cmd->server;
    jserv_config *cfg = jserv_server_config_get(s);
    const char *ret;

    /* Check if we already processed ApJServLogFile directives */
    if (cfg->logfile!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    /* Open the log file */
    cfg->logfile=ap_pstrdup(cmd->pool,value);
    ret=jserv_openfile(cmd->pool, cfg->logfile, JSERV_TRUE,
                    &cfg->logfilefd, JSERV_LOGFILE_FLAGS, JSERV_LOGFILE_MODE);

    /* If ret is not null, an error occourred and ret points to message */
    if (ret!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name, ": ", ret, NULL);
    return NULL;
}

/* ========================================================================= */
/* Handle ApJServSecretKey directive (TAKE1) */
static const char *jserv_cfg_secretkey(cmd_parms *cmd, void *dummy, 
                                       char *value) {
    jserv_config *cfg = jserv_server_config_get(cmd->server);
    const char *ret;

    /* Check if we already processed ApJServSecretKey directives */
    if (cfg->secretfile!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": cannot be specified more than once per host",
                          NULL);

    cfg->secretfile=ap_pstrdup(cmd->pool,value);

    /* Get the secret key file contents and length */
    ret=jserv_readfile(cmd->pool, cfg->secretfile, JSERV_TRUE, &cfg->secret,
                       &cfg->secretsize);

    /* If ret is not null, an error occourred and ret points to message */
    if (ret!=NULL)
        return ap_pstrcat(cmd->pool, cmd->cmd->name, ": ", ret, NULL);
    return NULL;
}

/* ========================================================================= */
/* Handle ApJServProtocolProperty directive (TAKE23) */
static const char *jserv_cfg_parameter(cmd_parms *cmd,void *dummy,char *value1,
                                       char *value2,char *value3) {
    jserv_config *cfg = jserv_server_config_get(cmd->server);
    jserv_protocol *proto = jserv_protocol_getbyname(value1);

    if (proto==NULL) 
        return ap_pstrcat(cmd->pool, cmd->cmd->name,
                          ": protocol '",
                          value1, "' cannot be found", NULL);
    /* Pass this parameters to the specified protocol and return */
    return jserv_protocol_parameter (proto, cfg, value2, value3);
}

/* ========================================================================= */
/* Handle ApJServAction directive (TAKE2) */
static const char *jserv_cfg_action(cmd_parms *cmd,void *dummy,char *value1,
                                       char *value2) {
    jserv_config *cfg = jserv_server_config_get(cmd->server);

    ap_table_setn(cfg->actions, value1, value2);
    return NULL;
}


/*****************************************************************************
 * Apache Module procedures                                                  *
 *****************************************************************************/

/* ========================================================================= */
/* Clean up Apache JServ Module */
static void jserv_exit(void *data) {
    jserv_config *cfg=jserv_servers;
    int ret;

    /* Call our cleanup functions */
    ret=jserv_protocol_cleanupall(cfg, JSERV_FALSE);

    /* Log our exit if it was not clean */
    if (ret==-1) {
        jserv_error(JSERV_LOG_EMERG,cfg,"Error cleaning-up protocols");
    }

    /* Log our init */
    jserv_error(JSERV_LOG_INFO,cfg,"Apache Module was cleaned-up");

    /* Our memory pool will be destroyed when the pool for which this cleanup
       was registered will be destroyed */
}

/* ========================================================================= */
/* Clean up JServ when a child is exiting */
static void jserv_child_exit(void *data) {
    jserv_config *cfg=jserv_servers;
    int ret;

    /* Call our cleanup functions */
    ret=jserv_protocol_cleanupall(cfg, JSERV_TRUE);

    /* Log our exit if it was not clean */
    if (ret==-1) {
        jserv_error(JSERV_LOG_EMERG,cfg,
                    "Error cleaning-up protocols (ap_child)");
    }

    /* Log our init */
    jserv_error(JSERV_LOG_INFO,cfg,"Apache JServ Module was cleaned-up (ap_child)");
}

/* ========================================================================= */
/* Initialize Apache JServ Module */
static void jserv_init(server_rec *s, pool *p) {
    jserv_config *cfg=jserv_servers;
    int ret;

    /* Check ApJServSecretKey */
    if (cfg->secretfile==NULL) {
         printf("You must specify a secret key, or disable this feature.\nTo disable, add \"ApJServSecretKey DISABLED\" to your Apache configuration file.\nTo use, add \"ApJServSecretKey {filename}\" where filename is document\nwith more or less random contents, and perhaps a few kb in length.\nThe JServ documentation explains this in more detail.\n");
         exit(1);
    }

    /* Setup base server defaults (in case merge_config was never called) */
    jserv_server_config_default(p, cfg);

    /* Log our init */
    jserv_error(JSERV_LOG_DEBUG,cfg,"Apache JServ Module is initializing");

#if MODULE_MAGIC_NUMBER >= 19980527
    /* Tell apache we're here */
    ap_add_version_component(JSERV_NAME "/" JSERV_VERSION);
#endif

    /* Init all protocols */
    ret=jserv_protocol_initall(cfg, JSERV_FALSE);

    /* Exit if we didn't initialize all protocols */
    if (ret==-1) {
        jserv_error_exit(JSERV_LOG_EMERG,cfg,"Error initializing protocols");
    }

    /* Create our memory pool */
    jserv_pool=ap_make_sub_pool(p);

    /* Register for clean exit */
    ap_register_cleanup(p, cfg, jserv_exit, ap_null_cleanup);
}

/* ========================================================================= */
/* Initialize Apache JServ Module when a new child is created */
static void jserv_child_init(server_rec *s, pool *p) {
    jserv_config *cfg=jserv_servers;
    int ret;

    /* Log our init */
    jserv_error(JSERV_LOG_DEBUG,cfg,"Apache JServ Module is initializing (ap_child)");

    /* Init all protocols */
    ret=jserv_protocol_initall(cfg, JSERV_TRUE);

    /* Exit if we didn't initialize all protocols */
    if (ret==-1) {
        jserv_error(JSERV_LOG_EMERG, cfg, 
                    "Error initializing protocols (ap_child)");
    }

    /* Register for clean exit */
    ap_register_cleanup(p, cfg, ap_null_cleanup, jserv_child_exit);
}

/* ========================================================================= */
/* Match a request with a mount structure */
static jserv_request *jserv_translate_match(request_rec *r,jserv_mount *mount){
    char *mnt=mount->mountpoint;
    char *uri=r->uri;
    jserv_request *req=NULL;
    int x,y;

    /* Matching URI and MNT discarding double slashes*/
    x=0; y=0;
    while ((uri[x]==mnt[y]) & (uri[x]!='\0')  & (mnt[y]!='\0')) {
        if (uri[x]=='/') while (uri[x+1]=='/') x++;
        x++;
        y++;
    }

    /* Check whether URI finished */
    if (uri[x]=='\0') {
        /* If MNT finished too or the remaining is only trailing slash we
           found a directory*/
        if ((mnt[y]=='\0') | ((mnt[y]=='/') & (mnt[y+1]=='\0'))) {
            req=(jserv_request *)ap_pcalloc(r->pool,sizeof(jserv_request));
            req->isdir=JSERV_TRUE;
            req->mount=mount;
            req->zone=mount->zone;
            req->servlet=NULL;
            return req;
        }

    /* If URI still contains data while MNT does not and the last character of
       URI parsed was a slash after it we find the target (zone/class) */
    } else if ((uri[x-1]=='/') & (mnt[y]=='\0')) {
        char *tmp=&uri[x];  /* Pointer to zone/class or class */
        int k=0;

        req=(jserv_request *)ap_pcalloc(r->pool,sizeof(jserv_request));
        req->isdir=JSERV_FALSE;
        req->mount=mount;
        req->zone=NULL;
        req->servlet=NULL;

        /* If zone is not specified in our mount structyre, check if we
           have a zone after the directory in URI */
        if (mount->zone==NULL) {
            while (tmp[k]!='\0') {
                if (tmp[k]=='/') {
                    req->zone=ap_pstrndup(r->pool,tmp,k);
                    /* Remove double slashes */
                    while (tmp[k]=='/') k++;
                    /* Check whether we have a null class name (dir request) */
                    if (tmp[k]!='\0') req->servlet=ap_pstrdup(r->pool,&tmp[k]);
                    else req->isdir=JSERV_TRUE;
                    return req;
                }
                k++;
            }

            /* If we didn't get a zone/class then we have a directory */
            req->zone=ap_pstrdup(r->pool,tmp);
            req->servlet=NULL;
            req->isdir=JSERV_TRUE;
            return req;
        }
        x = 0;
        while (tmp[x] != '/' && tmp[x] != 0)
            x++;
        if (tmp[x] == '/') {
            r->path_info = ap_pstrdup(r->pool,tmp+x);
            tmp[x] = 0;
        }
        req->zone=mount->zone;
        req->servlet=ap_pstrdup(r->pool,tmp);
        return req;
    }

    /* Otherwise what we parsed was not our business */
    return NULL;
}

/* ========================================================================= */
/* Match a request in current server config */
static int jserv_translate_handler(request_rec *r) {
    server_rec *s = r->server;
    jserv_config *cfg = jserv_server_config_get(s);
    jserv_mount *cur;
    jserv_request *result=NULL;

    /* If we didn't get our server config we'll decline the request*/
    if (cfg==NULL) return DECLINED;

    /* If we didn't define any mounts we'll decline the request*/
    if (cfg->mount==NULL) return DECLINED;

    /* We are sure we have at least one mount point */
    cur=cfg->mount;

    /* Check all our structures for a possible match */
    while (cur!=NULL) {
        /* If our translator returns a non NULL pointer, we got it */
        if ((result=jserv_translate_match(r, cur))!=NULL) {
            /* Block direct requests to JServ as servlet */
            if (strstr(r->uri,"/" JSERV_SERVLET)!=NULL) {
                return FORBIDDEN;
            }
            ap_set_module_config(r->request_config, &jserv_module,result);
            r->handler=ap_pstrdup(r->pool,"jserv-servlet");
            return OK;
        }
        cur=cur->next;
    }

    /* If this was not matched we decline */
    return DECLINED;
}

/* ========================================================================= */
/* Match a request in current server config (from mod_mime and mod_actions) */
static int jserv_type_match(request_rec *r) {
    server_rec *s = r->server;
    jserv_config *cfg = jserv_server_config_get(s);
    const char *servlet=NULL;
    char *file=NULL;
    char *ext=NULL;

    /* Check filename */
    if (r->filename==NULL) return DECLINED;
    file=strrchr(r->filename, '/');
    if (file==NULL) file=r->filename;
    ext=strrchr(file, '.');
    if (ext==NULL) return DECLINED;

    /* Get and check extension */
    servlet=ap_table_get(cfg->actions, ext);
    if (servlet==NULL) return DECLINED;

    /* Set the servlet name in r->notes table */
    ap_table_set(r->notes,"jserv-action",servlet);
    r->handler=ap_pstrdup(r->pool,"jserv-action");

    return OK;
}

/* ========================================================================= */
/* Handle an action request (redirect from extension to servlet) */
static int jserv_handler_action(request_rec *r) {
    server_rec *s = r->server;
    jserv_config *cfg = jserv_server_config_get(s);
    const char *servlet=NULL;
    char *uri=NULL;

    /* Check the servlet name passed by jserv_type_match */
    servlet=ap_table_get(r->notes, "jserv-action");
    if (servlet==NULL) {
        jserv_error(JSERV_LOG_INFO,cfg,"Action with no servlet name received",
                    r->filename, r->path_info);
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    /* Format servlet uri and redirect */
    uri=ap_pstrcat(r->pool, servlet, r->args?"?":NULL, r->args, NULL);
    ap_internal_redirect_handler(uri,r);
    
    /* Apache expects the handler to return 0.  not sure the right way to
       do this, but this fixed the immediate problem.  -- James */
    return JSERV_FALSE;
}

/* ========================================================================= */
/* Handle a servlet request */
static int jserv_handler_servlet(request_rec *r) {
    server_rec *s = r->server;
    jserv_config *cfg = jserv_server_config_get(s);
    jserv_request *req=ap_get_module_config(r->request_config, &jserv_module);
    jserv_protocol *proto=NULL;
    int ret;

    /* If this was an internal redirection from Apache JServ then our path_info is
       previous uri */
    if (r->prev!=NULL)
        if (strcasecmp(r->prev->handler,"jserv-action")==0) {
            /* Remove date header (FIX BUG: StringIndexOutOfBoundsException) */
            ap_table_unset(r->headers_in,"If-Modified-Since");
            r->path_info=r->prev->uri;
            r->filename=NULL;
        }

    /* Check if we have a per request or per server protocol and use it */
    if (req->mount->protocol!=NULL) proto=req->mount->protocol;
    else if (cfg->protocol!=NULL) proto=cfg->protocol;
    else {
        /* Why we did not find a protocol? */
        jserv_error(JSERV_LOG_EMERG,cfg,
                    "cannot find a protocol for request %s on host %s",
                    r->uri, r->hostname);
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    
    /* Handle request to protocol */
    ret=jserv_protocol_handler (proto,cfg,req,r);
    return ret;
}

/* ========================================================================= */
/* Handle a status request */
static int jserv_handler_status(request_rec *r) {
    server_rec *s = r->server;
    jserv_config *cfg = jserv_server_config_get(s);
    jserv_protocol *proto = jserv_protocol_getbyname("status");
    jserv_request *req = ap_pcalloc(r->pool, sizeof(jserv_request));
    int ret;

    if (proto==NULL) {
        jserv_error(JSERV_LOG_EMERG,cfg,
                    "cannot find protocol 'status' for status handler");
        return HTTP_INTERNAL_SERVER_ERROR;
    }
    
    ret=jserv_protocol_handler (proto,cfg,req,r);
    return ret;
}


/*****************************************************************************
 * Predefined Apache and JServ stuff                                         *
 *****************************************************************************/

/* ========================================================================= */
/* List of Apache JServ handlers */
static handler_rec jserv_handlers[] = {
    {"jserv-servlet", jserv_handler_servlet},
    {"jserv-status", jserv_handler_status},
    {"jserv-action", jserv_handler_action},
    {NULL}
};

/* ========================================================================= */
/* List of directives for Apache */
static command_rec jserv_commands[] = {
    {"ApJServManual", jserv_cfg_manual, NULL, RSRC_CONF, FLAG,
     "Whether Apache JServ is running in manual or automatic mode."},
    {"ApJServProperties", jserv_cfg_properties, NULL, RSRC_CONF, TAKE1,
     "The full pathname of jserv.properties file."},
    {"ApJServDefaultProtocol", jserv_cfg_protocol, NULL, RSRC_CONF, TAKE1,
     "The default protocol used for connecting to Apache-JServ."},
    {"ApJServDefaultHost", jserv_cfg_host, NULL, RSRC_CONF, TAKE1,
     "The default host running Apache JServ."},
    {"ApJServDefaultPort", jserv_cfg_port, NULL, RSRC_CONF, TAKE1,
     "The default port on which Apache JServ is running on."},
    {"ApJServMount", jserv_cfg_mount, NULL, RSRC_CONF, TAKE123,
     "Where Apache JServ servlets will be mounted under Apache."},
    {"ApJServMountCopy", jserv_cfg_mountcopy, NULL, RSRC_CONF, FLAG,
     "Whether <VirtualHost> inherits base host mount points or not."},
    {"ApJServLogFile", jserv_cfg_logfile, NULL, RSRC_CONF, TAKE1,
     "Apache JServ log file relative to Apache root directory."},
    {"ApJServSecretKey", jserv_cfg_secretkey, NULL, RSRC_CONF, TAKE1,
     "Apache JServ secret key file relative to Apache root directory."},
    {"ApJServProtocolParameter", jserv_cfg_parameter, NULL, RSRC_CONF, TAKE23,
     "Apache JServ protocol-dependant property."},
    {"ApJServAction", jserv_cfg_action, NULL, RSRC_CONF, TAKE2,
     "Apache JServ action mapping extension to servlets."},
    {NULL}
};

/* ========================================================================= */
/* Our mod_jserv module structure */
module MODULE_VAR_EXPORT jserv_module = {
    STANDARD_MODULE_STUFF,
    jserv_init,                 /* module initializer */
    NULL,                       /* per-directory config creator */
    NULL,                       /* dir config merger */
    jserv_server_config_create, /* server config creator */
    jserv_server_config_merge,  /* server config merger */
    jserv_commands,             /* command table */
    jserv_handlers,             /* [7] list of handlers */
    jserv_translate_handler,    /* [2] filename-to-URI translation */
    NULL,                       /* [5] check/validate user_id */
    NULL,                       /* [6] check user_id is valid *here* */
    NULL,                       /* [4] check access by host address */
    jserv_type_match,           /* [7] MIME type checker/setter */
    NULL,                       /* [8] fixups */
    NULL,                       /* [10] logger */
    NULL,                       /* [3] header parser */
#if MODULE_MAGIC_NUMBER > 19970622
    jserv_child_init,           /* apache child process initializer */
    NULL,                       /* apache child process exit/cleanup */
    NULL                        /* [1] post read_request handling */
#endif
};

