/*
 * Copyright (c) 1997-1998 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */

package org.apache.jserv;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;
import javax.servlet.*;
import javax.servlet.http.*;
import org.apache.java.io.*;

/**
 * This class is the thread that handles all communications between
 * the Java VM and the web server.
 *
 * @author Alexei Kosut
 * @author Francis J. Lacoste
 * @author Stefano Mazzocchi
 * @author Vadim Tkachenko
 * @version $Revision: 1.23 $ $Date: 1998/12/18 14:19:52 $
 **/
public class JServConnection
implements Runnable, HttpServletRequest, HttpServletResponse,
        JServSendError, JServLogChannels {

    private Socket client;
    private InputStream in;
    private OutputStream out;

    private Hashtable servletMgrTable;
    private JServLog log;

    // Servlet setup stuff
    private JServServletManager mgr;
    private JServContext context;
    private Servlet servlet;
    private String servletname;
    private String servletzone;
    private String hostname;
    private boolean stm_servlet;

    private String auth;
    private String signal;

    private JServInputStream servlet_in;
    private JServOutputStream servlet_out;

    private BufferedReader servlet_reader;
    private boolean called_getInput;

    private PrintWriter servlet_writer;
    private boolean called_getOutput;

    // HTTP Stuff
    private Hashtable headers_in;
    private Hashtable headers_out;
    private Hashtable env_vars;
    private Cookie[] cookies_in;
    private Vector cookies_out;

    private int status;
    private String status_string;

    private Hashtable params;
    private boolean got_input;

    private boolean sent_header;

    // Session stuff
    JServSession session;
    String requestedSessionId;
    boolean idCameAsCookie;
    boolean idCameAsUrl;

    // Stuff used for profiling
    private static boolean PROFILE = false;
    private static int counter = 0;

    /**
     * Initalize the streams and starts the thread.
     */
    public JServConnection(Socket clientSocket, Hashtable mgrTable,
            JServLog log) {
        this.client = clientSocket;
        this.servletMgrTable = mgrTable;
        this.log = log;

        try {
            this.in = new ReadFullyInputStream(new BufferedInputStream(client.getInputStream()));
            this.out = new BufferedOutputStream(client.getOutputStream());
        } catch(IOException e) {
            try {
                client.close();
            } catch(IOException ignored) {}
            if (log.active) {
                log.log(CH_SERVICE_REQUEST,
                    "Exception while getting socket streams: ");
            }
            log.log(e);
            return;
        }
    }

    /**
     * This methods provides and incapsulates the Servlet service.
     */
    public void run() {

        // Profiler stuff
        if (PROFILE && (counter > 0)) {
            // Runtime.getRuntime().traceInstructions(true);
            Runtime.getRuntime().traceMethodCalls(true);
        }

        if (log.active) {
            log.log(CH_SERVICE_REQUEST, "Initializing servlet request");
        }

        // Set up hash tables
        headers_in = new Hashtable();
        headers_out = new Hashtable();
        env_vars = new Hashtable();

        // cookies_in is initialized after the header are parsed.
        cookies_out = new Vector(20);

        // Set up HTTP defaults
        status = SC_OK;
        status_string = null;

        sent_header = false;
        got_input = false;
        called_getInput = false;
        called_getOutput = false;

        auth = null;
        signal = null;

        // Read in the data
        if (log.active) {
            log.log(CH_SERVICE_REQUEST, "Reading request data");
        }

        // Read data, and check to make sure it read ok
        if (!readData()) {
            sendError(SC_BAD_REQUEST, "Malformed data send to JServ");
            return;
        }

        // override environemnt values
        env_vars.put("GATEWAY_INTERFACE", JServ.version);

        // Look for a signal
        if (signal != null) {
            try {
                // close the socket connection before handling any signal
                client.close();
            } catch (IOException ignored) {}

            if (log.active) {
                log.log(CH_SIGNAL, "Received signal " + signal);
            }

            if (signal.equals("01")) { // SIGHUP
                JServ.restart();
            } else if (signal.equals("15")) { // SIGTERM
                JServ.terminate();
            }

            return;
        }

        // Look for the servlet zone
        if (servletzone == null) {
            // If servletzone is null, probably the client asks for
            // some system class so we pick the first available
            // servlet zone.
            servletzone = (String) servletMgrTable.keys().nextElement();
        }

        // Look for the servlet
        if (servletname == null) {
            sendError(SC_BAD_REQUEST, "Received empty servlet name");
            return;
        }

        // Look for the servlet
        if (hostname == null) {
            sendError(SC_BAD_REQUEST, "Received empty host name");
            return;
        }

        // Find the servlet manager that handles this zone
        mgr = (JServServletManager) servletMgrTable.get(servletzone);

        if (mgr == null) {
            sendError(SC_NOT_FOUND,    "Servlet zone \"" + servletzone
                + "\" not found.");
            return;
        }

        // Parse the cookies
        if (log.active) {
            log.log(CH_SERVICE_REQUEST, "Parsing cookies");
        }
        cookies_in = JServUtils.parseCookieHeader(getHeader("Cookie"));

        // Now to the session stuff
        requestedSessionId =
            JServServletManager.getUrlSessionId(getQueryString());

        idCameAsUrl = requestedSessionId != null;
        String cookieSession =
            JServServletManager.getCookieSessionId(cookies_in);
        idCameAsCookie = cookieSession != null;

        // FIXME: What do we do if url and cookie don't have same id ?
        requestedSessionId = ( requestedSessionId == null )
            ? cookieSession
            : requestedSessionId;

        if (requestedSessionId != null) {
            if (log.active) {
                log.log(CH_SERVICE_REQUEST, "Request is in session "
                    + requestedSessionId);
            }
            JServSession s = (JServSession)
                mgr.getSession(requestedSessionId);
            if (requestedSessionId != null && s != null) {
                s.access();
            }
        }

        try {
            // Let the mgr check for changes.
            mgr.checkReload(this);
        } catch (IllegalArgumentException ex) {
            // this exception may be thrown by JServClassLoader when
            // it's repository was altered in a dangerous way
            // (removed directories or zip/jar files; corrupted zip/jar
            // archives etc.)
            sendError(ex);
            return;
        }

        try {
            context = mgr.loadServlet(servletname, this);
        } catch ( ServletException initError ) {
            sendError(SC_INTERNAL_SERVER_ERROR,
                "Initialization error while loading the servlet: "
                    + initError.getMessage());
        }

        if (context == null  ||  context.servlet == null) {
            sendError(SC_INTERNAL_SERVER_ERROR,
                "An unknown error occured loading the servlet.");
            return;
        }

        servlet = context.servlet;

        // is this a SingleThreadModel servlet?
        boolean stm_servlet = servlet instanceof SingleThreadModel;
        if (stm_servlet && log.active) {
            log.log(CH_SINGLE_THREAD_MODEL,
                "We've got a SingleThreadModel servlet.");
        }

        // Set up a read lock on the servlet. Note that anytime
        // we return, we need to make sure to unlock this. Otherwise,
        // we'll end up holding onto the lock forever. Oops.
        // This is done in the finally clause.
        // The lock is acquired outside of the try block so that
        // we do not unlock a lock that was never acquired.
        try {
            context.lock.readLock();
        } catch ( InterruptedException stop ) {
            if (log.active) {
                log.log(CH_SERVICE_REQUEST,
                    "Caught interrupted exception while waiting for servlet "
                        + servletname + ": sending error.");
            }
            sendError(SC_INTERNAL_SERVER_ERROR,
                "InterruptedException while waiting for servlet lock.");
            if (stm_servlet) {
              if (log.active) {
                log.log(CH_SERVICE_REQUEST,
                    "Returning the SingleThreadModel servlet.");
              }
              mgr.returnSTMS(servletname, context);
            }
            return;
        }

        try {
            // Set up the servlet's I/O streams
            servlet_in = new JServInputStream(in);
            servlet_out = new JServOutputStream(out);

            // Start up the servlet
            try {
                if (log.active) {
                    log.log(CH_SERVICE_REQUEST, "Calling service()");
                }
                servlet.service(this, this);
            } catch(Exception e) {
                sendError(e);
                return;
            } catch(Error e) {
                sendError(e);
                throw (Error) e.fillInStackTrace();
            }

            // Make sure we've send the HTTP header, even if no
            // entity data has been
            //
            // - KNOWN BUG - After the header has been sent to Apache
            // all other headers are lost (and cookies).
            // - FIXME - The new protocol AJPv2.1 will fix this.
            sendHttpHeaders();

            // All done close the streams and the connection
            try {
                if (servlet_writer != null) { 
                    servlet_writer.flush();
                    servlet_writer.close();
                }
                out.flush();
                out.close();
                client.close();
            } catch(IOException ignored) {}
            
        } finally {
            // Clean up
            context.lock.readUnlock();
            if (stm_servlet) {
                if (log.active) {
                    log.log(CH_SERVICE_REQUEST,
                        "Returning the SingleThreadModel servlet.");
                }
                mgr.returnSTMS(servletname, context);
            }

            // Profiler stuff
            if (PROFILE && (counter++ > 0)) {
                // Runtime.getRuntime().traceInstructions(false);
                Runtime.getRuntime().traceMethodCalls(false);
            }
        }
    }

    /**
     * Return the hostname
     */
    public String getHostName() {
        return hostname;
    }

    /**
     * Read a line from the input.
     */
    private String readHexLine() throws IOException {
        byte str[];

        try {
            // Read four bytes from the input stream
            byte hex[] = new byte[4];
            if (in.read(hex) != 4) {
                // Fail read..
                throw new IOException("Malformed Request: reading line length");
            }

            // Convert them from hex to decimal
            int len = Character.digit((char)hex[0], 16);
            len = len << 4;
            len += Character.digit((char)hex[1], 16);
            len = len << 4;
            len += Character.digit((char)hex[2], 16);
            len = len << 4;
            len += Character.digit((char)hex[3], 16);

            if (len > 0) {
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Will read " + len
                        + " bytes for this line");
                }

                // Read len bytes from the input stream
                str = new byte[len];
                if (in.read(str) != len) {
                    // Fail read
                    throw new IOException("Malformed request: reading line data");
                }

                // return the bytes as a string
                return new String(str);
            } else {
                return null;
            }
        } catch  (Exception anythingWrong) {
            throw new IOException("Malformed request: " + anythingWrong);
        }
    }

    /**
     * Read all the data.
     */
    private boolean readData() {
        String line;
        char id;

        while (true) {
            try {
                line = readHexLine();
            } catch(IOException e) {
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Malformed request data: "
                        + e.getMessage());
                }
                return false;
            }

            // Done reading request data
            if (line == null) {
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "All data read.");
                }
                return true;
            }

            // Log the request line read
            if (log.active) {
                log.log(CH_REQUEST_DATA, "Read: " + line);
            }

            // Get the identifier from the first character
            try {
                id = line.charAt(0);
                line = line.substring(1);
            } catch(StringIndexOutOfBoundsException e) {
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "No data to command: "
                        + line.charAt(0));
                }
                return false;
            }

            // All ids' take one or two pieces of data separated by a tab.
            StringTokenizer tokens = new StringTokenizer(line, "\t");
            String token1, token2;

            try {
                token1 = tokens.nextToken();
            } catch(NoSuchElementException e) {
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Missing first piece of data for "
                        + id);
                }
                return false;
            }

            try {
                token2 = tokens.nextToken();
            } catch(NoSuchElementException e) {
                token2 = null;
            }

            // Switch, depending on what the id is
            switch (id) {
            case 'C': // ServletZone + Servlet request
                servletzone = token1;
                servletname = token2;
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Servlet Zone: " + token1
                        + " Servlet: " + token2);
                }
                break;
            case 'S': // Host name
                hostname = token1;
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Hostname: " + token1);
                }
                break;
            case 'E': // Env variable
                env_vars.put(token1, (token2 != null) ? token2 : "");
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Env: " + token1 + "=" + token2);
                }
                break;
            case 'H': // Header
                headers_in.put(token1.toLowerCase(),
                    (token2 != null) ? token2 : "");
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Header: " + token1 + "=" + token2);
                }
                break;
            case 's': // Signal
                signal = token1;
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Signal: " + token1);
                }
                break;
            default: // What the heck is this?
                if (log.active) {
                    log.log(CH_REQUEST_DATA, "Unknown: " + token1 + "=" + token2);
                }
                return false;
            }
        }
    }

    /**
     * Finds a status string from one of the standard
     * status code.
     * @param sc The status code to find a descriptive string.
     * @return A string describing this status code.
     */
    public static final String findStatusString(int sc) {
        switch (sc) {
        case SC_ACCEPTED:
            return "Accepted";
        case SC_BAD_GATEWAY:
            return "Bad Gateway";
        case SC_BAD_REQUEST:
            return "Bad Request";
        case SC_CREATED:
            return "Created";
        case SC_FORBIDDEN:
            return "Forbidden";
        case SC_INTERNAL_SERVER_ERROR:
            return "Internal Server Error";
        case SC_MOVED_PERMANENTLY:
            return "Moved Permanently";
        case SC_MOVED_TEMPORARILY:
            return "Moved Temporarily";
        case SC_NO_CONTENT:
            return "No Content";
        case SC_NOT_FOUND:
            return "Not Found";
        case SC_NOT_IMPLEMENTED:
            return "Method Not Implemented";
        case SC_NOT_MODIFIED:
            return "Not Modified";
        case SC_OK:
            return "OK";
        case SC_SERVICE_UNAVAILABLE:
            return "Service Temporarily Unavailable";
        case SC_UNAUTHORIZED:
            return "Authorization Required";
        default:
            return "Response";
        }
    }

    /**
     * Send the HTTP headers and prepare to send response.
     */
    protected void sendHttpHeaders() {
        if (sent_header) {
            return;
        } else {
            sent_header = true;
        }

        // Use a PrintWriter around the socket out.
        PrintWriter printOut = new PrintWriter(this.out);

        if (log.active) {
            log.log(CH_SERVICE_REQUEST, "Sending response headers.");
        }

        // Send the status info
        if (status_string == null) {
            status_string = findStatusString(status);
        }

        String statusLine = "Status: " + status + " " + status_string;
        printOut.print(statusLine + "\r\n");
        if (log.active) {
            log.log(CH_RESPONSE_HEADERS, statusLine);
        }

        if (headers_out != null) {
            // Send the headers
            for (Enumeration e = headers_out.keys(); e.hasMoreElements();) {
                String key = (String) e.nextElement();
                String hdr = key + ": " + headers_out.get(key);
                printOut.print(hdr + "\r\n");
                if (log.active) {
                    log.log(CH_RESPONSE_HEADERS, hdr);
                }
            }
        }

        // Send the cookies
        Enumeration cookies = cookies_out.elements();
        while (cookies.hasMoreElements()) {
            Cookie cookie = (Cookie) cookies.nextElement();
            String cookieHdr = "Set-Cookie: " +
                JServUtils.encodeCookie(cookie);
            printOut.print(cookieHdr + "\r\n");
            if (log.active) {
                log.log(CH_RESPONSE_HEADERS, cookieHdr);
            }
        }

        // Send a terminating blank line
        printOut.print("\r\n");
        // Flush the PrintWriter
        printOut.flush();
    }

    //---------------------------------------- Implementation of ServletRequest

    /**
     * Returns the size of the request entity data, or -1 if not
     * known. Same as the CGI variable CONTENT_LENGTH.
     */
    public int getContentLength() {
        String lenstr = (String) env_vars.get("CONTENT_LENGTH");

        if (lenstr == null) {
            return -1;
        }

        try {
            return Integer.parseInt(lenstr);
        } catch(NumberFormatException e) {
            return -1;
        }
    }

    /**
     * Returns the Internet Media Type of the request entity data,
     * or null if not known. Same as the CGI variable
     * CONTENT_TYPE.
     */
    public String getContentType() {
        return (String) env_vars.get("CONTENT_TYPE");
    }

    /**
     * Returns the protocol and version of the request as a string of
     * the form <code>&lt;protocol&gt;/&lt;major version&gt;.&lt;minor
     * version&gt</code>.  Same as the CGI variable SERVER_PROTOCOL.
     */
    public String getProtocol() {
        return (String) env_vars.get("SERVER_PROTOCOL");
    }

    /**
     * Returns the scheme of the URL used in this request, for example
     * "http", "https", or "ftp".  Different schemes have different
     * rules for constructing URLs, as noted in RFC 1738.  The URL used
     * to create a request may be reconstructed using this scheme, the
     * server name and port, and additional information such as URIs.
     */
    public String getScheme() {
        // Stronghold and Ben Laurie's Apache-SSL set the "HTTPS" environment
        // variable in secure mode.
        String val = (String) env_vars.get("HTTPS");

        if (val != null) {
            if (val.equalsIgnoreCase("on")) {
                return "https";
            } else if (val.equalsIgnoreCase("off")) {
                return "http";
            }
        }

        // FIXME: We just don't have this information available. We'll
        // look at the port, and return https if it's 443, but that's
        // not a real solution.
        if (getServerPort() == 443) {
            return "https";
        } else {
            return "http";
        }
    }

    /**
     * Returns the host name of the server that received the request.
     * Same as the CGI variable SERVER_NAME.
     */
    public String getServerName() {
        return (String) env_vars.get("SERVER_NAME");
    }

    /**
     * Returns the port number on which this request was received.
     * Same as the CGI variable SERVER_PORT.
     */
    public int getServerPort() {
        String portstr = (String) env_vars.get("SERVER_PORT");
        int port;

        if (portstr == null) {
            return -1;
        }

        try {
            port = Integer.parseInt(portstr);
        } catch(NumberFormatException e) {
            return -1;
        }

        return port;
    }

    /**
     * Returns the IP address of the agent that sent the request.
     * Same as the CGI variable REMOTE_ADDR.
     */
    public String getRemoteAddr() {
        return (String) env_vars.get("REMOTE_ADDR");
    }

    /**
     * Returns the fully qualified host name of the agent that sent the
     * request. Same as the CGI variable REMOTE_HOST.
     */
    public String getRemoteHost() {
        return (String) env_vars.get("REMOTE_HOST");
    }

    /**
     * Applies alias rules to the specified virtual path and returns
     * the corresponding real path, or null if the translation can not
     * be performed for any reason.  For example, an HTTP servlet would
     * resolve the path using the virtual docroot, if virtual hosting
     * is enabled, and with the default docroot otherwise.  Calling
     * this method with the string "/" as an argument returns the
     * document root.
     * @param path The virtual path to be translated to a real path.
     */
    public String getRealPath(String path) {
        // FIXME: Make this somehow talk to Apache, do a subrequest
        // and get the real filename. Until then, we just tack the path onto
        // the doc root and hope it's right. *sigh*

        // DOCUMENT_ROOT is not a standard CGI var, although Apache always
        // gives it. So we allow for it to be not present.
        String doc_root = (String) env_vars.get("DOCUMENT_ROOT");

        if (doc_root == null) {
            return null;
        } else {
            return doc_root + path;
        }
    }

    /**
     * Returns an input stream for reading binary data in the request body.
     * @exception IllegalStateException if getReader has been
     *        called on this same request.
     * @exception IOException n other I/O related errors.
     */
    public ServletInputStream getInputStream() throws IOException {
        if (servlet_reader != null) {
            throw new IllegalStateException(
                "getReader() has already been called.");
        }

        got_input = true;
        called_getInput = true;
        return servlet_in;
    }

    /**
     * Parse parameter stuff.
     */
    private boolean parseParams() {
        // Have we already done it?
        if (params != null) {
            return false;
        }

        try {
            // Is this a post or a get?
            String method = getMethod();
            if (method.equals("GET")) {
                params = HttpUtils.parseQueryString(getQueryString());
            } else if (method.equals("POST")) {
                // Hmm... must have given the stream through getInputStream()
                if (got_input) {
                    return true;
                } else {
                    got_input = true;
                    params = HttpUtils.parsePostData(getContentLength(),
                        servlet_in);
                    if (params == null) {
                        params = new Hashtable();
                    }
                    return false;
                }
            } else {
                // Unknown method
                return true;
            }
        } catch (IllegalArgumentException e) {
            return true;
        }

        return false;
    }

    /**
     * Returns a string containing the lone value of the specified
     * parameter, or null if the parameter does not exist. For example,
     * in an HTTP servlet this method would return the value of the
     * specified query string parameter. Servlet writers should use
     * this method only when they are sure that there is only one value
     * for the parameter.  If the parameter has (or could have)
     * multiple values, servlet writers should use
     * getParameterValues. If a multiple valued parameter name is
     * passed as an argument, the return value is implementation
     * dependent.
     * @param name the name of the parameter whose value is required.
     * @deprecated Please use getParameterValues
     */
    public String getParameter(String name) {
        if (parseParams()) {
            return null;
        }

        Object val = params.get(name);

        if (val == null) {
            return null;
        } else if (val instanceof String[]) {
            // It's an array, return the first element
            return ((String[])val)[0];
        } else {
            // Its a string so return it
            return (String) val;
        }
    }

    /**
     * Added for the <servlet> tag support - RZ.
     * Provides the ability to add to the request object
     * the parameter of an embedded servlet.
     */
    public void setParameter(String name, String value) {
        // add the parameter in the hashtable, overrides any previous value
        parseParams();

        if(params != null) {
            params.put(name, value);
        }
    }

    /**
     * Returns the values of the specified parameter for the request as
     * an array of strings, or null if the named parameter does not
     * exist. For example, in an HTTP servlet this method would return
     * the values of the specified query string or posted form as an
     * array of strings.
     * @param name the name of the parameter whose value is required.
     */
    public String[] getParameterValues(String name) {
        if (parseParams()) {
            return null;
        }

        Object val = params.get(name);

        if (val == null) {
            return null;
        } else if (val instanceof String) {
            // It's a string, convert to an array and return
            String va[] = {(String) val};
            return va;
        } else {
            // It'a an array so return it
            return (String[]) val;
        }
    }

    /**
     * Returns the parameter names for this request as an enumeration
     * of strings, or an empty enumeration if there are no parameters
     * or the input stream is empty.  The input stream would be empty
     * if all the data had been read from the stream returned by the
     * method getInputStream.
     */
    public Enumeration getParameterNames() {
        if (parseParams()) {
            return (new Vector()).elements();
        }
        return params.keys();
    }

    /**
     * Returns the value of the named attribute of the request, or
     * null if the attribute does not exist.  This method allows
     * access to request information not already provided by the other
     * methods in this interface.  Attribute names should follow the
     * same convention as package names.
     * The following predefined attributes are provided.
     *
     * <TABLE BORDER>
     * <tr>
     *   <th>Attribute Name</th>
     *   <th>Attribute Type</th>
     *   <th>Description</th>
     * </tr>
     * <tr>
     *   <td VALIGN=TOP>javax.net.ssl.cipher_suite</td>
     *   <td VALIGN=TOP>string</td>
     *   <td>The string name of the SSL cipher suite in use, if the
     *       request was made using SSL</td>
     * </tr>
     * <tr>
     *   <td VALIGN=TOP>javax.net.ssl.peer_certificates</td>
     *   <td VALIGN=TOP>array of java.security.cert.X509Certificate</td>
     *   <td>The chain of X.509 certificates which authenticates the client.
     *       This is only available when SSL is used with client
     *       authentication is used.</td>
     * </tr>
     * <tr>
     *   <td VALIGN=TOP>javax.net.ssl.session</td>
     *   <td VALIGN=TOP>javax.net.ssl.SSLSession</td>
     *   <td>An SSL session object, if the request was made using SSL.</td>
     * </tr>
     *
     * </TABLE>
     *
     * <BR>
     * <P>The package (and hence attribute) names beginning with java.*,
     * and javax.* are reserved for use by Javasoft. Similarly, com.sun.*
     * is reserved for use by Sun Microsystems.
     *
     * <p><b>Note</b> The above attributes are not yet implemented by
     * JServ.
     * <p>On the other hand, attribute named
     * "org.apache.jserv.&lt;variable&gt;" returns the content of the
     * environment (CGI) variable "&lt;variable&gt;".
     */
    public Object getAttribute(String name) {
        // We return "org.apache.jserv.<variable>" as the contents
        // of environment (CGI) variable "<variable>"
        if (!name.startsWith("org.apache.jserv.")) {
            return null;
        }

        // interface to get attribute names
        if (name.equals("org.apache.jserv.attribute_names")) {
            return env_vars.keys();
        }

        return env_vars.get(name.substring("org.apache.jserv.".length()));
    }

    /**
     * Returns a buffered reader for reading text in the request body.
     * This translates character set encodings as appropriate.
     * @exception IllegalStateException if getOutputStream has been
     *        called on this same request.
     * @exception IOException on other I/O related errors.
     * @exception UnsupportedEncodingException if the character set encoding
     *  is unsupported, so the text can't be correctly decoded.
     */
    public BufferedReader getReader() throws IOException {
        if (called_getInput) {
            throw new IllegalStateException("Already called getInputStream");
        } else if (servlet_reader == null) {
            // UnsupportedEncodingException is thrown not by
            // getCharacterEncoding, which only parses the content-type header
            // which specifies the encoding, but by the Reader constructor,
            // which possibly cannot support the encoding specified.
            String encoding =
                JServUtils.parseCharacterEncoding(getContentType());
            InputStreamReader reader =
                new InputStreamReader(servlet_in, encoding);
            servlet_reader = new BufferedReader(reader);
            got_input = true;
        }
        return servlet_reader;
    }

    //------------------------------------ Implementation of HttpServletRequest

    /**
     * Gets the array of cookies found in this request.
     * @return the array of cookies found in this request.
     */
    public Cookie[] getCookies() {
        return cookies_in;
    }

    /**
     * Gets the HTTP method (for example, GET, POST, PUT) with which
     * this request was made. Same as the CGI variable REQUEST_METHOD.
     * @return the HTTP method with which this request was made.
     */
    public String getMethod() {
        return (String) env_vars.get("REQUEST_METHOD");
    }

    /**
     * Gets this request's URI as a URL.
     * @return this request's URI as a URL.
     */
    public String getRequestURI() {
        // If the web server's version is available, use it
        String uri = (String) env_vars.get("REQUEST_URI");

        if (uri != null) {
            int queryStringOffset = uri.indexOf('?');
            // Remove any query string at the end
            if (queryStringOffset >= 0) {
                return uri.substring(0, queryStringOffset);
            } else {
                return uri;
            }
        }

        if (getPathInfo() != null) {
            return getServletPath() + getPathInfo();
        } else {
            return getServletPath();
        }
    }

    /**
     * Gets the part of this request's URI that refers to the servlet
     * being invoked. Analogous to the CGI variable SCRIPT_NAME.
     * @return the servlet being invoked, as contained in this
     * request's URI.
     */
    public String getServletPath() {
        return (String) env_vars.get("SCRIPT_NAME");
    }

    /**
     * Gets any optional extra path information following the servlet
     * path of this request's URI, but immediately preceding its query
     * string. Same as the CGI variable PATH_INFO.
     *
     * @return the optional path information following the servlet
     * path, but before the query string, in this request's URI; null
     * if this request's URI contains no extra path information.
     */
    public String getPathInfo() {
        return (String) env_vars.get("PATH_INFO");
    }

    /**
     * Gets any optional extra path information following the servlet
     * path of this request's URI, but immediately preceding its query
     * string, and translates it to a real path.  Same as the CGI
     * variable PATH_TRANSLATED.
     *
     * @return extra path information translated to a real path or null
     * if no extra path information is in the request's URI.
     */
    public String getPathTranslated() {
        return (String) env_vars.get("PATH_TRANSLATED");
    }

    /**
     * Gets any query string that is part of the servlet URI.  Same as
     * the CGI variable QUERY_STRING.
     * @return query string that is part of this request's URI, or null
     * if it contains no query string.
     */
    public String getQueryString() {
        String query = (String) env_vars.get("QUERY_STRING");
        return (query != "") ? query : null;
    }

    /**
     * Gets the name of the user making this request.  The user name is
     * set with HTTP authentication.  Whether the user name will
     * continue to be sent with each subsequent communication is
     * browser-dependent.  Same as the CGI variable REMOTE_USER.
     *
     * @return the name of the user making this request, or null if not
     * known.
     */
    public String getRemoteUser() {
        return (String) env_vars.get("REMOTE_USER");
    }

    /**
     * Gets the authentication scheme of this request.  Same as the CGI
     * variable AUTH_TYPE.
     *
     * @return this request's authentication scheme, or null if none.
     */
    public String getAuthType() {
        return (String) env_vars.get("AUTH_TYPE");
    }

    /**
     * Gets the value of the requested header field of this request.
     * The case of the header field name is ignored.
     * @param name the String containing the name of the requested
     * header field.
     * @return the value of the requested header field, or null if not
     * known.
     */
    public String getHeader(String name) {
        return (String) headers_in.get(name.toLowerCase());
    }

    /**
     * Gets the value of the specified integer header field of this
     * request.  The case of the header field name is ignored.  If the
     * header can't be converted to an integer, the method throws a
     * NumberFormatException.
     * @param name  the String containing the name of the requested
     * header field.
     * @return the value of the requested header field, or -1 if not
     * found.
     */
    public int getIntHeader(String name) {
        String hdrstr = (String) headers_in.get(name.toLowerCase());
        if (hdrstr == null) {
            return -1;
        }

        return Integer.parseInt(hdrstr);
    }

    /**
     * Gets the value of the requested date header field of this
     * request.  If the header can't be converted to a date, the method
     * throws an IllegalArgumentException.  The case of the header
     * field name is ignored.
     *
     * <PRE>  From RFC2068:
     *  3.3.1 Full Date
     *
     *
     *   HTTP applications have historically allowed three different formats
     *   for the representation of date/time stamps:
     *
     *    Sun, 06 Nov 1994 08:49:37 GMT  ; RFC 822, updated by RFC 1123
     *    Sunday, 06-Nov-94 08:49:37 GMT ; RFC 850, obsoleted by RFC 1036
     *    Sun Nov  6 08:49:37 1994       ; ANSI C's asctime() format
     *
     *   The first format is preferred as an Internet standard and
     *   represents a fixed-length subset of that defined by RFC 1123
     *   (an update to RFC 822).  The second format is in common use,
     *   but is based on the obsolete RFC 850 [12] date format and
     *   lacks a four-digit year.  HTTP/1.1 clients and servers that
     *   parse the date value MUST accept all three formats (for
     *   compatibility with HTTP/1.0), though they MUST only generate
     *   the RFC 1123 format for representing HTTP-date values in
     *   header fields
     * <pre>
     * @param name  the String containing the name of the requested
     * header field.
     * @return the value the requested date header field, or -1 if not
     * found.
     */
    public long getDateHeader(String name) {
        String val = (String) headers_in.get(name.toLowerCase());
        SimpleDateFormat sdf;

        if ( val == null ) {
            return -1;
        }

        sdf = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
        try {
            Date date = sdf.parse(val);
            return date.getTime();
        } catch(ParseException formatNotValid) {
            // try another format
        }

        sdf = new SimpleDateFormat("EEEEEE, dd-MMM-yy HH:mm:ss zzz");
        try {
            Date date = sdf.parse(val);
            return date.getTime();
        } catch(ParseException formatNotValid) {
            // Try another format
        }

        sdf = new SimpleDateFormat("EEE MMMM d HH:mm:ss yyyy");
        try {
            Date date = sdf.parse(val);
            return date.getTime();
        } catch(ParseException formatStillNotValid) {
            throw new IllegalArgumentException(val);
        }
    }

    /**
     * Gets the header names for this request.
     * @return an enumeration of strings representing the header names
     * for this request. Some server implementations do not allow
     * headers to be accessed in this way, in which case this method
     * will return null.
     */
    public Enumeration getHeaderNames() {
        return headers_in.keys();
    }

    /**
     * Gets the current valid session associated with this request, if
     * create is false or, if necessary, creates a new session for the
     * request, if create is true.
     *
     * <p><b>Note</b>: to ensure the session is properly maintained,
     * the servlet developer must call this method (at least once)
     * before any output is written to the response.
     *
     * <p>Additionally, application-writers need to be aware that newly
     * created sessions (that is, sessions for which
     * <code>HttpSession.isNew</code> returns true) do not have any
     * application-specific state.
     *
     * @return the session associated with this request or null if
     * create was false and no valid session is associated
     * with this request.
     */
    public HttpSession getSession( boolean create ) {

        // FIXME: What happen if someone calls invalidate on a new session
        // and calls this method again ? Should we create another one, or
        // return the invalid one ?

        if (session != null) {
          return session;
        }

        if (requestedSessionId != null) {
          session = (JServSession) mgr.getSession(requestedSessionId);
          if (session != null) {
            return session;
          }
        }

        if (create == true) {
          session = mgr.createSession(this);
          return session;
        }

        return null;
    }

    /**
     * Gets the session id specified with this request.  This may
     * differ from the actual session id.  For example, if the request
     * specified an id for an invalid session, then this will get a new
     * session with a new id.
     *
     * @return the session id specified by this request, or null if the
     * request did not specify a session id.
     * @see #isRequestedSessionIdValid()
     */
    public String getRequestedSessionId() {
        return requestedSessionId;
    }

    /**
     * Checks whether this request is associated with a session that
     * is valid in the current session context.  If it is not valid,
     * the requested session will never be returned from the
     * <code>getSession</code> method.
     * @return true if this request is assocated with a session that is
     * valid in the current session context.
     * @see #getRequestedSessionId()
     */
    public boolean isRequestedSessionIdValid() {
        if (requestedSessionId == null) {
            return false;
        } else {
            return mgr.getSession(requestedSessionId) != null;
        }
    }

    /**
     * Checks whether the session id specified by this request came in
     * as a cookie.  (The requested session may not be one returned by
     * the <code>getSession</code> method.)
     * @return true if the session id specified by this request came in
     * as a cookie; false otherwise.
     * @see #getSession( boolean )
     */
    public boolean isRequestedSessionIdFromCookie() {
        return idCameAsCookie;
    }

    /**
     * Checks whether the session id specified by this request came in
     * as part of the URL.  (The requested session may not be the one
     * returned by the <code>getSession</code> method.)
     * @return true if the session id specified by the request for this
     * session came in as part of the URL; false otherwise.
     * @see #getSession( boolean )
     */
    public boolean isRequestedSessionIdFromUrl() {
        return idCameAsUrl;
    }

    //--------------------------------------- Implementation of ServletResponse

    /**
     * Sets the content length for this response.
     * @param len the content length.
     */
    public void setContentLength(int len) {
        Integer length = new Integer(len);
        headers_out.put("Content-Length", length.toString());
    }

    /**
     * Sets the content type for this response.  This type may later
     * be implicitly modified by addition of properties such as the MIME
     * <em>charset=&lt;value&gt;</em> if the service finds it necessary,
     * and the appropriate media type property has not been set.  This
     * response property may only be assigned one time.
     * @param type the content's MIME type
     */
    public void setContentType(String type) {
        headers_out.put("Content-Type", type);
    }

    /**
     * Returns an output stream for writing binary response data.
     * @exception IllegalStateException if getWriter has been
     *      called on this same request.
     * @exception IOException if an I/O exception has occurred.
     * @see #getWriter
     */
    public ServletOutputStream getOutputStream() throws IOException {
        if ( servlet_writer != null ) {
            throw new IllegalStateException( "Already called getWriter" );
        } else {
            called_getOutput = true;
            return servlet_out;
        }
    }

    /**
     * Returns a print writer for writing formatted text responses.  The
     * MIME type of the response will be modified, if necessary, to reflect
     * the character encoding used, through the <em>charset=...</em>
     * property.  This means that the content type must be set before
     * calling this method.
     * @exception IllegalStateException if getOutputStream has been
     *        called on this same request.
     * @exception IOException on other errors.
     * @exception UnsupportedEncodingException if the character set encoding
     * @see #getOutputStream
     * @see #setContentType
     */
    public PrintWriter getWriter() throws IOException {
        if (called_getOutput) {
            throw new IllegalStateException("Already called getOutputStream.");
        } else if (servlet_writer == null) {
            // UnsupportedEncodingException is thrown not by
            // getCharacterEncoding, which only parses the content-type header
            // which specifies the encoding, but by the Writer constructor,
            // which possibly cannot support the encoding specified.
            OutputStreamWriter out =
                new OutputStreamWriter(servlet_out, getCharacterEncoding());
            servlet_writer = new PrintWriter(out);
        }
        return servlet_writer;
    }

    /**
     * Returns the character set encoding used for this MIME body.
     * The character encoding is either the one specified in the
     * assigned content type, or one which the client understands.
     * If no content type has yet been assigned, it is implicitly
     * set to <em>text/plain</em>
     */
    public String getCharacterEncoding() {
        String contentType = (String)headers_out.get("Content-Type");
        if (contentType == null) {
            contentType = "text/plain";
            setContentType(contentType);
        }

        return JServUtils.parseCharacterEncoding(contentType);
    }

    //--------------------------------- Implementation of HttpServletResponse

    /**
     * Adds the specified cookie to the response.  It can be called
     * multiple times to set more than one cookie.
     * @param cookie the Cookie to return to the client
     */
    public void addCookie(Cookie cookie) {
        cookies_out.addElement(cookie);
    }

    /**
     * Checks whether the response message header has a field with
     * the specified name.
     * @param name the header field name.
     * @return true if the response message header has a field with
     * the specified name; false otherwise.
     */
    public boolean containsHeader(String name) {
        return headers_out.contains(name.toLowerCase());
    }

    /**
     * Sets the status code and message for this response.  If the
     * field had already been set, the new value overwrites the
     * previous one.  The message is sent as the body of an HTML
     * page, which is returned to the user to describe the problem.
     * The page is sent with a default HTML header; the message
     * is enclosed in simple body tags (&lt;body&gt;&lt;/body&gt;).
     * @param sc the status code.
     * @param sm the status message.
     */
    public void setStatus(int sc, String sm) {
        status = sc;
        status_string = sm;
    }

    /**
     * Sets the status code for this response.  This method is used to
     * set the return status code when there is no error (for example,
     * for the status codes SC_OK or SC_MOVED_TEMPORARILY).  If there
     * is an error, the <code>sendError</code> method should be used
     * instead.
     * @param sc the status code
     * @see #sendError
     */
    public void setStatus(int sc) {
        setStatus(sc, null);
    }

    /**
     * Adds a field to the response header with the given name and value.
     * If the field had already been set, the new value overwrites the
     * previous one.  The <code>containsHeader</code> method can be
     * used to test for the presence of a header before setting its
     * value.
     * @param name the name of the header field
     * @param value the header field's value
     * @see #containsHeader
     */
    public void setHeader(String name, String value) {
        int offset_of_newline;

        // We need to make sure no newlines are present in the header:
        if ((offset_of_newline = value.indexOf((int)'\n')) > 0) {
            char msgAsArray[] = value.toCharArray();
            msgAsArray[offset_of_newline] = ' ';

            while ((offset_of_newline =
                value.indexOf((int)'\n',offset_of_newline+1)) > 0) {
                msgAsArray[offset_of_newline] = ' ';
            }
            value = new String(msgAsArray);
        }

        headers_out.put(name, value);
    }

    /**
     * Adds a field to the response header with the given name and
     * integer value.  If the field had already been set, the new value
     * overwrites the previous one.  The <code>containsHeader</code>
     * method can be used to test for the presence of a header before
     * setting its value.
     * @param name the name of the header field
     * @param value the header field's integer value
     * @see #containsHeader
     */
    public void setIntHeader(String name, int value) {
        Integer val = new Integer(value);
        headers_out.put(name, val.toString());
    }

    /**
     * Adds a field to the response header with the given name and
     * date-valued field.  The date is specified in terms of
     * milliseconds since the epoch.  If the date field had already
     * been set, the new value overwrites the previous one.  The
     * <code>containsHeader</code> method can be used to test for the
     * presence of a header before setting its value.
     * @param name the name of the header field
     * @param value the header field's date value
     * @see #containsHeader
     */
    public void setDateHeader(String name, long date) {
        SimpleDateFormat sdf =
            new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
        TimeZone tz = TimeZone.getTimeZone("GMT");

        sdf.setTimeZone(tz);

        headers_out.put(name, sdf.format(new Date(date)));
    }

    /**
     * Sends an error response to the client using the specified status
     * code and descriptive message.  If setStatus has previously been
     * called, it is reset to the error status code.  The message is
     * sent as the body of an HTML page, which is returned to the user
     * to describe the problem.  The page is sent with a default HTML
     * header; the message is enclosed in simple body tags
     * (&lt;body&gt;&lt;/body&gt;).
     * @param sc the status code
     * @param msg the detail message
     */
    public void sendError(int sc, String msg) {
        try {
            // Tell Apache to send an error
            status = sc;
            setHeader("Servlet-Error", msg);
            sendHttpHeaders();

            // Flush and close, so the error can be returned right
            // away, and so any additional data sent is ignored
            out.flush();
            client.close();
        } catch (IOException e) {
            // Not much more we can do...
            if (log.active) log.log(e);
        }
    }

    /**
     * Sends an error response to the client using the specified
     * status code and a default message.
     * @param sc the status code
     */
    public void sendError(int sc) {
        sendError(sc, findStatusString( sc ) );
    }

    /**
     * JServSendError method. This sends an error message to Apache
     * when an exception occur in the ServletEngine.
     */
    public void sendError(Throwable e) {
        sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
            e.toString() + ": " + e.getMessage());
        if (log.active) log.log(e);
    }

    /**
     * Sends a temporary redirect response to the client using the
     * specified redirect location URL.  The URL must be absolute (for
     * example, <code><em>https://hostname/path/file.html</em></code>).
     * Relative URLs are not permitted here.
     * @param location the redirect location URL
     * @exception IOException If an I/O error has occurred.
     */
    public void sendRedirect(String location) throws IOException {
        // We use Apache's internal status mechanism: set the status to
        // 200, with a Location: header and no body.

        setStatus(SC_OK);
        setHeader("Location", location);
        sendHttpHeaders();
    }

    /**
     * Encodes the specified URL by including the session ID in it,
     * or, if encoding is not needed, returns the URL unchanged.
     * The implementation of this method should include the logic to
     * determine whether the session ID needs to be encoded in the URL.
     * For example, if the browser supports cookies, or session
     * tracking is turned off, URL encoding is unnecessary.
     *
     * <p>All URLs emitted by a Servlet should be run through this
     * method.  Otherwise, URL rewriting cannot be used with browsers
     * which do not support cookies.
     *
     * @param url the url to be encoded.
     * @return the encoded URL if encoding is needed; the unchanged URL
     * otherwise.
     */
    public String encodeUrl(String url) {
        // Encode only if there is no cookie support
        // and there is a valid session associated with this request
        if (idCameAsCookie) {
            return url;
        } else if (session == null) {
            return url;
        } else {
            return JServServletManager.encodeUrl(url, session.id);
        }
    }

    /**
     * Encodes the specified URL for use in the
     * <code>sendRedirect</code> method or, if encoding is not needed,
     * returns the URL unchanged.  The implementation of this method
     * should include the logic to determine whether the session ID
     * needs to be encoded in the URL.  Because the rules for making
     * this determination differ from those used to decide whether to
     * encode a normal link, this method is seperate from the
     * <code>encodeUrl</code> method.
     *
     * <p>All URLs sent to the HttpServletResponse.sendRedirect
     * method should be run through this method.  Otherwise, URL
     * rewriting canont be used with browsers which do not support
     * cookies.
     *
     * @param url the url to be encoded.
     * @return the encoded URL if encoding is needed; the unchanged URL
     * otherwise.
     * @see #sendRedirect
     * @see #encodeUrl
     */
    public String encodeRedirectUrl(String url) {
        // Encode only if there is a session associated to the request
        // And if the redirection will come back here
        if (session == null) {
            return url;
        } else if (url.indexOf(hostname) == -1) {
            return url;
        } else {
            return JServServletManager.encodeUrl(url, session.id);
        }
    }

    /**
     * ServletInputStream implementation as inner class
     */
    private class JServInputStream extends ServletInputStream {

        // bytes remaining to be read from the input stream. This is
        // initialized from CONTENT_LENGTH (or getContentLength()).
        // This is used in order to correctly return a -1 when all the
        // data POSTed was read.
        long available = -1;

                private InputStream in;

        public JServInputStream(InputStream in) {
                this.in = in;
            available = getContentLength();
        }

        public int read() throws IOException {
            if (available > 0) {
                available--;
                return in.read();
            }
            return -1;
        }

        public int read(byte b[]) throws IOException {
            return read(b, 0, b.length);
        }

        public int read(byte b[], int off, int len) throws IOException {
            if (available > 0) {
                if (len > available) {
                    // shrink len
                    len = (int) available;
                }
                int read = in.read(b, off, len);
                if (read != -1) {
                    available -= read;
                } else {
                    available = -1;
                }
                return read;
            }
            return -1;
        }

        public long skip(long n) throws IOException {
            long skip = in.skip(n);
            available -= skip;
            return skip;
        }

        public void close() throws IOException {
            // Ignore closing of the input stream since it also
            // close the output stream.
            // conn.in.close();
        }
    }

    /**
	 * ServletOutputStream implementation as inner class
	 */
    class JServOutputStream extends ServletOutputStream {
                private OutputStream out;

		public JServOutputStream(OutputStream out) {
	                this.out = out;
		}

		public void write(int b) throws IOException {
			sendHttpHeaders();
			out.write(b);
		}

		public void write(byte b[], int off, int len) throws IOException {
			sendHttpHeaders();
			out.write(b, off, len);
		}

		public void flush() throws IOException {
			sendHttpHeaders();
			out.flush();
		}

		public void close() throws IOException {
			sendHttpHeaders();
			out.close();
		}
	}
}
