// ====================================================================
// Copyright (c) 1997 The Apache Group.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
//
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer. 
//
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in
//    the documentation and/or other materials provided with the
//    distribution.
//
// 3. All advertising materials mentioning features or use of this
//    software must display the following acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// 4. The names "Apache Server" and "Apache Group" must not be used to
//    endorse or promote products derived from this software without
//    prior written permission.
//
// 5. Redistributions of any form whatsoever must retain the following
//    acknowledgment:
//    "This product includes software developed by the Apache Group
//    for use in the Apache HTTP server project (http://www.apache.org/)."
//
// THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
// EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
// ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
// OF THE POSSIBILITY OF SUCH DAMAGE.
// ====================================================================
//
// This software consists of voluntary contributions made by many
// individuals on behalf of the Apache Group and was originally based
// on public domain software written at the National Center for
// Supercomputing Applications, University of Illinois, Urbana-Champaign.
// For more information on the Apache Group and the Apache HTTP server
// project, please see <http://www.apache.org/>.

// JServ - Serve up Java servlets
// by Alexei Kosut <akosut@apache.org>

// Parts are based on examples from  _Java in a Nutshell_ by David Flanagan:
// Written by David Flanagan.  Copyright (c) 1996 O'Reilly & Associates.
// You may study, use, modify, and distribute this example for any purpose.
// This example is provided WITHOUT WARRANTY either expressed or implied.

// JServHandler.java:
// - apache.jserv.JServHandler

package apache.jserv;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

import javax.servlet.*;
import javax.servlet.http.*;

// The main class. This sets up the server, initalizes everything, and listens
// on a port. When it gets a request, it launches a JServConnection thread.

public class JServHandler extends Thread implements JServSendError {
    public final static String version = "JServ/0.9.5";

    public final static int DEFAULT_PORT = 8007;
    protected static int port;
    protected ServerSocket listen_socket;

    private static BufferedReader in;

    protected Properties properties;
    protected Hashtable servlets;

    protected static String startups;

    protected static String auth = null;
    protected static boolean stackt = false;

    // Start the server up, listening on the specified port
    public static void main(String[] args) {
	String propfile = null;
	String auth = null;
	boolean noauth = false;
        int argn, port = 0;

	try {
	    for (argn = 0; argn < args.length; argn++) {
		if (args[argn].equals("-m"))
		    noauth = true;
		else if (args[argn].equals("-t"))
		    stackt = true;
		else
		    break;
	    }
	  
	    propfile = args[argn++];
	    
	    try { port = Integer.parseInt(args[argn++]);  }
	    catch (NumberFormatException e) {
		throw new ArrayIndexOutOfBoundsException();
	    }
	}
	catch (ArrayIndexOutOfBoundsException e) {
	    System.err.println("JServHandler: illegal arguments");
	    System.err.println("usage: java apache.jserv.JServHandler" +
			       " [-m] [-t] propfile port");
	    System.exit(1);
	}
	    
	if (!noauth) {
	    // Read the auth from the first line of "stdin"
	    try {
		// Read until newline
		byte a[] = new byte[25];
		int i;

		for (i = 0; i < 25; i++) {
		    int b = System.in.read();

		    if (b == -1 || b == '\n')
			break;

		    a[i] = (byte)b;
		}

		auth = new String(a, 0, i);
	    } catch (IOException e) {
		fail(e, "Exception while reading authentication");
	    }
	}
	
        new JServHandler(propfile, port, auth);
    }

    // Exit with an error message, when an exception occurs.
    public static void fail(Exception e, String msg) {
	System.err.println(msg + ": " +  e);
        System.exit(1);
    }
    
    // Create a ServerSocket to listen for connections on;  start the thread.
    public JServHandler(String propfile, int port, String auth) {
	// Read in the server properties
	properties = new Properties();

	File propf = new File(propfile);

	if (propf.exists() && propf.canRead()) {
	    try {
		properties.load(new FileInputStream(propf));
	    } catch (IOException e) {
		properties.clear();	/* Reset as empty */
	    }
	}

	// Auth string
	this.auth = auth;

	// Set up the servlet hashtable
	servlets = new Hashtable();

	// Load startup servlets
	startups = properties.getProperty("servlets.startup");
	loadStartupServlets(servlets, properties, this);

        if (port == 0) port = DEFAULT_PORT;
        this.port = port;

	// Wait for signals
	new JServSignals(servlets, properties);

        try { listen_socket = new ServerSocket(port); }
        catch (IOException e) { fail(e, "Exception creating server socket"); }

        this.start();
    }
    
    // The body of the server thread.  Loop forever, listening for and
    // accepting connections from clients.  For each connection, 
    // create a JServConnection object to handle communication through the
    // new Socket.
    public void run() {
        try {
            while(true) {
                Socket client_socket = listen_socket.accept();

		JServConnection c = new
		    JServConnection(client_socket, properties, servlets);
            }
        }
        catch (IOException e) { 
            fail(e, "Exception while listening for connections");
        }
    }

    // To deal with errors received when loading servlets
    public void sendError(int sc, String msg) {
	System.err.println("JServ: " + msg);
    }
    
    // Static method to load the startup servlets
    protected static void loadStartupServlets(Hashtable servlets,
					      Properties properties,
					      JServSendError se) {
	if (startups != null) {
	    StringTokenizer st = new StringTokenizer(startups, " ");
	    
	    try {
		while (st.hasMoreTokens()) {
		    String servname = st.nextToken();
		    
		    if (servname == null)
			continue;
		    
		    loadServlet(servname, servlets, properties, se);
		}
	    } catch (NoSuchElementException e) {}
	}	
    }

    // Static method to add a servlet to the servlets hashtable
    // Note that this does not call the init methods.

    protected static JServContext loadServlet(String classname,
					      Hashtable servlets,
					      Properties properties,
					      JServSendError se) {
	synchronized (servlets) {
	    JServContext context = (JServContext)servlets.get(classname);
	    Servlet servlet;
	    
	    if (context != null)
		return context;

	    // Find the servlet's full name if an alias
	    String sdname = "servlet." + classname;
	    String fullname = properties.getProperty(sdname + ".code");
	    
	    if (fullname == null)
		fullname = sdname = classname;
	    
	    // Load the servlet
	    try {
		servlet = (Servlet)Class.forName(fullname).newInstance();
	    } catch (ClassNotFoundException e) {
		try {
		    se.sendError(HttpServletResponse.SC_NOT_FOUND, "Class " +
				 fullname + " could not be found");
		} catch (IOException ie) {}
		return null;
	    } catch (IllegalAccessException e) {
		try {
		    se.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
				 "Caught IllegalAccessException when loading "
				 + fullname + ": " + e.getMessage());
		} catch (IOException ie) {}
		return null;
	    } catch (InstantiationException e) {
		try {
		    se.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
				 "Caught InstantiationException when loading "
				 + fullname + ": " + e.getMessage());
		} catch (IOException ie) {}
		return null;
	    }
	    
	    // Setup the init parameters
	    
	    String argsline = properties.getProperty(sdname + ".initArgs");
	    Properties initargs = parseInitArgs(argsline);
	    
	    // Init the servlet
	    try {
		context = new JServContext(servlet, servlets, initargs);
		servlet.init(context);
	    } catch (Exception e) {
		// Something happened...
		try {
		    se.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
				 "Exception caught in " + classname +
				 ".init(): " + e);
		    if (stackt) e.printStackTrace();
		} catch (IOException ie) {}
		return null;
	    } catch (Error e) {
		// Something really bad happened...
		try {
		    se.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
				 "Error caught in " + classname +
				 ".init(): " + e);
		    
		    if (stackt) e.printStackTrace();
		} catch (IOException ie) {}
		return null;
	    }
	    
	    // Add the servlet to servlets, so we can use it again
	    servlets.put(classname, context);
	    return context;
	}
    }

    // Convert initArgs line into Properties
    // "<name=value>,<name=value>"
    private static Properties parseInitArgs(String argsline) {
	Properties initargs = new Properties();
	if (argsline == null)
	    return initargs;

	StringTokenizer stc = new StringTokenizer(argsline, ",");

	try {
	    while (stc.hasMoreTokens()) {
		StringTokenizer ste =
		    new StringTokenizer(stc.nextToken(), "=");
		String key, val;
		
		key = ste.nextToken();
		val = ste.nextToken();
		
		initargs.put(key, val);
	    }
	} catch (NoSuchElementException e) {}
	
	return initargs;
    }

}
