/*
 * Copyright (c) 1997-1999 The Java Apache Project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *
 * 4. The names "Apache JServ", "Apache JServ Servlet Engine" and 
 *    "Java Apache Project" must not be used to endorse or promote products 
 *    derived from this software without prior written permission.
 *
 * 5. Products derived from this software may not be called "Apache JServ"
 *    nor may "Apache" nor "Apache JServ" appear in their names without 
 *    prior written permission of the Java Apache Project.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Java Apache 
 *    Project for use in the Apache JServ servlet engine project
 *    (http://java.apache.org/)."
 *    
 * THIS SOFTWARE IS PROVIDED BY THE JAVA APACHE PROJECT "AS IS" AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE JAVA APACHE PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Java Apache Group. For more information
 * on the Java Apache Project and the Apache JServ Servlet Engine project,
 * please see <http://java.apache.org/>.
 *
 */
/*
 * mod_jserv.c: Implements part of Java Server API for Apache
 * by the Apache JServ group, parts based on mod_alias.c and mod_cgi.c
 *
 * For more information, see http://java.apache.org/
 */

#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_main.h"
#include "http_protocol.h"
#include "util_script.h"
#include "http_conf_globals.h"
/* This is for Apache 1.3b6 compatibility */
#if MODULE_MAGIC_NUMBER >= 19980413 
#if MODULE_MAGIC_NUMBER >= 19980713
#include "ap_compat.h"
#else
#include "compat.h"
#endif
#endif

/* Name of the class to run - not configurable */
#define JSERV_CLASS "org.apache.jserv.JServHandler"

/* Some defaults */
#define JSERV_DEFAULT_PORT 8007
#define JSERV_DEFAULT_SERVER inet_addr("127.0.0.1")
#define JSERV_DEFAULT_PROPERTIES "conf/servlets.properties"
#define JSERV_AUTH_CONF_SIZE 10

#define CP_SEP ":"

/* This is taken from mod_log_config */

static int xfer_flags = (O_WRONLY | O_APPEND | O_CREAT);
#if defined(__EMX__) || defined(WIN32)
/* OS/2 dosen't support users and groups */
static mode_t xfer_mode = (S_IREAD | S_IWRITE);
#else
static mode_t xfer_mode = (S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
#endif

/* this is to avoid spawn/die cycles */
static time_t last_restart_time = 0;
static int    failed_restart_count = 0;

/* this is used for the load balancing code */
typedef struct jserv_vm_host jserv_vm_host;

struct jserv_vm_host {
    unsigned long remote_server;
    int port;
    unsigned long requests;
    unsigned long total_time;
    jserv_vm_host *next;
};

typedef struct {
    char *real;
    char *fake;
    int is_dir;
} jserv_entry;

typedef struct {
    array_header *dirs;
    array_header *envs;
    array_header *vhost;	/* Only set in the base host configuration
				 * contains pointer to the virtual host
				 * information to pass to java process.
				 */
    server_rec *host;		/* The server record for wich we are configured */
    char *classpath;
    char *gen_properties_file;
    int add_path_env, add_tz_env;

    char *binary;
    array_header *binary_args;	/* Extra argument on the commandline */

    char *properties;
    char *errlog;
    int port;
    unsigned long remote_server;

    jserv_vm_host *vm_hosts;  /* linked list of VMs to use (load balancing) */

    int auth_export;
    int manual;
    char * servlet_manager_hostname;
} jserv_conf;

module jserv_module;

/* Holds info for the signal handler code, so it'll know how to shut down
 * jserv. */
static jserv_conf jserv_config;

void jserv_find_vm(struct sockaddr_in *addr, jserv_conf *conf, request_rec *r);
void jserv_next_vm(struct sockaddr_in *addr, jserv_conf *conf);

void *create_jserv_config(pool * p, server_rec * s)
{
    jserv_conf *j = (jserv_conf *) pcalloc(p, sizeof(jserv_conf));
    time_t now;
    unsigned long bits1, bits2;

    j->dirs = make_array(p, 20, sizeof(jserv_entry));
    j->envs = make_array(p, 5, sizeof(char *));
    j->vhost = make_array(p, 10, sizeof(jserv_conf *));

    /* We use this to log errors, and (dep. on configuration) for
     * the hostname.
     */
    j->host = s;

    j->classpath = NULL;
    j->gen_properties_file = NULL;

    j->add_path_env = 1;
    j->add_tz_env = 1;

    j->binary = NULL;
    j->binary_args = make_array(p, 10, sizeof(char *));

    j->properties = NULL;
    j->errlog = NULL;
    j->port = JSERV_DEFAULT_PORT;
    j->remote_server = JSERV_DEFAULT_SERVER;

    j->vm_hosts = NULL;

    j->auth_export = 0;
    j->manual = 0;
    j->servlet_manager_hostname = NULL;

    return j;
}

/** Merge per-virtual-host configuration information with the default
 * configuration for all hosts. 
 *
 * Returns: a pointer to a jserv_conf structure containing the 
 * results of the merge.
 **/
void *merge_jserv_config(pool * p, void *basev, void *overridesv)
{
    jserv_conf *j = (jserv_conf *) pcalloc(p, sizeof(jserv_conf));
    jserv_conf *base = (jserv_conf *) basev;
    jserv_conf *overrides = (jserv_conf *) overridesv;

    /* Put a pointer to the virtual host configuration in the base server */
    jserv_conf **new = (jserv_conf **) push_array(base->vhost);
    *new = j;

    /* In general, virtual directories can override the options
      * specified as the default. */
    j->dirs = overrides->dirs ? overrides->dirs : base->dirs;
    j->properties = (overrides->properties ? 
		     overrides->properties : base->properties );
    j->host = overrides->host ? overrides->host : base->host;
    j->servlet_manager_hostname = overrides->servlet_manager_hostname ?
                                  overrides->servlet_manager_hostname :
                                  base->servlet_manager_hostname;
 
    /* However, Apache runs all servlets from all virtual hosts and
      * repositories in a single JVM, so it doesn't make sense to let
      * people override these properties in a VHost configuration.  */
    j->add_path_env = base->add_path_env;
    j->add_tz_env = base->add_tz_env;
    j->binary = base->binary;
    j->binary_args = base->binary_args;
    j->auth_export = base->auth_export;
    j->port = base->port;
    j->classpath = base->classpath;
    j->gen_properties_file = base->gen_properties_file;
    j->envs =  base->envs;
    j->errlog = base->errlog;

    return j;
}

const char *add_jserv_dir(cmd_parms * cmd, void *dummy, char *f, char *r)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);
    jserv_entry *new = push_array(conf->dirs);
    struct stat finfo;

    if (stat(r, &finfo) == -1)
	return pstrcat(cmd->pool, "\"", r, "\" does not exist", NULL);

    new->fake = f;
    new->real = r;
    new->is_dir = S_ISDIR(finfo.st_mode);

    return NULL;
}

const char *set_jserv_gen_properties(cmd_parms * cmd, void * dummy, char * path)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);
    conf->gen_properties_file = server_root_relative(cmd->pool, path);
    return NULL;
}


const char *set_jserv_classpath(cmd_parms * cmd, void *dummy, char *path)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    if (conf->classpath)
	conf->classpath = pstrcat(cmd->pool, path, CP_SEP, conf->classpath,
				  NULL);
    else
	conf->classpath = path;

    return NULL;
}

const char *set_jserv_bin(cmd_parms * cmd, void *dummy, char *bin)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    conf->binary = bin;

    return NULL;
}

const char *set_jserv_properties(cmd_parms * cmd, void *dummy, char *props)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    conf->properties = server_root_relative(cmd->pool, props);

    return NULL;
}

const char *set_servlet_host (cmd_parms * cmd, void *dummy, char *host) {
    return "ServletHost is deprecated; use ServletVMHost to specify the machine on which the VM is running.";
}

const char *set_servlet_vm_host (cmd_parms * cmd, void * dummy, char * host) {
    server_rec *s = cmd->server;
    pool *p = cmd->pool;
    jserv_conf *conf =
        (jserv_conf *)get_module_config(s->module_config,&jserv_module);

    jserv_vm_host *vm_host, *cur_host;
    
    struct hostent *hp;
    u_long ipaddr;
    int i;
    char *c;
    int port = conf->port;

    if (!host)
        return "you must provide a hostname in the ServletVMHost directive";

    /* check if value contains a colon (for the port) */
    c = strchr(host, ':');
    if(c != NULL && *(c + 1) != '\0') {
      *c = '\0';
      port = atoi(++c);
    }

    for (i=0; host[i] != '\0'; i++)
        if (!isdigit(host[i]) && host[i] != '.')
            break;

    if (host[i] != '\0')
    {
        hp = gethostbyname(host);
        if (hp == NULL)
            return "Host for ServletVMHost not found";
        ipaddr = ((struct in_addr*)hp->h_addr_list[0])->s_addr;
    } else
    {
        ipaddr = inet_addr(host);
    }
    
    vm_host = (jserv_vm_host*)pcalloc(p, sizeof(jserv_vm_host));
    
    vm_host->remote_server = ipaddr;
    vm_host->port = port;
    vm_host->requests = 0;
    vm_host->total_time = 0;
    
    if(conf->vm_hosts) {
      /* insert at the end of the list */
      cur_host = conf->vm_hosts->next;
      while(cur_host->next != conf->vm_hosts) cur_host = cur_host->next;
      
      cur_host->next = vm_host;
      vm_host->next = conf->vm_hosts;
    }
    else {
      /* start list. point at ourself */
      vm_host->next = vm_host;
      conf->vm_hosts = vm_host;
    }

    return NULL;
}

const char *set_servlet_mgr_hostname (cmd_parms * cmd, void * dummy,
                                      char * name) {
    server_rec *s = cmd->server;
    jserv_conf *conf =
        (jserv_conf *)get_module_config(s->module_config,&jserv_module);

    if (!name)
        return "you must provide a name in the ServletManagerHostName directive";
    
    conf->servlet_manager_hostname = pstrdup(cmd->pool, name);
    return NULL;
}

const char *set_jserv_errlog(cmd_parms * cmd, void *dummy, char *fname)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    conf->errlog = fname;

    return NULL;
}

const char *set_jserv_env(cmd_parms * cmd, void *dummy, char *varname, char *value)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);
    char **new = (char **) push_array(conf->envs);

    if (value == NULL)
        value = getenv(varname);
    *new = pstrcat(cmd->pool, varname, "=", value, NULL); 

    if (strcmp(varname, "PATH") == 0)
        conf->add_path_env = 0;
    else if (strcmp(varname, "TZ") == 0)
        conf->add_tz_env = 0;

    return NULL;
}

const char *set_jserv_arg(cmd_parms * cmd, void *dummy, char *arg)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);
    char **new = (char **) push_array(conf->binary_args);

    *new = arg;

    return NULL;
}

const char *set_jserv_port(cmd_parms * cmd, void *dummy, char *port)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    conf->port = atoi(port);

    return NULL;
}

const char *set_jserv_manual(cmd_parms * cmd, void *dummy, int arg)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    conf->manual = arg;

    return NULL;
}

const char *set_auth_export(cmd_parms * cmd, void *dummy, int arg)
{
    server_rec *s = cmd->server;
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);

    conf->auth_export = arg;

    return NULL;
}

command_rec jserv_cmds[] =
{
    {"ServletAlias", add_jserv_dir, NULL, RSRC_CONF, TAKE2,
     "a URL path prefix and a directory which stores class files"},
    {"ServletAuthExport", set_auth_export, NULL, RSRC_CONF, FLAG,
     "flag whether mod_jserv may let other modules access confidential info"},
    {"ServletBinary", set_jserv_bin, NULL, RSRC_CONF, TAKE1,
     "the path to the Java runtime binary"},
    {"ServletGeneralProperties", set_jserv_gen_properties, NULL, RSRC_CONF,
     TAKE1, "the path for the general properties file for JServ"},
    {"ServletClassPath", set_jserv_classpath, NULL, RSRC_CONF, TAKE1,
     "a Java class path to search"},
    {"ServletEnvironment", set_jserv_env, NULL, RSRC_CONF, TAKE12,
     "A list of environment variables to pass to JServ"},
    {"ServletBinaryArgument", set_jserv_arg, NULL, RSRC_CONF, ITERATE,
     "A list of arguments to give to the Java interpreter."},
    {"ServletErrorLog", set_jserv_errlog, NULL, RSRC_CONF, TAKE1,
     "a filename to log System.err output from servlets to"},
    {"ServletManual", set_jserv_manual, NULL, RSRC_CONF, FLAG,
     "whether or not to run JServHandler manually"},
    {"ServletPort", set_jserv_port, NULL, RSRC_CONF, TAKE1,
     "the port to run the Java servlet server on"},
    {"ServletProperties", set_jserv_properties, NULL, RSRC_CONF, TAKE1,
     "the filename that holds the Java servlet properties"},
    {"ServletHost", set_servlet_host, NULL, RSRC_CONF, TAKE1,
     "deprecated function: replaced by ServletVMHost"},
    {"ServletVMHost", set_servlet_vm_host, NULL, RSRC_CONF, TAKE1,
     "the name of the host where JServ is running, if it's not local"},
    {"ServletManagerHostName", set_servlet_mgr_hostname, NULL, RSRC_CONF, TAKE1,
     "the name for the servlet manager which corresponds to this virtual host"},
    {NULL}
};

/* for the sake of simplicity, supply dummy values for Apache 1.2 */

#if MODULE_MAGIC_NUMBER < 19970831
#define APLOG_MARK "dummy", 1
#define APLOG_ERR 1
#define APLOG_NOTICE 2
#define ap_user_name user_name
#define ap_group_id group_id
#define ap_group_id_list group_id_list
#define ap_table_get table_get
#define ap_pstrcat pstrcat
#define ap_pstrdup pstrdup
#define ap_pstrndup pstrndup
#endif

void jserv_log_error(const char *file, int line, int level,
                     const server_rec *s, const char * msg) {
#if MODULE_MAGIC_NUMBER >= 19970831
    int save_errno = errno;  /* don't log info about an unrelated event */
    errno = 0;           
#if MODULE_MAGIC_NUMBER >= 19980413
    ap_log_error(APLOG_MARK, APLOG_ERR, s, msg);
#else /* after change in the function, before the great symbol renaming */
    aplog_error(APLOG_MARK, APLOG_ERR, s, msg);
#endif
    errno = save_errno;
#else /* old style -- the casts avoid a compiler error; log_error only needs
       * const pointers, but the prototype doesn't say so in Apache 1.2.x */
    log_error((char *)msg, (server_rec *)s);
#endif
}

typedef struct {
    jserv_conf *conf;
    server_rec *s;
    pool *p;
    int log_fd;
} jserv_child_conf;

/* Child #2 - This is created by child #1 */

#if MODULE_MAGIC_NUMBER < 19970623
static void jserv_child_child(void *data)
#elif MODULE_MAGIC_NUMBER < 19980519
static int jserv_child_child(void *data)
#else
static int jserv_child_child(void *data, child_info *child)
#endif
{
    jserv_conf *conf = ((jserv_child_conf *) data)->conf;
    server_rec *s = ((jserv_child_conf *) data)->s;
    pool *p = ((jserv_child_conf *) data)->p;
    int i, log_fd = ((jserv_child_conf *) data)->log_fd;
    int j;

    char port[10];
    char **env;
    char **args;
    int do_debugging = (failed_restart_count > 6);


    if (log_fd >= 0) {
	args = malloc((sizeof(char *)) * (5 + conf->binary_args->nelts));
    }
    else {
	args = malloc((sizeof(char *)) * (4 + conf->binary_args->nelts));
    }

    /** 
     * Set command line option 
     * java [ java-option ] org.apache.jserv.JServHandler [ -t ] port
     */
    args[0] = conf->binary;

    /* Pass extra options to Java */
    for (i = 1; i < conf->binary_args->nelts + 1; i++) {
	char *arg = ((char **) (conf->binary_args->elts))[i - 1];

	args[i] = arg;
    }


    args[i] = JSERV_CLASS;
    i++;

    /* Add trace option if necessary */
    if (log_fd >= 0) {
	args[i] = "-t";
	i++;
    }

    /* Port */
    ap_snprintf(port, sizeof(port), "%d", conf->port);
    args[i] = pstrdup(p, port);
    i++;

    args[i] = NULL;
    if (do_debugging) {
        for (j = 0; j < i; ++j) {
            fprintf(stderr, "mod_jserv: command line argument %d is: %s\n", j, args[j]);
        }
    }

/** Set up the Environment */

    env = malloc((sizeof(char *)) * (4 + conf->envs->nelts));

    for (i = 0; i < conf->envs->nelts;) {
        char *var = ((char **) (conf->envs->elts))[i];

        env[i++] = pstrdup(p, var);
    }

    env[i++] = pstrcat(p, "CLASSPATH=", conf->classpath, NULL);
    if (conf->add_path_env)
        env[i++] = pstrcat(p, "PATH=", getenv("PATH"), NULL);
    if (conf->add_tz_env && getenv("TZ") != NULL)
        env[i++] = pstrcat(p, "TZ=", getenv("TZ"), NULL);
    env[i] = NULL;

    if (do_debugging) {
        for (j = 0; j < i; ++j) {
            fprintf(stderr, "mod_jserv: env variable %d is: %s\n", j, env[j]);
        }
    }


    /* If we're not root, we don't have to do any of this. */
    if (!geteuid()) {
        /* First, change the group id.  This is code copied from http_main.c;
         * we can't use the function there because it's static to that file. */
	char *name;

	/* Get username if passed as a uid */

	if (ap_user_name[0] == '#') {
	    struct passwd *ent;
	    uid_t uid = atoi(&ap_user_name[1]);

	    if ((ent = getpwuid(uid)) == NULL) {
                jserv_log_error(APLOG_MARK, APLOG_ERR, s,
			 "getpwuid: couldn't determine user name from uid, "
			 "you probably need to modify the User directive");
	        exit(1); /* don't know enough to exit cleanly... oh well. */
	    }

	    name = ent->pw_name;
	}
	else
	    name = ap_user_name;

#ifndef OS2
	/* OS/2 dosen't support groups. */

	/* Reset `groups' attributes. */

	if (initgroups(name, ap_group_id) == -1) {
            jserv_log_error(APLOG_MARK, APLOG_ERR, s,
			"initgroups: unable to set groups");
	    exit(1); /* don't know enough to exit cleanly... oh well. */
	}
#ifdef MULTIPLE_GROUPS
	if (getgroups(NGROUPS_MAX, ap_group_id_list) == -1) {
            jserv_log_error(APLOG_MARK, APLOG_ERR, s,
			"getgroups: unable to get group list");
	    exit(1); /* don't know enough to exit cleanly... oh well. */
	}
#endif
	if (setgid(ap_group_id) == -1) {
            jserv_log_error(APLOG_MARK, APLOG_ERR, s,
			"setgid: unable to set group id");
	    exit(1); /* don't know enough to exit cleanly... oh well. */
	}
#endif
        /* Change uid to the server's User, if appropriate */
        if (
#ifdef _OSD_POSIX
            os_init_job_environment(s, ap_user_name) != 0 ||
#endif
            setuid(s->server_uid) == -1) {

            jserv_log_error(APLOG_MARK, APLOG_ERR, s,
                            "jserv wrapper: unable to change uid for JVM");
            exit(1);
        }
    }

    if (log_fd >= 0) {
	dup2(log_fd, STDERR_FILENO);
    }
    else {
	error_log2stderr(s);
    }

    cleanup_for_exec();

    if (execve(conf->binary, args, env) < 0) {
	fprintf(stderr, "mod_jserv: Jserv failed to exec Java interpreter.\n");
	exit(1);
    }

#if MODULE_MAGIC_NUMBER >= 19970623
    return 0;
#endif
}

/* Send a line in the format JServ expects; four hex characters indicating
 * length, then the string.
 */
#if MODULE_MAGIC_NUMBER >= 19980413 
static void write_hexline(BUFF *b, const char *line) {
    ap_bprintf(b, "%04x%s", strlen(line), line);
}
#else
static void send_hexline(FILE *f, const char *line) {
    fprintf(f, "%04x%s", strlen(line), line);
}
#endif

/* Send a line in the format JServ expects; this is for use w/ binary data,
 * specifically the env variables, headers, & etc..
 */
#if MODULE_MAGIC_NUMBER >= 19980413 
static void write_binhexline(BUFF *b, const char *line, int length) {
    ap_bprintf(b,"%04x", length); /* not going to bother checking return yet */
    ap_bwrite(b, line, length);
}
#else
static void write_binhexline(FILE *f, const char *line, int length) {
   fprintf(f, "%04x", length);
   fwrite(line, sizeof(char), length, f);
} 
#endif

/* Used to send data to jserv if sockets fail */
static FILE *jserv_in = NULL;

static int count = 0;
static signal_current;
/* These is a signal handler for child #1. When it gets a signal,
 * it passes it on to JServ.
 */

static void jserv_signal(int sig) {
    sigset_t blocked_sigs;
    char sigstring[4];
    int i, sock, use_stdin = 0;
    struct sockaddr_in addr;

    sigemptyset(&blocked_sigs);
    sigaddset(&blocked_sigs, SIGTERM);
    sigaddset(&blocked_sigs, SIGHUP);

    sigprocmask(SIG_BLOCK, &blocked_sigs, NULL);

    if (sig == SIGALRM)
        sig = signal_current;
    addr.sin_addr.s_addr = jserv_config.remote_server;
    addr.sin_port = htons(jserv_config.port);
    addr.sin_family = AF_INET;

    if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
        fprintf(stderr,"mod_jserv: Error creating socket to terminate jserv process.\n");
        /* we're not going to be able to connect, so use the STDIN method*/
        use_stdin = 1;
    }
    else {
        do {
              i = connect(sock, (struct sockaddr *)&addr,
                                 sizeof(struct sockaddr_in));
#ifdef WIN32
              if(i == SOCKET_ERROR)
                  errno = WSAGetLastError() - WSABASEERR;
#endif /* WIN32 */
        }
        while (i == -1 && errno == EINTR);
        if (i == -1) {
            /* we weren't able to connect, so we use the STDIN method*/
            use_stdin = 1;
        }
    }

    if (use_stdin) {
        /* give it two tries first: */
        if (++count <= 2) {
	    signal_current = sig;
	    signal(SIGALRM, jserv_signal);
	    sleep(1);
	}
        fprintf(jserv_in, "S%02d", sig);
        fflush(jserv_in);
    }
    else {
#if MODULE_MAGIC_NUMBER >= 19980413 
        pool *p = ap_make_sub_pool(NULL);
        BUFF *buff_socket=ap_bcreate(p,B_SOCKET+B_RDWR);
        ap_bpushfd(buff_socket,sock,sock);
        ap_note_cleanups_for_socket(p, sock);
#else
        FILE *buff_socket = fdopen(sock, "rb+");
#endif

        /* format up the message */
        sprintf(sigstring, "s%02d\0\0", sig);

        /* sent the signal encoded as a (binary header) string */
        write_binhexline(buff_socket, sigstring, strlen(sigstring) + 2);

        /* clean up */
#if MODULE_MAGIC_NUMBER >= 19980413 
        ap_bflush(buff_socket);
        ap_destroy_pool(p);
#else
        fflush(buff_socket);
        fclose(buff_socket);
#endif

    }

    if (sig == SIGTERM) {
      if (jserv_in)
          fclose(jserv_in);
      waitpid(-1, NULL, 0);
      exit(0);
    }
    count=0;
    sigprocmask(SIG_UNBLOCK, &blocked_sigs, NULL);
}

/* currently only used to send the general properties file information
 */
static void send_general_conf(pool *p, jserv_conf * conf)
{
    if (conf->gen_properties_file) {
        fprintf(jserv_in, "jserv.global.properties.file\t%s\n",
                conf->gen_properties_file);
    }
}


/* Send the configuration informations per host in the proper form 

 * hostname\tproperty_file(\tservlet_dir)+\n
 *
 */
static void send_host_config(pool * p, jserv_conf * conf)
{
    jserv_entry *entries = (jserv_entry *) conf->dirs->elts;
    int i, save_errno;

    /* Verify that there is a servlet alias for that host
     */
    if (!(conf->dirs->nelts)) {
        const char * err = "mod_jserv: missing ServletAlias or ServletProperties directive.";
        jserv_log_error(APLOG_MARK, APLOG_ERR, conf->host, err);

	return;
    }

    /* Use default properties */
    if (!conf->properties) {
	conf->properties = server_root_relative(p, JSERV_DEFAULT_PROPERTIES);
    }

    fprintf(jserv_in, "%s\t%s", ( conf->servlet_manager_hostname ? 
                                  conf->servlet_manager_hostname :
                                  conf->host->server_hostname ),
	    conf->properties);
    for (i = 0; i < conf->dirs->nelts; i++) {
	jserv_entry *dir = &entries[i];
	fprintf(jserv_in, "\t%s", dir->real);
    }

    fprintf(jserv_in, "\n");
    fflush(jserv_in);
}

/* Child #1 - This is created by Apache */

#if MODULE_MAGIC_NUMBER < 19970623
static void jserv_child(void *data)
#elif MODULE_MAGIC_NUMBER < 19980519
static int jserv_child(void *data)
#else
static int jserv_child(void *data, child_info *child)
#endif
{
    jserv_conf *conf = ((jserv_child_conf *) data)->conf;
    pool *p = ((jserv_child_conf *) data)->p;

    sigset_t blocked_sigs;
    int pid;
    int v;

    jserv_conf *host;

    /* Apache sends a signal to it's process group, so we should enter
     * a new group before spawning the JVM, to avoid being killed by SIGTERM
     */
    setpgid(0, 0);

    sigemptyset(&blocked_sigs);
    sigaddset(&blocked_sigs, SIGTERM);
    sigaddset(&blocked_sigs, SIGHUP);

    do {
	/* Block signals to prevent a race condition when starting
	 * child #2 (we recieve a SIGTERM after starting Java, but before
	 * we set up the signal handlers, so Java gets stuck in limbo
	 * forever).
	 */
	sigprocmask(SIG_BLOCK, &blocked_sigs, NULL);

	if (jserv_in)
	    fclose(jserv_in);

	/* Spawn child #2 */
	if (!(pid = spawn_child(p, jserv_child_child, data, kill_never,
				&jserv_in, NULL))) {
	    fprintf(stderr, "mod_jserv: could not spawn JServ child process");
	    exit(1);
	}

        /* send global configuration */
        send_general_conf(p, conf);

	/* Send base configuration */
	send_host_config(p, conf);

	/* Send virtual configuration */
	for (v = 0; v < conf->vhost->nelts; v++) {
	    host = ((jserv_conf **) conf->vhost->elts)[v];
	    send_host_config(p, host);
	}

	/* Terminate the sending of data */
	fprintf(jserv_in, "end\n");
	fflush(jserv_in);

	/* Set up signal handlers */
	signal(SIGTERM, jserv_signal);
	signal(SIGHUP, jserv_signal);

	/* Unblock the signals, so we can receive them */
	sigprocmask(SIG_UNBLOCK, &blocked_sigs, NULL);

	/* Poll for a change. Every second, check the following items
	 * to see if we should do something:
	 */
	for (;;) {
	    /* Wait for a sec */
	    sleep(1);

	    /* Check our parent. If it's 1, it means we're alone,
	     * and should die.
	     */
	    if (getppid() == 1)
		jserv_signal(SIGTERM);

	    /* Check on the child pid. If the status changed, it means
	     * that JServ has exited, and we should reinstall it.
	     */
	    if (waitpid(pid, NULL, WNOHANG) == pid)
		break;
	}

        if (time(NULL) - last_restart_time < 6) ++failed_restart_count;
        else
           failed_restart_count = 0;

       fprintf(stderr, "mod_jserv: VM died. time: %d, last_restart: %d, failed_restarts: %d\n",
               time(NULL), last_restart_time, failed_restart_count);

        last_restart_time = time(NULL);

        if (failed_restart_count > 6) {
            if (failed_restart_count > 7) {
                fprintf(stderr, "mod_jserv: VM died, no more restart attempts\n");
                exit(1);
            }
            /* print debugging data and exit -- something is wrong. */
	    fprintf(stderr, "mod_jserv: JServ child process died too many times, just one more attempt\n");
	    fprintf(stderr, "mod_jserv: debugging data follows\n");
        }

	/* Sleep for a few secs before reinstalling Java */
	sleep(3);
    } while (1);
}

/** Initialize the module: called once when Apache starts.
 *
 * After initializing data, we fork a child which will in turn
 * fork and monitor a JVM.
 **/
void init_jserv(server_rec * s, pool * p)
{
    jserv_conf *conf =
    (jserv_conf *) get_module_config(s->module_config, &jserv_module);
    jserv_child_conf *jcc;

    /* Are we correctly configured for automatic startup? */
    if (!conf->binary || !conf->classpath || conf->manual)
	return;

    /* Property file location are passed by sdtin */
    /*if (!conf->properties)
       conf->properties = server_root_relative(p, JSERV_DEFAULT_PROPERTIES);
     */

    if (conf->remote_server != JSERV_DEFAULT_SERVER) {
	fprintf(stderr, "mod_jserv: you cannot use ServletVMHost in automatic mode; use 'ServletManual on' (and set up a server manually) to use manual mode\n");
	exit(1);
    }

    /* Setup child data */
    jcc = pcalloc(p, sizeof(jserv_child_conf));
    jcc->conf = conf;
    jcc->s = s;
    jcc->p = p;

    /* Open error log */
    if (conf->errlog) {
	char *fname = server_root_relative(p, conf->errlog);
	if ((jcc->log_fd = popenf(p, fname, xfer_flags, xfer_mode)) < 0) {
	    perror("open");
	    fprintf(stderr, "mod_jserv: could not open JServ error log file\n");
	    exit(1);
	}
    }
    else {
	jcc->log_fd = -1;
    }

    jserv_config.port = -1;
    jserv_config.remote_server = (unsigned long) -1;
    memcpy(&jserv_config, conf, sizeof(jserv_conf));

    /* Spawn child */
    if (!(spawn_child(p, jserv_child, (void *) jcc, kill_after_timeout,
		      NULL, NULL))) {
	fprintf(stderr, "mod_jserv: could not spawn JServ child process");
	exit(1);
    }
}

static int jserv_alias_matches(char *uri, char *alias_fakename)
{
    char *end_fakename = alias_fakename + strlen(alias_fakename);
    char *aliasp = alias_fakename, *urip = uri;

    while (aliasp < end_fakename) {
	if (*aliasp == '/') {
	    /* any number of '/' in the alias matches any number in
	     * the supplied URI, but there must be at least one...
	     */
	    if (*urip != '/')
		return 0;

	    while (*aliasp == '/')
		++aliasp;
	    while (*urip == '/')
		++urip;
	}
	else {
	    /* Other characters are compared literally */
	    if (*urip++ != *aliasp++)
		return 0;
	}
    }

    /* Check last alias path component matched all the way */

    if (aliasp[-1] != '/' && *urip != '\0' && *urip != '/')
	return 0;

    /* Return number of characters from URI which matched (may be
     * greater than length of alias, since we may have matched
     * doubled slashes)
     */

    return urip - uri;
}

static char *try_jserv_list(request_rec * r, array_header * dirs)
{
    jserv_entry *entries = (jserv_entry *) dirs->elts;
    int i;

    for (i = 0; i < dirs->nelts; ++i) {
	jserv_entry *e = &entries[i];
	int l = jserv_alias_matches(r->uri, e->fake);

	if (l > 0) {
	    /* Note some info for later use */
	    table_set(r->notes, "jserv-dirname", e->real);
	    if (e->is_dir)
		table_set(r->notes, "jserv-isdir", "true");
	    r->handler = pstrdup(r->pool, "jserv-servlet");

	    return pstrcat(r->pool, e->real, r->uri + l, NULL);
	}
    }

    return NULL;
}

int translate_jserv(request_rec * r)
{
    void *sc = r->server->module_config;
    jserv_conf *conf = (jserv_conf *) get_module_config(sc, &jserv_module);
    char *ret;

#if defined(__EMX__) || defined(WIN32)
    /* Add support for OS/2 drive names */
    if ((r->uri[0] != '/' && r->uri[0] != '\0') && r->uri[1] != ':')
#else
    if (r->uri[0] != '/' && r->uri[0] != '\0')
#endif
	return DECLINED;

    if ((ret = try_jserv_list(r, conf->dirs))) {
	r->filename = ret;
	return OK;
    }
    return DECLINED;
}

/* export info early, during the header parsing stage */
int jserv_req_init(request_rec * r)
{
    void *sc = r->server->module_config;
    jserv_conf *conf = (jserv_conf *) get_module_config(sc, &jserv_module);

    /* some notes for other modules (i.e. perl, php), possibly used by
     * Ian's serlvet module */
    char port[15];
 
    if (conf->auth_export) {
        ap_snprintf(port, sizeof(port), "%d", conf->port);
        table_set(r->notes, "jserv-port", pstrdup(r->pool, port));
    }

    return DECLINED;
}

int jserv_handler(request_rec * r)
{
    void *s = r->server->module_config;
    jserv_conf *conf = (jserv_conf *) get_module_config(s, &jserv_module);
    char *p, *classname;
    char const *dirname = table_get(r->notes, "jserv-dirname");
    char const *hdr;
    char const *is_dir = table_get(r->notes, "jserv-isdir");
    int ret, tempint, tempint2;
    jserv_vm_host *first_vm;
    char *block;
/* to use ap_send_fb... I know it existed earlier, but this is conservative
 * and safe.
 */
#if MODULE_MAGIC_NUMBER >= 19980713
    BUFF *buff_socket;
#else
    FILE *buff_socket;
#endif

    struct sockaddr_in addr;
    int i, sock;

    /* Make sure we're set up correctly */
    if (!conf->binary && !conf->manual ) {
	log_reason("ServletBinary not configured", r->uri, r);
	return SERVER_ERROR;
    }
    if (!conf->classpath && !conf->manual) {
	log_reason("ServletClassPath not configured", r->uri, r);
	return SERVER_ERROR;
    }

    /* Make sure we have a dir name - we must be called with
     * a matching ServletAlias directive, found when translating.
     */
    if (!dirname) {
	log_reason("Servlet class name unknown (illegal access?)", r->uri, r);
	return FORBIDDEN;
    }

    /* If it's a directory, then this is simple. If not, this is more
     * complex, since we need to identify the class name from the path info
     */

    if (is_dir) {
	classname = pstrdup(r->pool, r->filename + strlen(dirname) + 1);

	/* Translate slash (/) to period (.), so you can use physical
	 * names to identify classes.
	 */
	for (p = classname; *p; p++) {
	    if (*p == '/')
		*p = '.';
	}
    }
    else if (!r->path_info) {
	/* We should have some path info... if not, it means we didn't
	 * get a class to match!
	 */
	return NOT_FOUND;
    }
    else {
	char *path = r->path_info;
	char *slash;

	/* If there's a leading /, strip it off */
	if (path[0] == '/')
	    path++;

	slash = strchr(path, '/');

	if (!slash) {
	    /* No extra path info */
	    classname = pstrdup(r->pool, path);
	    r->path_info = "";
	}
	else {
	    classname = pstrndup(r->pool, path, slash - path);
	    r->path_info = slash;
	}
    }

    /* Make sure we have a class name. */

    if (!classname || !*classname) {
	log_reason("Class name could not be identified", r->uri, r);
	return NOT_FOUND;
    }

    /* Open the connection to the server */
    addr.sin_family = AF_INET;
    if(conf->vm_hosts) {
      /* Figure out which host to connect to */
      jserv_find_vm(&addr, conf, r);
    }
    else {
      /* no load balancing.  use whatever single VM has been setup */
      addr.sin_addr.s_addr = conf->remote_server;
      addr.sin_port = htons(conf->port);
    }

    if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
	log_reason("Could not create JServ socket", r->uri, r);
	return SERVER_ERROR;
    }
#if MODULE_MAGIC_NUMBER >= 19980413 
        ap_note_cleanups_for_socket(r->pool, sock);
#else
        note_cleanups_for_fd(r->pool, sock);
#endif

    hard_timeout("proxy connect", r);

    do {
	i = connect(sock, (struct sockaddr *) &addr,
		    sizeof(struct sockaddr_in));
#ifdef WIN32
	if (i == SOCKET_ERROR)
	    errno = WSAGetLastError() - WSABASEERR;
#endif /* WIN32 */
    }
    while (i == -1 && errno == EINTR);
    if (i == -1) {
      if(conf->vm_hosts) {
	/* attempt to connect to other JVMs in the list */
	first_vm = conf->vm_hosts;
	do {
	  jserv_next_vm(&addr, conf);
	  do {
	    /* is this ok to do, or do we need to clean up the socket
	     * ourselves?  seems like the previous call to
	     * note_cleanups_for_fd will do that for us... but I'm not sure
	     */
	    if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
	      log_reason("Could not create JServ socket", r->uri, r);
	      return SERVER_ERROR;
	    }
#if MODULE_MAGIC_NUMBER >= 19980413 
            ap_note_cleanups_for_socket(r->pool, sock);
#else
	    note_cleanups_for_fd(r->pool, sock);
#endif
	    i = connect(sock, (struct sockaddr *) &addr,
			sizeof(struct sockaddr_in));
#ifdef WIN32
	    if (i == SOCKET_ERROR)
	      errno = WSAGetLastError() - WSABASEERR;
#endif /* WIN32 */
	  }
	  while (i == -1 && errno == EINTR);
	} while(first_vm != conf->vm_hosts && i == -1);
	
      }

      if(i == -1) {
	log_reason("JServ connect failed", r->uri, r);
	return SERVER_ERROR;
      }
    }

    kill_timeout(r);

#if MODULE_MAGIC_NUMBER >= 19980413 
    buff_socket=ap_bcreate(r->pool,B_SOCKET+B_RDWR);
    ap_bpushfd(buff_socket,sock,sock);
#else
    buff_socket = pfdopen(r->pool, sock, "rb+");
#endif

    /* Send the data */

    add_cgi_vars(r);
    add_common_vars(r);

    if ((ret = setup_client_block(r, REQUEST_CHUNKED_ERROR)))
	return ret;

    hard_timeout("jserv send", r);

    /* time to construct the block to send to the JVM.
     * HUGE_STRING_LEN won't be large enough for anything, but it
     * will be large enough for the vast majority of cases. */
    tempint = HUGE_STRING_LEN;
    block = (char *)palloc(r->pool, tempint * sizeof(char));

    /* don't walk off our memory block */
    if ((3 + strlen(dirname) + strlen(classname) +
        strlen (conf->servlet_manager_hostname ? conf->servlet_manager_hostname:
                conf->host->server_hostname)) > tempint) {
        /* something is probably broken... oh well. */
	log_reason("error in mod_jserv's handled: the total of the dir "
                   "name, the classname, and the manager name is more\n"
                   "than 65535 bytes ... this looks like misconfiguration or a "
                   "bug.\n", r->uri, r);

	/* Zap the existing status, or Apache will do odd things */
	r->status = HTTP_OK;
	r->status_line = NULL;

        return SERVER_ERROR;
    }

    /* Send servlet alias matched and class name : C = class */
    sprintf(block, "C%s\t%s\0", dirname, classname);
    tempint2 = 3 + strlen(dirname) + strlen(classname);

    /* Send host name S = Server */
    sprintf(block + tempint2, "S%s\0", (conf->servlet_manager_hostname ?
                                        conf->servlet_manager_hostname :
                                        conf->host->server_hostname));

    tempint2 += 2 + strlen(conf->servlet_manager_hostname ?
                           conf->servlet_manager_hostname :
                           conf->host->server_hostname);
 
    /* Send environment vars, except for the header ones */
    if (r->subprocess_env) {
	array_header *env_arr = table_elts(r->subprocess_env);
	table_entry *elts = (table_entry *) env_arr->elts;

	for (i = 0; i < env_arr->nelts; ++i) {
            int tempint3;
	    if (!elts[i].key)
		continue;
	    if (!strncmp(elts[i].key, "HTTP_", 5))
		continue;
            tempint3 = (tempint2+strlen(elts[i].key)+strlen(elts[i].val)+3);
            while (tempint3 > tempint) {
                tempint *= 2;
                if (tempint >= tempint3) {
                    char *block2 = palloc(r->pool, tempint);
                    memcpy (block2, block, tempint2);
                    block = block2;
                }
            }
            sprintf(block + tempint2, "E%s\t%s\0", elts[i].key, elts[i].val);
            tempint2 = tempint3;
	}
    }

    /* Send the request headers */

    if (r->headers_in) {
        int tempint3;
	array_header *hdr_arr = table_elts(r->headers_in);
	table_entry *elts = (table_entry *) hdr_arr->elts;

	for (i = 0; i < hdr_arr->nelts; ++i) {
	    if (!elts[i].key)
		continue;

            tempint3 = (tempint2+strlen(elts[i].key)+strlen(elts[i].val)+3);
            while (tempint3 > tempint) {
                tempint *= 2;
                if (tempint >= tempint3) {
                    char *block2 = palloc(r->pool, tempint);
                    memcpy (block2, block, tempint2);
                    block = block2;
                }
            }
            sprintf(block + tempint2, "H%s\t%s\0", elts[i].key, elts[i].val);
            tempint2 = tempint3;
	}
    }

    if ((tempint2 + 1) > tempint) {
        char *block2 = palloc(r->pool, (tempint * 2));
        memcpy(block2, block, tempint2);
        block = block2;
    }
    *(block + tempint2++) = '\0';

    /* Send the header info */
    tempint = 0;
    if (tempint2 > (256 * 256 - 1)) {
        while (1) {
            if (tempint2 > (256 * 256 - 1)) {
                write_binhexline(buff_socket, block + tempint, 256 * 256 - 1);
                tempint2 -=  256 * 256 - 1;
                tempint  +=  256 * 256 - 1;
            } else {
                write_binhexline(buff_socket, block + tempint, tempint2);
                break;
            }
        }
    } else {
        write_binhexline(buff_socket, block, tempint2);
    }

    /* flush the data. */
#if MODULE_MAGIC_NUMBER >= 19980413 
   ap_bflush(buff_socket);
#else
    fflush(buff_socket);
#endif

    /* If there is a request entity, send it */

    if (should_client_block(r)) {
	size_t len_read;
        char buffer[HUGE_STRING_LEN];

#if MODULE_MAGIC_NUMBER >= 19980413 
        /* If we did read something we'll post it to JServ */
        while ((len_read=ap_get_client_block(r,buffer,
                HUGE_STRING_LEN)) > 0 ) {
            /* Reset our writing timeout */
            ap_reset_timeout(r);
            /* Check that what we writed was the same of what we read */
            if (ap_bwrite(buff_socket,buffer,len_read) < len_read) {
                /* Discard all further characters left to read */
                while (ap_get_client_block(r, buffer, HUGE_STRING_LEN) > 0);
                    break;
            }
        }
#else
	while ((len_read = get_client_block(r, buffer, HUGE_STRING_LEN)) > 0) {
	    reset_timeout(r);
	    if (fwrite(buffer, sizeof(char), len_read, buff_socket) < len_read){
		/* Something happened, soak up the rest */
		while (get_client_block(r, buffer, HUGE_STRING_LEN) > 0);
		break;
	    }
	}
#endif
    }

#if MODULE_MAGIC_NUMBER >= 19980413 
   ap_bflush(buff_socket);
#else
    fflush(buff_socket);
#endif
    kill_timeout(r);

    /* Receive the response */

    hard_timeout("jserv receive", r);

#if MODULE_MAGIC_NUMBER >= 19980413 
    if ((ret = ap_scan_script_header_err_buff(r,buff_socket,NULL))) {
#else
    if ((ret = scan_script_header(r, buff_socket))) {
#endif
	kill_timeout(r);
	return ret;
    }

    /* Check for our special headers */

    if ((hdr = table_get(r->err_headers_out, "Servlet-Log"))) {
        int save_errno;
	table_unset(r->err_headers_out, "Servlet-Log");
        jserv_log_error(APLOG_MARK, APLOG_NOTICE, r->server, hdr);
    }

    if ((hdr = table_get(r->err_headers_out, "Servlet-Error"))) {
	int status = r->status;

	table_unset(r->err_headers_out, "Servlet-Error");
	log_reason(hdr, r->uri, r);

	/* Zap the existing status, or Apache will do odd things */
	r->status = HTTP_OK;
	r->status_line = NULL;

	return status;
    }

    /* Do CGI-style Location-handling */

    if (table_get(r->headers_out, "Location") && r->status == 200) {
	kill_timeout(r);

	r->status = HTTP_OK;
	r->status_line = NULL;

	return REDIRECT;
    }

#if MODULE_MAGIC_NUMBER >= 19980413 
   /* Send headers and data collected (if this was not a "header only" req. */
    ap_send_http_header(r);
        if (!r->header_only) ap_send_fb(buff_socket, r);
 
    /* Kill timeouts, close buffer and socket and return */
    ap_kill_timeout(r);
    ap_bclose(buff_socket);
    ap_pclosesocket(r->pool,sock);

#else
    send_http_header(r);
    if (!r->header_only)
	send_fd(buff_socket, r);

    kill_timeout(r);
    pfclose(r->pool, buff_socket);
#endif

    return OK;
}

handler_rec jserv_handlers[] =
{
    {"jserv-servlet", jserv_handler},
    {NULL}
};

module jserv_module =
{
    STANDARD_MODULE_STUFF,
    init_jserv,			/* initializer */
    NULL,			/* dir config creater */
    NULL,			/* dir merger --- default is to override */
    create_jserv_config,	/* server config */
    merge_jserv_config,		/* merge server configs */
    jserv_cmds,			/* command table */
    jserv_handlers,		/* handlers */
    translate_jserv,		/* filename translation */
    NULL,			/* check_user_id */
    NULL,			/* check auth */
    NULL,			/* check access */
    NULL,			/* type_checker */
    NULL,			/* fixups */
    NULL,			/* logger */
    jserv_req_init		/* header parser */
};

/*****************************************************************************
 *
 * Load balancing code
 *
 * Written by Bernard Bernstein for the Apache JServ 1.0 project.  Grafted
 * back into JServ 0.9.x by James Cooper
 *
 *****************************************************************************/

#define SESSION_IDENTIFIER "JServSessionId"


static char*
get_param(char *name, request_rec *r)
{
  char *pname = ap_pstrdup(r->pool, name);
  char *value = NULL;
  char *varg = NULL;
  int len = 0;

  pname = ap_pstrcat(r->pool, pname, "=", NULL);

  if (!r->args) {
    return NULL;
  }

  value = strstr(r->args, pname);
  if (value) {
    value += strlen(pname);
    varg = value;
    while (*varg && *varg != '&') {
      varg++;
      len++;
    }
  }

  if (len == 0) {
    return NULL;
  }

  return ap_pstrndup(r->pool, value, len);
}

static char*
get_cookie(char *name, request_rec *r)
{
  const char *cookie;
  char *value;
  char *cname = ap_pstrdup(r->pool, name);
  char *varg;

  cname = ap_pstrcat(r->pool, cname, "=", NULL);
  if ((cookie = ap_table_get(r->headers_in, "Cookie")) != NULL) {
    if ((value = strstr(cookie, cname)) != NULL) {
      char *cookiebuf, *cookieend;
      value += strlen(cname);
      cookiebuf = ap_pstrdup(r->pool, value);
      cookieend = strchr(cookiebuf, ';');
      if (cookieend)
        *cookieend = '\0';      /* Ignore anything after a ; */

      return cookiebuf;    /* Theres already a cookie, no new one */
    }
  }

  return NULL;
}


static char*
get_jserv_sessionid(request_rec *r)
{
  char *val;
  val = get_param(SESSION_IDENTIFIER, r);
  if (val == NULL)
    val = get_cookie(SESSION_IDENTIFIER, r);
  return val;
}

static int
get_jserv_session_route(struct sockaddr_in *addr, request_rec *r)
{
  char *sessionid = get_jserv_sessionid(r);
  char *ch;
  char *addr_port;
  int segment = 0;    /* segments of the id. they are: */
                      /* 0 - random */
                      /* 1 - session_count */
                      /* 2 - time */
                      /* 3 - hostaddr:port "hex_long:hex_int" */
  
  if (sessionid == NULL)
    return 0;
  
  ch = sessionid;
  while (*ch != '\0') {
    if (*ch == '.') {
      segment++;
    }
    else if (segment == 3 ) {
      unsigned char addr_byte;
      char *colon = strchr(ch, ':');
      unsigned int port;
      if (colon != NULL) {
        *colon = '\0';
	colon++;
	port = (unsigned int)strtol(colon, NULL, 16);
        addr->sin_port = htons(port);
      }
      addr->sin_addr.s_addr = ntohl((unsigned long)strtol(ch, NULL, 16));

      return 1;
    }
    ch++;
  }
  return 0;
}



/**
 * figures out which host and port to connect to and sets the variables in
 * sockaddr_in accordingly
 */
void jserv_find_vm(struct sockaddr_in *addr, jserv_conf *conf, request_rec *r) {
  if(!get_jserv_session_route(addr, r)) {
    /* use the next VM in the list */
    jserv_next_vm(addr, conf);
  }
}

void jserv_next_vm(struct sockaddr_in *addr, jserv_conf *conf) {
  addr->sin_addr.s_addr = conf->vm_hosts->remote_server;
  addr->sin_port = htons(conf->vm_hosts->port);
  conf->vm_hosts = conf->vm_hosts->next;
}

