 /* ====================================================================
    * Copyright (c) 1997,1998 The Apache Group.  All rights reserved.
    *
    * Redistribution and use in source and binary forms, with or without
    * modification, are permitted provided that the following conditions
    * are met:
    *
    * 1. Redistributions of source code must retain the above copyright
    *    notice, this list of conditions and the following disclaimer. 
    *
    * 2. Redistributions in binary form must reproduce the above copyright
    *    notice, this list of conditions and the following disclaimer in
    *    the documentation and/or other materials provided with the
    *    distribution.
    *
    * 3. All advertising materials mentioning features or use of this
    *    software must display the following acknowledgment:
    *    "This product includes software developed by the Apache Group
    *    for use in the Apache HTTP server project (http://www.apache.org/)."
    *
    * 4. The names "Apache Server" and "Apache Group" must not be used to
    *    endorse or promote products derived from this software without
    *    prior written permission.
    *
    * 5. Redistributions of any form whatsoever must retain the following
    *    acknowledgment:
    *    "This product includes software developed by the Apache Group
    *    for use in the Apache HTTP server project (http://www.apache.org/)."
    *
    * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
    * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
    * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
    * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
    * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
    * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
    * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
    * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
    * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
    * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
    * OF THE POSSIBILITY OF SUCH DAMAGE.
    * ====================================================================
    *
    * This software consists of voluntary contributions made by many
    * individuals on behalf of the Apache Group and was originally based
    * on public domain software written at the National Center for
    * Supercomputing Applications, University of Illinois, Urbana-Champaign.
    * For more information on the Apache Group and the Apache HTTP server
    * project, please see <http://www.apache.org/>.
    *
  */

/*
 * jserv.c: Implements part of Java Server API for Apache
 * by Alexei Kosut <akosut@apache.org>
 *
 * This is a standalone C program that controls the Java process
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
  
/* order matters here...don't switch these two */
#include <netinet/in.h>
#include <arpa/inet.h>

/* Name of the class to run - not configurable */
#define JSERV_CLASS "org.apache.jserv.JServHandler"

/* Port number our child is running on*/
static int port;

/* Name of the default lock file */
#define DEFAULT_LOCK_FILE "jserv.pid"

/* Child #2 - This is created by child #1 */
static int jserv_child(char *args[], char *envp[])
{
    if (execve(args[0], args, envp) < 0) {
	perror("jserv exec failed\n");
    }

    return 0;
}

/* Make java arguments */
char **create_args(int argc, char *argv[])
{
    char **args;
    int i;
    char *javaexe;
    char *propfile;

    javaexe = getenv("JAVA_BINARY");
    if (!javaexe) {
	fprintf(stderr, "Variable JAVA_BINARY is not set.\n");
	exit(1);
    }

    propfile = getenv("JSERV_PROPERTY_FILE");
    if (!propfile) {
	fprintf(stderr, "Variable JSERV_PROPERTY_FILE is not set.\n");
	exit(1);
    }

    args = malloc((argc + 4) * (sizeof(char *)));
    /* java org.apache.jserv.JServHandler -m propFile <args> */
    args[0] = strdup(javaexe);

    /* Pass -J options to Java */
    for (i = 1; !strncmp(argv[i], "-J", 2); i++) {
	args[i] = 2 + argv[i];
    }

    args[i++] = JSERV_CLASS;
    args[i++] = "-m";
    args[i++] = strdup(propfile);

    for (; (i - 3) < argc; i++) {
	args[i] = argv[i - 3];
    }

    args[++i] = NULL;

    return args;
}

/* These is a signal handler for child #1. When it gets a signal,
 * it passes it on to JServ.
 */

static FILE *jserv_in = NULL;
  
/* it's ugly, but basically this is just a couple of buffers in which to hold
 * data in case the signal handler is called.
 */
static int jserv_child_argc;
static char **jserv_child_args;
static char **jserv_child_envp;

/* Send a line in the format JServ expects; this is for use w/ binary data,
 * specifically the env variables, headers, & etc..
 */
static void write_hexline(FILE *f, const char *line, int length) {
   fprintf(f, "%04x", length);
   fwrite(line, sizeof(char), length, f);
}

/* These is a signal handler for child #1. When it gets a signal,
 * it passes it on to JServ.
 */
static void jserv_signal(int signal) {
  /* Open up a socket and send a message to the JServHandler sending
   * an encoded signal for it to handle.
   * If we can't connect by the socket, try stdin instead.
   */
  
    sigset_t blocked_sigs;
    char sigstring[4];
    FILE * j;
    int i, sock, use_stdin = 0;
    struct sockaddr_in addr;

    sigemptyset(&blocked_sigs);
    sigaddset(&blocked_sigs, SIGTERM);
    sigaddset(&blocked_sigs, SIGHUP);

    sigprocmask(SIG_BLOCK, &blocked_sigs, NULL);

    addr.sin_addr.s_addr = inet_addr("127.0.0.1");
    addr.sin_port = htons(port);
    addr.sin_family = AF_INET;

    if ((sock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
        fprintf(stderr,"Error creating socket to terminate jserv process.\n");
        /* we're not going to be able to connect, so use the STDIN method*/
        use_stdin = 1;
    }
    else {
        do {
            i = connect(sock, (struct sockaddr *)&addr,
                                 sizeof(struct sockaddr_in));
#ifdef WIN32
              if(i == SOCKET_ERROR)
                  errno = WSAGetLastError() - WSABASEERR;
#endif /* WIN32 */
        }
        while (i == -1 && errno == EINTR);
        if (i == -1) {
            /* we weren't able to connect, so we use the STDIN method*/
            use_stdin = 1;
        }
    }

    if (use_stdin) {
        fprintf(stderr,"using stdin");
        fprintf(jserv_in, "S%02d", signal);
        fflush(jserv_in);
    }
    else {
        j = fdopen(sock, "rb+");

        /* format up the message */
        sprintf(sigstring, "s%02d\0\0", signal);

        /* sent the signal encoded as a string */
        write_hexline(j, sigstring, 5);

        /* clean up */
        fflush(j);
        fclose(j);
    }

    if (signal == SIGTERM) {
        if (jserv_in)
            fclose(jserv_in);
        waitpid(-1, NULL, 0);
        exit(0);
    }
    sigprocmask(SIG_UNBLOCK, &blocked_sigs, NULL);
    parent_main(jserv_child_argc,jserv_child_args,jserv_child_envp,0);
}
  
/*
 * Write our PID to a lock to facilitate stop/restart scripts
 */
void write_pid()
{
    char *lock_fname;
    FILE *pid_file;

    lock_fname = getenv("JSERV_LOCK_FILE");
    if (!lock_fname) {
	lock_fname = DEFAULT_LOCK_FILE;
    }

    pid_file = fopen(lock_fname, "w");
    if (!pid_file) {
	fprintf(stderr, "Can't create lock file!\n");
	exit(1);
    }

    fprintf(pid_file, "%u", getpid());
    fclose(pid_file);
}

/* Child #1 - This is created by Apache */

int main(int argc, char *argv[], char *envp[])
{
    char ** args;

    if (argc < 2) {
	fprintf(stderr, "jserv: Illegal arguments\n");
	fprintf(stderr, "usage: jserv [-Jopt1 -Jopt2...] [-t] port\n");
	exit(1);
    }

    /* set our port number for later */
    port = atoi(argv[argc-1]);

    args = create_args(argc, argv);

    /* Write pid */
    write_pid();
    jserv_child_argc = argc;
    jserv_child_args = args;
    jserv_child_envp = envp;
    parent_main(argc,args,envp,1);
}

int parent_main(int argc, char *args[], char *envp[], int spawn_child) {
    int pid;
    int in_fds[2];
    do {
	if (spawn_child) { /* used so that we can get back after SIGHUP */
	    if (jserv_in) {
		fclose(jserv_in);
	    }

	    /* Spawn child */

	    if (pipe(in_fds) < 0) {
		fprintf(stderr, "Could not open JServ pipes\n");
		exit(1);
	    }

	    if ((pid = fork()) < 0) {
		close(in_fds[0]);
		close(in_fds[1]);
		fprintf(stderr, "Could not fork JServ child process\n");
		exit(1);
	    }

	    if (!pid) {
		/* Child process */

		close(in_fds[1]);
		dup2(in_fds[0], STDIN_FILENO);
		close(in_fds[0]);

		signal(SIGCHLD, SIG_DFL);

		return jserv_child(args, envp);
	    }
	}

	/* Parent */
        spawn_child = 1; /* if something below fails, we should try again */

	close(in_fds[0]);
	jserv_in = fdopen(in_fds[1], "w");

	/* Set up signal handlers */
	signal(SIGTERM, jserv_signal);
	signal(SIGHUP, jserv_signal);

	/* Poll for a change. Every second, check the following items
	 * to see if we should do something:
	 */
	for (;;) {
	    /* Wait for a sec */
	    sleep(1);

	    /* Check on the child pid. If the status changed, it means
	     * that JServ has exited, and we should reinstall it.
	     */
	    if (waitpid(pid, NULL, 0) == pid)
		break;
	}

	/* Sleep for a few secs before reinstalling Java */
	sleep(3);
    } while (1);
    /* TODO: It would be good to avoid looping forever if there's a 
     * fatal problem with starting the server -- for example, if the 
     * Java server binary is missing.  Perhaps we could do something
     * like SysVinit and disable the service if it fails five times
     * in quick succession. -- mbp@pharos.com.au 19971022 */
}
