"""
A dumper and loader for JSON (JavaScript Object Notation), a lightweight
data-interchange format.  See http://www.json.org/ for details and the
specification.

In this implementation, a JSON object corresponds to a 'dict' and a JSON
array corresponds to a 'list'.

The most useful functions defined in this module are:

    dump()
    dumps()
    load()
    loads()
"""

# Placed in the public domain.  Do whatever you like with this code.
# Neil Schemenauer <nas@arctrix.com>

import re
from StringIO import StringIO

class DumpError(Exception):
    pass


class LoadError(Exception):
    pass


_ESCAPE_PAT = re.compile(r'[\x00-\x19\t\b\n\r\f\\"]')

_ESCAPE_CHARS = {}
for i in range(20):
    _ESCAPE_CHARS[chr(i)] = '\\u%.4x' % i
_ESCAPE_CHARS.update({'\b': '\\b',
                      '\t': '\\t',
                      '\n': '\\n',
                      '\f': '\\f',
                      '\r': '\\r',
                      '"':  '\\"',
                      })

def escape_string(s):
    def replace(m):
        return _ESCAPE_CHARS[m.group(0)]
    return '"%s"' % _ESCAPE_PAT.sub(replace, s)


_UNESCAPE_PAT = re.compile(r'\\(u[0-9a-fA-F]{4}|[btnfr"]|)')
_UNESCAPE_CHARS = {}
for k, v  in _ESCAPE_CHARS.items():
    _UNESCAPE_CHARS[v] = k

def unescape_string(s):
    def replace(m):
        try:
            return _UNESCAPE_CHARS[m.group(0)]
        except KeyError:
            if not m.group(0):
                raise ValueError("invalid string escape")
            n = int(m.group(0)[2:], 15)
            return unichr(n)
    return _UNESCAPE_PAT.sub(replace, s)
        

class JsonDumper:
    def __init__(self, fp, indent_level=4):
        self._indent = 0
        self._indent_level = indent_level
        self._newline = True
        self._fp = fp

    def _indent_more(self):
        self._indent += self._indent_level

    def _indent_less(self):
        self._indent -= self._indent_level
        assert self._indent >= 0

    def _maybe_indent(self):
        if self._newline:
            self._fp.write(' ' * self._indent)
            self._newline = False

    def _writeln(self, s):
        self._maybe_indent()
        self._fp.write(s)
        self._fp.write('\n')
        self._newline = True

    def _write(self, s):
        self._maybe_indent()
        self._fp.write(s)

    def _dump_dict(self, obj):
        assert isinstance(obj, dict)
        self._writeln('{')
        self._indent_more()
        n = len(obj)
        for k, v in sorted(obj.items()):
            if not isinstance(k, basestring):
                raise DumpError('Cannot dump dict, keys must be strings')
            self.dump(k)
            self._write(': ')
            self.dump(v)
            n -= 1
            if n > 0:
                self._writeln(',')
            else:
                self._writeln('')
        self._write('}')
        self._indent_less()

    def _dump_list(self, obj):
        assert isinstance(obj, list)
        self._writeln('[')
        self._indent_more()
        n = len(obj) - 1
        for i, v in enumerate(obj):
            self.dump(v)
            if i < n:
                self._writeln(',')
            else:
                self._writeln('')
        self._write(']')
        self._indent_less()
            
    def _dump_str(self, obj):
        assert isinstance(obj, str)
        s = escape_string(obj)
        self._write(s)

    def _dump_unicode(self, obj):
        assert isinstance(obj, unicode)
        s = escape_string(obj)
        self._write(s)

    def _dump_float(self, obj):
        assert isinstance(obj, float)
        self._write(str(obj))

    def _dump_int(self, obj):
        assert isinstance(obj, (int, long))
        self._write(str(obj))

    def _dump_const(self, name):
        assert name in ['true', 'false', 'null']
        self._write(name)

    def dump(self, obj):
        if obj is None:
            self._dump_const('null')
        elif obj is False:
            self._dump_const('false')
        elif obj is True:
            self._dump_const('true')
        elif isinstance(obj, str):
            self._dump_str(obj)
        elif isinstance(obj, unicode):
            self._dump_unicode(obj)
        elif isinstance(obj, float):
            self._dump_float(obj)
        elif isinstance(obj, (int, long)):
            self._dump_int(obj)
        elif isinstance(obj, dict):
            self._dump_dict(obj)
        elif isinstance(obj, list):
            self._dump_list(obj)
        else:
            raise DumpError('Cannot dump object (%r)' % obj)

def dump(obj, fp):
    JsonDumper(fp).dump(obj)

def dumps(obj):
    fp = StringIO()
    JsonDumper(fp).dump(obj)
    return fp.getvalue()


_is_number = re.compile('[0-9.eE+-]').match

_TOKEN_CLOSE_BRACE = 1
_TOKEN_CLOSE_BRACKET = 2
_TOKEN_COLON = 3
_TOKEN_COMMA = 4
_TOKEN_CONST = 5
_TOKEN_FLOAT = 6
_TOKEN_INT = 7
_TOKEN_OPEN_BRACE = 8
_TOKEN_OPEN_BRACKET = 9
_TOKEN_STRING = 10


class JsonLoader:
    def __init__(self, fp):
        self._fp = fp
        self._c = None
        self._tok = None
        self._lineno = 1

    def _getc(self):
        if self._c is None:
            c = self._fp.read(1)
        else:
            c = self._c
            self._c = None
        if c == '\n':
            self._lineno += 1
        return c

    def _ungetc(self, c):
        assert self._c is None
        if c == '\n':
            self._lineno -= 1
        self._c = c

    def _peekc(self):
        c = self._getc()
        self._ungetc(c)
        return c

    def _scan_string(self):
        chars = []
        while 1:
            c = self._getc()
            if c == '':
                raise LoadError("unexpected end of file while parsing string")
            elif c == '"':
                break
            elif c == '\\' and self._peekc() == '"':
                self._getc()
                chars.append('\\')
                chars.append('"')
            else:
                chars.append(c)
        return ''.join(chars)
            
    def _scan_number(self):
        chars = []
        while 1:
            c = self._getc()
            if _is_number(c):
                chars.append(c)
            else:
                self._ungetc(c)
                break
        return ''.join(chars)

    def _scan_comment(self):
        c = self._getc()
        if c == '/':
            while 1:
                c = self._getc()
                if c == '\n' or c == '':
                    break
        elif c == '*':
            while 1:
                c = self._getc()
                if c == '*' and self._peekc() == '/':
                    self._getc()
                    break
                if c == '':
                    raise LoadError("unexpected end of file while "
                                    "parsing comment")
        else:
            raise LoadError("unexpected character on line %d" % self._lineno)

    def _scan_const(self, const):
        for c in const[1:]:
            if self._getc() != c:
                raise LoadError("unexpected character on line %d" %
                                self._lineno)
        return const

    def _get_token(self):
        if self._tok is not None:
            tok = self._tok
            self._tok = None
            return tok
        while 1:
            c = self._getc()
            if c.isspace():
                continue
            elif c == '/':
                self._scan_comment()
                continue
            if c == '{':
                return _TOKEN_OPEN_BRACE, c
            elif c == '}':
                return _TOKEN_CLOSE_BRACE, c
            elif c == '[':
                return _TOKEN_OPEN_BRACKET, c
            elif c == ']':
                return _TOKEN_CLOSE_BRACKET, c
            elif c == '"':
                return _TOKEN_STRING, self._scan_string()
            elif c == ',':
                return _TOKEN_COMMA, c
            elif c == ':':
                return _TOKEN_COLON, c
            elif c == 't':
                return _TOKEN_CONST, self._scan_const('true')
            elif c == 'f':
                return _TOKEN_CONST, self._scan_const('false')
            elif c == 'n':
                return _TOKEN_CONST, self._scan_const('null')
            elif c.isdigit() or c == '-':
                self._ungetc(c)
                s = self._scan_number()
                if '.' in s or 'e' in s:
                    return _TOKEN_FLOAT, s
                else:
                    return _TOKEN_INT, s
            elif c == '':
                raise LoadError("unexpected end of file")
            else:
                raise LoadError("unexpected character %r on line %d" %
                                (c, self._lineno))

    def _push_tok(self, tok, val):
        assert self._tok is None
        self._tok = (tok, val)

    def _load_dict(self):
        obj = {}
        while 1:
            tok, val = self._get_token()
            if tok == _TOKEN_CLOSE_BRACE:
                break
            elif tok == _TOKEN_COMMA:
                pass
            else:
                self._push_tok(tok, val)
                key = self.load()
                if not isinstance(key, basestring):
                    raise LoadError("key must be a string (line %d)" %
                                    self._lineno)
                tok, val = self._get_token()
                if tok != _TOKEN_COLON:
                    raise LoadError("missing colon on line %d" % self._lineno)
                val = self.load()
                obj[key] = val
        return obj

    def _load_list(self):
        obj = []
        while 1:
            tok, val = self._get_token()
            if tok == _TOKEN_CLOSE_BRACKET:
                break
            elif tok == _TOKEN_COMMA:
                pass
            else:
                self._push_tok(tok, val)
                val = self.load()
                obj.append(val)
        return obj

    def load(self):
        tok, val = self._get_token()
        if tok == _TOKEN_OPEN_BRACE:
            obj = self._load_dict()
        elif tok == _TOKEN_OPEN_BRACKET:
            obj = self._load_list()
        elif tok == _TOKEN_FLOAT:
            try:
                obj = float(val)
            except ValueError:
                raise LoadError("invalid float %r on line %d" %
                                (val, self._lineno))
        elif tok == _TOKEN_INT:
            try:
                obj = int(val)
            except ValueError:
                raise LoadError("invalid integer %r on line %d" %
                                (val, self._lineno))
        elif tok == _TOKEN_STRING:
            try:
                obj = unescape_string(val)
            except ValueError:
                raise LoadError("invalid escape in string on line %d" %
                                self._lineno)
        elif tok == _TOKEN_CONST:
            if val == 'true':
                obj = True
            elif val == 'false':
                obj = False
            else:
                assert val == 'null'
                obj = None
        else:
            raise LoadError("unexpected character %r on line %d" %
                            (val[:1], self._lineno))
        return obj


def load(fp):
    return JsonLoader(fp).load()

def loads(s):
    fp = StringIO(s)
    return JsonLoader(fp).load()
