import unittest
import re
import json

def collapse_whitespace(s):
    return re.sub(r'\s+', '', s)

def dumps(obj):
    return collapse_whitespace(json.dumps(obj))

class Test(unittest.TestCase):
    def test_dump(self):
        self.assertEqual(dumps(1), '1')
        self.assertEqual(dumps(True), 'true')
        self.assertEqual(dumps(False), 'false')
        self.assertEqual(dumps(None), 'null')
        self.assertEqual(dumps("\t\0"), '"\\t\\u0000"')
        self.assertEqual(dumps({}), '{}')
        self.assertEqual(dumps({'a': 1, 'b': 2}), '{"a":1,"b":2}')
        self.assertEqual(dumps([1.2, "a"]), '[1.2,"a"]')
        self.assertEqual(dumps([1.2, {"a": None}]), '[1.2,{"a":null}]')

    def test_load(self):
        self.assertEqual(json.loads('1'), 1)
        self.assertEqual(json.loads('true'), True)
        self.assertEqual(json.loads('false'), False)
        self.assertEqual(json.loads('null'), None)
        self.assertEqual(json.loads('{}'), {})
        self.assertEqual(json.loads('[]'), [])
        self.assertEqual(json.loads('"abc"'), "abc")
        self.assertEqual(json.loads('1.2'), 1.2)
        self.assertEqual(json.loads('1.2e10'), 1.2e10)
        self.assertEqual(json.loads('-10'), -10)
        self.assertEqual(json.loads('0.10'), 0.1)
        self.assertEqual(json.loads('" \\" "'), ' " ')
        self.assertEqual(json.loads('{"a": 1, "b": null}'), {"a": 1, "b": None})
        self.assertEqual(json.loads('[1 , // xzy\n2]'), [1, 2])
        self.assertEqual(json.loads('[1,/* xzy */2]'), [1, 2])
        self.assertEqual(json.loads(r'"\r\n\u0001"'), u"\r\n\u0001")

    def test_load_error(self):
        self.assertRaises(json.LoadError, json.loads, '')
        self.assertRaises(json.LoadError, json.loads, 'a')
        # must have leading zero according to spec
        self.assertRaises(json.LoadError, json.loads, '.1')
        self.assertRaises(json.LoadError, json.loads, '{1: 1}')
        self.assertRaises(json.LoadError, json.loads, '{"a": }')
        self.assertRaises(json.LoadError, json.loads, '[1 ')
        self.assertRaises(json.LoadError, json.loads, '"a')
        self.assertRaises(json.LoadError, json.loads, '"a\\u000"')

    def test_unicode(self):
        self.assertEqual(json.dumps(u"\u1234"), u'"\u1234"')
        self.assertEqual(json.loads(u'"\u1234"'), u"\u1234")

if __name__ == '__main__':
    unittest.main()
