#import "UNSystemConfigurationDynamicStore.h"
#import "UNSystemConfigurationErrors.h"

void dynStoreCallBack(SCDynamicStoreRef store, CFArrayRef changedKeys, void *info)
{
    [(UNSystemConfigurationDynamicStore*)info keysChanged:(NSArray*)changedKeys];
}

@implementation UNSystemConfigurationDynamicStore
-init
{
    self = [super init];
    _dynamicStore = NULL;
    _delegate = NULL;
    _rlsource = NULL;
    _attachedToRL = NO;
    return self;
}

-(void)dealloc
{
    if (_dynamicStore != NULL) {
        CFRelease(_dynamicStore);
        _dynamicStore = NULL;
    }
    if (_delegate != NULL) {
        [_delegate release];
        _delegate = NULL;
    }
    if (_rlsource != NULL) {
        CFRelease(_rlsource);
        _rlsource = NULL;
    }
    [super dealloc];
}

-initWithName:(NSString*)name
{
    SCDynamicStoreContext context = {0, NULL, &CFRetain, &CFRelease, NULL};
    SCDynamicStoreRef rval;
    self = [self init];
    if (name == nil) {
        name = @"UNSystemConfiguration";
    }
    context.info = (void*)self;
    rval = SCDynamicStoreCreate(NULL, (CFStringRef)name, &dynStoreCallBack, &context);
    if (rval == NULL) [UNSystemConfigurationErrors checkError];
    return [self setDynamicStore:rval];
}

-(void)keysChanged:(NSArray*)changedKeys
{
    id delegate = [self delegate];
    if (changedKeys == nil) {
        [NSException raise:NSInvalidArgumentException format:@"changedKeys may not be nil"];
    }
    if (delegate && [delegate respondsToSelector:@selector(keysChanged:inDynamicStore:)]) {
        [((id <UNSystemConfigurationDynamicStoreDelegate>)delegate) keysChanged:changedKeys inDynamicStore:self];
    }
}

-delegate
{
    return _delegate;
}

-setDelegate:(id)delegate
{ 
    if (_delegate != NULL) {
        [delegate release];
    }
    _delegate = [delegate retain];
    return self;
}

-setDynamicStore:(SCDynamicStoreRef)sRef
{
    if (_rlsource != NULL) {
        CFRelease(_rlsource);
        _rlsource = NULL;
    }
    if (_dynamicStore != NULL) {
        CFRelease(_dynamicStore);
    }
    _dynamicStore = sRef;
    return self;
}

-(SCDynamicStoreRef)dynamicStore
{
    return _dynamicStore;
}

-(CFRunLoopSourceRef)CFRunLoopSourceForOrder:(CFIndex)order
{
    CFRunLoopSourceRef rval;
    rval = SCDynamicStoreCreateRunLoopSource(NULL, [self dynamicStore], order);
    if (rval == NULL) [UNSystemConfigurationErrors checkError];
    return rval;
}

-(CFRunLoopSourceRef)CFRunLoopSource
{
    if (_rlsource == NULL) {
        _rlsource = [self CFRunLoopSourceForOrder:(CFIndex)0];
    }
    return _rlsource;
}

-(void)addToCurrentRunLoop
{
    if (_attachedToRL == YES) {
        [self removeFromCurrentRunLoop];
    }
    CFRunLoopAddSource([[NSRunLoop currentRunLoop] getCFRunLoop], [self CFRunLoopSource], kCFRunLoopCommonModes);
    _attachedToRL = YES;
}

-(void)removeFromCurrentRunLoop
{
    if (_attachedToRL == YES) {
        CFRunLoopRemoveSource([[NSRunLoop currentRunLoop] getCFRunLoop], _rlsource, kCFRunLoopCommonModes);
        CFRunLoopSourceInvalidate(_rlsource);
        _attachedToRL = NO;
    }
    if (_rlsource != NULL) {
        CFRelease([self CFRunLoopSource]);
    }
    _rlsource = NULL;
}

/* 
+(BOOL)setDHCPClientPreferencesForApplication:(NSString*)applicationID options:(NSArray*)options 
{
    CFIndex tempLength = [options count];
    NSMutableData* tempData = [NSMutableData dataWithCapacity:(tempLength * sizeof(UInt8))];
    NSEnumerator *enumerator = [options objectEnumerator];
    UInt8* bytes = [tempData mutableBytes];
    id obj;
    BOOL rval;
    while (obj = [enumerator nextObject]) {
        *bytes = [obj unsignedCharValue];
        ++bytes;
    }
    rval = DHCPClientPreferencesSetApplicationOptions((CFStringRef)applicationID, (UInt8*)[tempData mutableBytes], tempLength);
    [tempData release];
    return [rval autorelease];
}

+(NSArray*)DHCPClientPreferencesForApplication:(NSString*)applicationID
{
    UInt8* bytes;
    CFIndex count;
    NSMutableArray rval;
    int i;
    bytes = DHCPClientPreferencesCopyApplicationOptions((CFStringRef)applicationID, &count);
    rval = [NSMutableArray arrayWithCapacity:count];
    for (i=0; i<count; ++i) {
        [rval addObject:[NSNumber numberWithUnsignedChar:bytes[i]]];
    }
    free(bytes);
    return [rval autorelease];
}

+(NSData*)DHCPInfoOptionDataForInfo:(NSDictionary*)info code:(UInt8)code
{
    return [(NSData*)DHCPInfoGetOptionData((CFDictionaryRef)info, code) autorelease];
}

+(NSDate*)DHCPInfoLeaseStartTimeForInfo:(NSDictionary*)info
{
    return [(NSDate*)DHCPInfoGetLeaseStartTime((CFDictionaryRef)info) autorelease];
}

-(NSDictionary*)DHCPInfoForService:(NSString*)serviceId
{
    return [(NSDictionary*)SCDynamicStoreCopyDHCPInfo([self dynamicStore], (CFStringRef)serviceId) autorelease];
}
*/


-(NSString*)computerName
{
    NSString* rval = (NSString*)SCDynamicStoreCopyComputerName([self dynamicStore], NULL);
    return [rval autorelease];
}

-(NSDictionary*)consoleUser
{
    // assuming that uid_t and gid_t are UInt32
    uid_t uid;
    gid_t gid;
    NSDictionary *rdict;
    NSString* rval = [(NSString*)SCDynamicStoreCopyConsoleUser([self dynamicStore], &uid, &gid) autorelease];
    if (rval == NULL) return nil;
    rdict = [NSDictionary dictionaryWithObjectsAndKeys:rval, @"user",[NSNumber numberWithUnsignedInt:(unsigned int)uid], @"uid", [NSNumber numberWithUnsignedInt:(unsigned int)gid], @"gid", nil, nil];
    [rval release];
    return rdict;
}

-(NSString*)localHostName
{
    return [(NSString*)SCDynamicStoreCopyLocalHostName([self dynamicStore]) autorelease];
}

-(NSString*)location
{
    return [(NSString*)SCDynamicStoreCopyLocation([self dynamicStore]) autorelease];
}

-(NSDictionary*)proxies
{
    return [(NSDictionary*)SCDynamicStoreCopyProxies([self dynamicStore]) autorelease];
}

+(NSString*)networkGlobalEntityKeyForDomain:(NSString*)domain entity:(NSString*)entity
{
    return [(NSString*)SCDynamicStoreKeyCreateNetworkGlobalEntity(NULL, (CFStringRef)domain, (CFStringRef)entity) autorelease];
}

+(NSString*)networkInterfaceKeyForDomain:(NSString*)domain
{
    return [(NSString*)SCDynamicStoreKeyCreateNetworkInterface(NULL, (CFStringRef)domain) autorelease];
}

+(NSString*)networkInterfaceEntityKeyForDomain:(NSString*)domain interfaceName:(NSString*)ifname entity:(NSString*)entity
{
    return [(NSString*)SCDynamicStoreKeyCreateNetworkInterfaceEntity(NULL, (CFStringRef)domain, (CFStringRef)ifname, (CFStringRef)entity) autorelease];
}

+(NSString*)networkServiceEntityKeyForDomain:(NSString*)domain service:(NSString*)serviceId entity:(NSString*)entity
{
    return [(NSString*)SCDynamicStoreKeyCreateNetworkServiceEntity(NULL, (CFStringRef)domain, (CFStringRef)serviceId, (CFStringRef)entity) autorelease];
}

+(NSString*)computerNameKey
{
    return [(NSString*)SCDynamicStoreKeyCreateComputerName(NULL) autorelease];
}

+(NSString*)consoleUserKey
{
    return [(NSString*)SCDynamicStoreKeyCreateConsoleUser(NULL) autorelease];
}

+(NSString*)hostNamesKey
{
    return [(NSString*)SCDynamicStoreKeyCreateHostNames(NULL) autorelease];
}

+(NSString*)locationKey
{
    return [(NSString*)SCDynamicStoreKeyCreateLocation(NULL) autorelease];
}

+(NSString*)proxiesKey
{
    return [(NSString*)SCDynamicStoreKeyCreateProxies(NULL) autorelease];
}

-(NSArray*)keyListForPattern:(NSString*)pattern
{
    if (pattern == nil) {
        [NSException raise:NSInvalidArgumentException format:@"pattern may not be nil"];
    }
    return [(NSArray*)SCDynamicStoreCopyKeyList([self dynamicStore], (CFStringRef)pattern) autorelease];
}

-(void)addValue:(id)value forKey:(NSString*)key
{
    if (key == nil) {
        [NSException raise:NSInvalidArgumentException format:@"key may not be nil"];
    }
    if (value == nil) {
        [NSException raise:NSInvalidArgumentException format:@"value may not be nil"];
    }
    if (!SCDynamicStoreAddValue([self dynamicStore], (CFStringRef)key, (CFPropertyListRef)value)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-(void)addTemporaryValue:(id)value forKey:(NSString*)key
{
    if (key == nil) {
        [NSException raise:NSInvalidArgumentException format:@"key may not be nil"];
    }
    if (value == nil) {
        [NSException raise:NSInvalidArgumentException format:@"value may not be nil"];
    }
    if (!SCDynamicStoreAddTemporaryValue([self dynamicStore], (CFStringRef)key, (CFPropertyListRef)value)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-valueForKey:(NSString*)key
{
    if (key == nil) {
        [NSException raise:NSInvalidArgumentException format:@"key may not be nil"];
    }
    return [(id)SCDynamicStoreCopyValue([self dynamicStore], (CFStringRef)key) autorelease];
}

-(NSDictionary*)valuesForKeys:(NSArray*)keys matchingPatterns:(NSArray*)patterns
{
    return [(NSDictionary*)SCDynamicStoreCopyMultiple([self dynamicStore], (CFArrayRef)keys, (CFArrayRef)patterns) autorelease];
}

-(void)setValue:(id)value forKey:(NSString*)key
{
    if (key == nil) {
        [NSException raise:NSInvalidArgumentException format:@"key may not be nil"];
    }
    if (value == nil) {
        [NSException raise:NSInvalidArgumentException format:@"value may not be nil"];
    }
    if (!SCDynamicStoreSetValue([self dynamicStore], (CFStringRef)key, (CFPropertyListRef)value)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-(void)setKeysAndValues:(NSDictionary*)kv removeKeys:(NSArray*)removeKeys notifyKeys:(NSArray*)notifyKeys
{
    if (!SCDynamicStoreSetMultiple([self dynamicStore], (CFDictionaryRef)kv, (CFArrayRef)removeKeys, (CFArrayRef)notifyKeys)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-(void)removeValueForKey:(NSString*)key
{
    if (key == nil) {
        [NSException raise:NSInvalidArgumentException format:@"key may not be nil"];
    }
    if (!SCDynamicStoreRemoveValue([self dynamicStore], (CFStringRef)key)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-(void)notifyValueForKey:(NSString*)key
{
    if (key == nil) {
        [NSException raise:NSInvalidArgumentException format:@"key may not be nil"];
    }
    if (!SCDynamicStoreNotifyValue([self dynamicStore], (CFStringRef)key)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-(void)notifyValuesForKeys:(NSArray*)keys matchingPatterns:(NSArray*)patterns
{
    if (!SCDynamicStoreSetNotificationKeys([self dynamicStore], (CFArrayRef)keys, (CFArrayRef)patterns)) {
        [UNSystemConfigurationErrors checkError];
    }
}

-(NSArray*)notifiedKeys
{
    return [(NSArray*)SCDynamicStoreCopyNotifiedKeys([self dynamicStore]) autorelease];
}

+(NSNumber*)checkReachabilityForAddress:(NSData*)addr
{
    SCNetworkConnectionFlags flags;
    if (addr == nil) {
        [NSException raise:NSInvalidArgumentException format:@"addr may not be nil"];
    }
    if (SCNetworkCheckReachabilityByAddress((struct sockaddr *)[addr bytes], [addr length], &flags)) {
        return [NSNumber numberWithInt:(int)flags];
    }
    return nil;
}

+(NSNumber*)checkReachabilityForName:(NSString*)name
{
    SCNetworkConnectionFlags flags;
    if (name == nil) {
        [NSException raise:NSInvalidArgumentException format:@"name may not be nil"];
    }
    if (SCNetworkCheckReachabilityByName([name cString], &flags)) {
        return [NSNumber numberWithInt:(int)flags];
    }
    return nil;
}
@end
