using System;
using System.IO;
using System.Xml;
using System.Xml.XPath;
using System.Web.Mail;

namespace Caesar.SharpReader.Plugins
{

	abstract public class MailItemBase {

		static private MailItemConfig _config = new MailItemConfig();

		protected MailItemBase() {
			ConfigurationLoad();
#if DEBUG
			_config.SmtpServer = "mercurius.caesar.caesargroep.nl";
			_config.EmailFrom = "j.peschier@caesar.nl";
			_config.EmailTo = "j.peschier@caesar.nl";
#endif
		}

		protected void ShowConfigureDialog() {
			ShowConfigureDialog(null);
		}

		protected void ShowConfigureDialog(System.Windows.Forms.IWin32Window parent) {
			if ( parent == null ) _config.ShowDialog();
			else _config.ShowDialog(parent);
			ConfigurationSave();
		}

		protected void SendEmail(System.Xml.XPath.IXPathNavigable rssFragment) {
			SendEmail(rssFragment, "", "");
		}

		protected void SendEmail(System.Xml.XPath.IXPathNavigable rssFragment, string toEmail, string subject) {

			// Retrieve our information from the rssFragment
			string sourceTitle = FindString(rssFragment, "//channel/title");
			string sourceLink = FindString(rssFragment, "//channel/link");
			string newsTitle = FindString(rssFragment, "//item/title");
			string newsLink = FindString(rssFragment, "//item/link");
			string description = FindString(rssFragment, "//item/description");

			// Send the email
			try {
				string sentTo = "someone";
				MailMessage msg = new MailMessage();
				msg.From = string.Format("{0} <{1}>", _config.NameFrom, _config.EmailFrom);
				if ( toEmail.Length > 0 && subject.Length > 0 ) {
					msg.To = toEmail;
					msg.Subject = subject;
					sentTo = toEmail;
				} else {
					msg.To = string.Format("{0} <{1}>", _config.NameTo, _config.EmailTo);
					msg.Subject = newsTitle;
					sentTo = _config.NameTo;
				}
				msg.BodyFormat = MailFormat.Html;
				msg.Body = string.Format("<html><body><h1><a href=\"{1}\">{0}</a></h1><p>{2}</p><address>Source: <a href=\"{4}\">{3}</a></address></body></html>", newsTitle, newsLink, description, sourceTitle, sourceLink);
				msg.Priority = MailPriority.Low;
				SmtpMail.SmtpServer = _config.SmtpServer;
				SmtpMail.Send(msg);
				Message(string.Format("Sent to {0}.", sentTo));
			} catch {
				Message("Error sending email!", true);
				_config.ShowDialog();
				ConfigurationSave();
			}

		}

		private string FindString(System.Xml.XPath.IXPathNavigable rssFragment, string xpath) {
			try {
				XPathNavigator rss = rssFragment.CreateNavigator();
				XPathNodeIterator nodes = rss.Select(xpath);
				nodes.MoveNext();				// move to first matching node
				return nodes.Current.Value;		// return its value
			} catch (XmlException) {
				return "[" + xpath + "]";
			}
		}

		#region Configuration file handling

		private string ConfigurationFile() {
			string assemblyUri = this.GetType().Assembly.CodeBase;
			string assemblyPath = new Uri(assemblyUri).LocalPath;
			string assemblyDir = Path.GetDirectoryName(assemblyPath);
			return Path.Combine(assemblyDir, "MailItem.cfg");
		}

		private void ConfigurationLoad() {
			try {
				FileStream configFile = File.OpenRead(ConfigurationFile());
				StreamReader configData = new StreamReader(configFile, System.Text.Encoding.UTF8);
				_config.NameFrom = configData.ReadLine();
				_config.EmailFrom = configData.ReadLine();
				_config.SmtpServer = configData.ReadLine();
				_config.NameTo = configData.ReadLine();
				_config.EmailTo = configData.ReadLine();
				configData.Close();
			} catch {
			}
		}
	
		private void ConfigurationSave() {
			try {
				FileStream configFile = File.OpenWrite(ConfigurationFile());
				StreamWriter configData = new StreamWriter(configFile, System.Text.Encoding.UTF8);
				configData.WriteLine(_config.NameFrom);
				configData.WriteLine(_config.EmailFrom);
				configData.WriteLine(_config.SmtpServer);
				configData.WriteLine(_config.NameTo);
				configData.WriteLine(_config.EmailTo);
				configData.Close();
			} catch {
				Message("Error saving configuration!", true);
			}
		}

		#endregion

		#region MessageBox utilities

		private void Message(string text) {
			Message(text, false);
		}

		private void Message(string text, bool problem) {
			if ( problem )
				Message(text, System.Windows.Forms.MessageBoxIcon.Error);
			else
				Message(text, System.Windows.Forms.MessageBoxIcon.Information);
		}

		private void Message(string text, System.Windows.Forms.MessageBoxIcon icon) {
			System.Windows.Forms.MessageBox.Show(
				text, "Mail Plugin",
				System.Windows.Forms.MessageBoxButtons.OK,
				icon
				);
		}

		#endregion

	}

	public class MailItem : MailItemBase, Syndication.Extensibility.IBlogExtension {

		public string DisplayName {
			get {
				return "Email";
			}
		}

		public bool HasEditingGUI {
			get {
				return true;
			}
		}

		public bool HasConfiguration {
			get {
				return true;
			}
		}

		public void Configure(System.Windows.Forms.IWin32Window parent) {
			ShowConfigureDialog(parent);
		}

		public void BlogItem(System.Xml.XPath.IXPathNavigable rssFragment, bool edited) {
			SendEmail(rssFragment);
			edited = false;
		}

	}

	public class MailItemTo : MailItemBase, Syndication.Extensibility.IBlogExtension {

		private MailItemToDialog _dialog = new MailItemToDialog();

		public string DisplayName {
			get {
				return "Email to...";
			}
		}

		public bool HasEditingGUI {
			get {
				return true;
			}
		}

		public bool HasConfiguration {
			get {
				return false;
			}
		}

		public void Configure(System.Windows.Forms.IWin32Window parent) {}

		public void BlogItem(System.Xml.XPath.IXPathNavigable rssFragment, bool edited) {
			_dialog.ShowDialog();
			if ( _dialog.DialogResult == System.Windows.Forms.DialogResult.OK ) {
				SendEmail(rssFragment, _dialog.MailToEmail, _dialog.MailToSubject);				
			}
			edited = false;
		}

	}

}
