<?php
// Function to read rotation from the irot box of an AVIF file
function getAvifRotation(string $imagePath): int {
    $fp = fopen($imagePath, 'rb');
    if (!$fp) {
        return 0;
    }
    $data = fread($fp, 512);
    fclose($fp);

    if ($data === false) {
        return 0;
    }

    $pos = strpos($data, 'irot');
    if ($pos === false || ($pos + 1 >= strlen($data))) {
        return 0;
    }

    $rotationByte = ord($data[$pos + 4]);

    return match ($rotationByte) {
        1 => 90,
        2 => 180,
        3 => 270,
        default => 0,
    };
}

// Get all .avif files in the current directory
$files = scandir(__DIR__);
$avifFiles = array_filter($files, fn($f) => preg_match('/\.avif$/i', $f));

if (empty($avifFiles)) {
    die("No AVIF files found in the directory.");
}

if (!function_exists('imagecreatefromavif')) {
    die("GD does not support AVIF (requires PHP >= 8.1 and GD with AVIF support).");
}

$results = [];

foreach ($avifFiles as $avifFile) {
    $rotation = getAvifRotation($avifFile);
    $image = imagecreatefromavif($avifFile);

    if ($image === false) {
        $results[] = [
            'file' => $avifFile,
            'error' => 'Error loading file',
            'rotation' => null,
            'base64' => null
        ];
        continue;
    }

    // Rotate image if necessary
    if ($rotation !== 0) {
        $image = imagerotate($image, $rotation, 0);
    }

    ob_start();
    imagejpeg($image, null, 1);
    $imageData = ob_get_clean();
    imagedestroy($image);

    $base64 = base64_encode($imageData);

    $results[] = [
        'file' => $avifFile,
        'rotation' => $rotation,
        'base64' => $base64,
        'error' => null
    ];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>AVIF to JPEG Preview (Multiple Files)</title>
    <style>
        img { max-width: 300px; display: block; margin-bottom: 1em; }
        .preview { margin-bottom: 3em; border-bottom: 1px solid #ccc; padding-bottom: 2em; }
    </style>
</head>
<body>
    <h1>AVIF to JPEG Preview (Multiple Files)</h1>

    <?php foreach ($results as $result): ?>
        <div class="preview">
            <p><strong>File:</strong> <?= htmlspecialchars($result['file']) ?></p>
            <?php if ($result['error']): ?>
                <p style="color:red;"><strong>Error:</strong> <?= htmlspecialchars($result['error']) ?></p>
            <?php else: ?>
                <p>Rotation applied: <strong><?= $result['rotation'] ?>°</strong></p>
                <img src="data:image/jpeg;base64,<?= $result['base64'] ?>" alt="Preview of <?= htmlspecialchars($result['file']) ?>" />
            <?php endif; ?>
        </div>
    <?php endforeach; ?>
</body>
</html>
